/*
  FTDB database system
  Copyright (C) 1995 Erik Troan, North Carolina State University
  Copyright (C) 1996 Red Hat Software
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/file.h>
#include <netinet/in.h>				/* for ntohl(), htonl() */

#include "ftdb.h"
#include "locks.h"
#include "trie.h"

#ifndef MAP_FILE
#define MAP_FILE 0
#endif

#define indexchar(ch) (isalnum((ch)))

struct HashTableBucket
{
    char * Key;
    unsigned int Data;
} ;

typedef struct
{
    struct HashTableBucket * Buckets;
    unsigned int Size, Elements;
} HashTable;

struct Data
{
    unsigned int Length;			/* network order */
    char Data[1];
} ;

struct PointerTableEntry
{
    unsigned int Offset;			/* network order */
    unsigned int Entries;			/* network order */
} ;

struct Correspondance
{
    unsigned int CorrKey;			/* network order */
    unsigned int Recordnum;			/* network order */
} ;

struct CorrespondanceHeader
{
    unsigned int Magic;				/* network order */
    unsigned int PointerTableEntries;		/* network order */
    unsigned int Unstructured;			/* network order */
    unsigned int NextPointer;			/* network order */
} ;

struct OneField
{
    char * Name;
    enum FieldTypesFTDB Type;
    Trie Index;
    int cfd;      /* fd for correspondance file */
    struct OneField * Next;
    HashTable Cache;
    struct CorrespondanceHeader CorHeader;
    int DoIndex;
    int StoreData;
    int CacheWrites;
    struct Correspondance * WriteCache;
    int WriteCacheSize;
    int WriteCacheUsed;
} ;

struct FullTextDB
{
    int fd;
    int NumFields;
    struct OneField * FieldList;
    unsigned int Size;
    char * Path;
    int ReadOnly;
} ;

#define CORRESPONDANCE_MAGIC 0xA45BC12D

static char * ReadCorrespondanceList(struct OneField * Field, 
				     unsigned int CorrKey, 
				     MatchList * Matches);
static char * AppendToCorrespondanceList(struct OneField * Field,
					 unsigned int FirstOffset, 
					 unsigned int Recordnum);
static char * DoIndex(struct OneField * Field, char * Item, 
		      unsigned int RecOffset);

static char * CreateHashTable(HashTable * Table, unsigned int Size);
static char * AddToHashTable(HashTable Table, char * Item, unsigned int Data);
static int InHashTable(HashTable Table, char * Item, unsigned int * Data);
static void ClearHashTable(HashTable Table);
static void FreeHashTable(HashTable Table);
static char * ReindexField(FullTextDB db, struct OneField * Field);
static int CorCompare(const void * first, const void * second);

char * OpenFTDB(char * Path, FullTextDB * indb, int ReadOnly)
{
    char * filespec;
    int size;
    struct FullTextDB * db;
    
    filespec = alloca(strlen(Path) + 20);
    if (!filespec) return "Out of stack space";

    strcpy(filespec, Path);
    strcat(filespec, "/Database");

    db = malloc(sizeof(*db));
    if (!db) return "Out of memory";
    *indb = db;

    db->Path = strdup(Path);
    if (!db->Path)
    {
	free(db);
	return "Out of memory";
    }

    if (ReadOnly)
    {
	db->fd = open(filespec, O_RDONLY);
    }
    else
    {
	db->fd = open(filespec, O_RDWR | O_CREAT, 0644);
    }
    
    if (db->fd < 0)
    {
	free(db);
	return strerror(errno);
    }
    
    if (!ReadOnly)
    {
	if (lockfile(db->fd))
	{
	    free(db);
	    close(db->fd);
	    return "Cannot lock data file";
	}
    }	
    
    size = lseek(db->fd, 0, SEEK_END);
    db->Size = size;
    
    db->FieldList = NULL;
    db->NumFields = 0;
    db->ReadOnly = ReadOnly;

    return NULL;
}

static int writeCorHeader(int fd, struct CorrespondanceHeader * cor) {
    struct CorrespondanceHeader no;

    no.Magic = htonl(cor->Magic);
    no.Unstructured = htonl(cor->Unstructured);
    no.PointerTableEntries = htonl(cor->PointerTableEntries);
    no.NextPointer = htonl(cor->NextPointer);

    return (write(fd, &no, sizeof(no)) != sizeof(no));
}

static int readCorHeader(int fd, struct CorrespondanceHeader * cor) {
    int rc;

    rc = read(fd, cor, sizeof(*cor));
    if (rc != sizeof(*cor)) return 1;

    cor->Magic = ntohl(cor->Magic);
    cor->Unstructured = ntohl(cor->Unstructured);
    cor->PointerTableEntries = ntohl(cor->PointerTableEntries);
    cor->NextPointer = ntohl(cor->NextPointer);

    return 0;
}

char * DefineFieldFTDB(FullTextDB db, char * Name, enum FieldTypesFTDB Type,
			int StoreData, int Index, int WriteCache)
{
    struct OneField * Field, *Curr;
    char * FileSpec;
    char * rc;
    unsigned int Offset;
    
    if (Type != FTDB_STRING) return "I can only deal with strings right now";
        
    FileSpec = alloca(strlen(db->Path) + 20 + strlen(Name));
    if (!FileSpec) return "Out of memory";
    
    Field = malloc(sizeof(*Field));
    if (!Field) return "Out of memory";
    
    Field->Name = strdup(Name);
    if (!Field->Name) 
    {
	free(Field);
	return "Out of memory";
    }

    Field->Type = Type;
    Field->DoIndex = Index;
    Field->StoreData = StoreData;
    Field->Next = NULL;
    Field->WriteCache = NULL;
    
    if (Field->DoIndex)
    {
	if ((rc = CreateHashTable(&Field->Cache, 10000)))
	{
	    free(Field);
	    free(Field->Name);
	    return rc;
	}

	if (WriteCache)
	{
	    Field->WriteCache = malloc(2000 * sizeof(*Field->WriteCache));
	    if (!Field->WriteCache) return "Out of memory";
	    Field->WriteCacheSize = 2000;
	    Field->WriteCacheUsed = 0;
	}

	sprintf(FileSpec, "%s/%s.Index", db->Path, Name);
	rc = OpenTrie(FileSpec, &Field->Index, db->ReadOnly);
	if (rc)
	{
	    FreeHashTable(Field->Cache);
	    free(Field->Name);
	    free(Field);
	    return rc;
	}
	
	sprintf(FileSpec, "%s/%s.Cor", db->Path, Name);
	if (db->ReadOnly)
	{
	    Field->cfd = open(FileSpec, O_RDONLY);
	}
	else
	{
	    Field->cfd = open(FileSpec, O_RDWR | O_CREAT, 0644);
	}

	if (Field->cfd < 0)
	{
	    FreeHashTable(Field->Cache);
	    CloseTrie(Field->Index);
	    free(Field->Name);
	    free(Field);
	    return strerror(errno);
	}

	if (!db->ReadOnly)
	{
	    if (lockfile(Field->cfd))
	    {
		FreeHashTable(Field->Cache);
		CloseTrie(Field->Index);
		free(Field->Name);
		free(Field);
		close(Field->cfd);
		return "Cannot lock correspondance file";
	    }
	}
	
	Offset = lseek(Field->cfd, 0, SEEK_END);
	if (!Offset)
	{
	    if (db->ReadOnly)
	    {
		FreeHashTable(Field->Cache);
		CloseTrie(Field->Index);
		free(Field->Name);
		free(Field);
		close(Field->cfd);
		return "Correspondance file size should not be zero";
	    }
	    
	    Field->CorHeader.Magic = CORRESPONDANCE_MAGIC;
	    Field->CorHeader.Unstructured = sizeof(Field->CorHeader);
	    Field->CorHeader.PointerTableEntries = 0;
	    Field->CorHeader.NextPointer = 0;
	    writeCorHeader(Field->cfd, &Field->CorHeader);
	}
	else
	{
	    lseek(Field->cfd, 0, SEEK_SET);
	    readCorHeader(Field->cfd, &Field->CorHeader);
	    
	    if (Field->CorHeader.Magic != CORRESPONDANCE_MAGIC)
		return "Bad magic number for correspondance file";
	}
    }

    if (!db->FieldList)
    {
	db->FieldList = Field;
    }
    else
    {
	Curr = db->FieldList;
	while (Curr->Next) Curr = Curr->Next;
	Curr->Next = Field;
    }

    db->NumFields++;

    return NULL;
}
    
char * SearchFTDB(FullTextDB db, char * String, char * FieldName, 
		  MatchList * Matches)
{
    struct OneField * Field;
    char * rc;
    unsigned int CorresOffset;
    MatchList PartialMatchList;
    int FoundFirst = 0;

    *Matches = NULL;
    
    if (!FieldName) 
    {
	Field = db->FieldList;
	if (!Field) return NULL;
	
	while (Field && !FoundFirst)
	{
	    if (Field->DoIndex)
	    {
		rc = SearchFTDB(db, String, Field->Name, Matches);
		if (rc) return rc;
	 	FoundFirst = 1;
	    }
	    Field = Field->Next;
	}

	if (!FoundFirst) return "Database has no searchable fields";

	while (Field)
	{
	    if (!Field->DoIndex) {
		Field = Field->Next;
		continue;
	    }

	    rc = SearchFTDB(db, String, Field->Name, &PartialMatchList);
	    if (rc)
	    {
		FreeMatchesFTDB(*Matches);
		return rc;
	    }
	    
	    UnionMatchesFTDB(*Matches, PartialMatchList, Matches);
	    Field = Field->Next;
	}

	return NULL;
    }

    Field = db->FieldList;
    while (Field)
    {
	if (!strcmp(Field->Name, FieldName)) break;
	Field = Field->Next;
    }
    if (!Field) return "Bad field name";
    if (!Field->DoIndex) return "Field is not indexed";
    
    rc = FindWordInTrie(Field->Index, String, &CorresOffset);
    if (rc) return rc;
    if (!CorresOffset) return NULL;

    CorresOffset = ntohl(CorresOffset);
    
    CorresOffset -= 1;
    
    return ReadCorrespondanceList(Field, CorresOffset, Matches);
}

static char * FlushWriteCache(struct OneField * Field)
{
    if (!Field->WriteCache) return "No write cache found";

    lseek(Field->cfd, 0, SEEK_END);
    if (write(Field->cfd, Field->WriteCache, 
              sizeof(*Field->WriteCache) * Field->WriteCacheUsed) == -1)
        return strerror(errno);
    Field->WriteCacheUsed = 0;

    return NULL;
}

static char * AppendToCorrespondanceList(struct OneField * Field, 
					 unsigned int CorrKey, 
					 unsigned int Recordnum)
{
    struct Correspondance CorEntry;
    char * rc = NULL;

    CorrKey = htonl(CorrKey);
    Recordnum = htonl(Recordnum);
    
    if (Field->WriteCache)
    {
	Field->WriteCache[Field->WriteCacheUsed].CorrKey = CorrKey;
	Field->WriteCache[Field->WriteCacheUsed].Recordnum = Recordnum;
	Field->WriteCacheUsed++;

	if (Field->WriteCacheUsed == Field->WriteCacheSize)
	{
	    rc = FlushWriteCache(Field);
	}

        rc = NULL;
    }
    else
    {
	CorEntry.CorrKey = CorrKey;
	CorEntry.Recordnum = Recordnum;

	lseek(Field->cfd, 0, SEEK_END);
	if (write(Field->cfd, &CorEntry, sizeof(CorEntry)) == -1)
	    return strerror(errno);
    }

    return rc;
}

static char * ReadCorrespondanceList(struct OneField * Field, 
				     unsigned int CorKey, 
				     MatchList * Matches)
{
    struct PointerTableEntry Entry;
    unsigned int * Records;
    int i;
    struct OneMatch * Curr = NULL;
    struct Correspondance * CorTable;
    unsigned int Extra, Filesize;
    caddr_t Map;

    *Matches = NULL;
    if (CorKey < Field->CorHeader.PointerTableEntries)
    {
	if (lseek(Field->cfd, CorKey * sizeof(Entry) + 
		  sizeof(Field->CorHeader), SEEK_SET) == -1) 
	    return strerror(errno);
	if (read(Field->cfd, &Entry, sizeof(Entry)) == -1)
            return strerror(errno);

	Entry.Offset = ntohl(Entry.Offset);
	Entry.Entries = ntohl(Entry.Entries);
	
	if (lseek(Field->cfd, Entry.Offset, SEEK_SET) == -1) 
            return strerror(errno);
	
	Records = alloca(sizeof(int) * Entry.Entries);
	if (!Records) return "Out of memory";
	
	if (read(Field->cfd, Records, sizeof(int) * Entry.Entries) == -1)
	    return strerror(errno);
	
	for (i = 0; i < Entry.Entries; i++)
	{
	    if (!*Matches)
	    {
		*Matches = malloc(sizeof(**Matches));
		Curr = *Matches;
		if (!Curr) return "Out of memory";
		Curr->Next = NULL;
		Curr->Offset = ntohl(Records[i]);
	    }
	    else
	    {
		Curr->Next = malloc(sizeof(**Matches));
		Curr = Curr->Next;
		if (!Curr) return "Out of memory";
		Curr->Next = NULL;
		Curr->Offset = ntohl(Records[i]);
	    }
	}
    }

    Filesize = lseek(Field->cfd, 0, SEEK_END);
    if ((int) Filesize == -1) return strerror(errno);
    Extra = ((Filesize - Field->CorHeader.Unstructured) / 
	     sizeof(struct Correspondance));
    
    Map = (void *) mmap(NULL, Filesize, PROT_READ, MAP_FILE | MAP_PRIVATE, 
			Field->cfd, 0);
    if (Map == (void *) -1) return strerror(errno);

    CorTable = (void *) ((char *) Map + Field->CorHeader.Unstructured);
        
    for (i = 0; i < Extra; i++)
    {
	if (ntohl(CorTable[i].CorrKey) != CorKey) continue;
	
	if (!*Matches)
	{
	    *Matches = malloc(sizeof(**Matches));
	    Curr = *Matches;
	    if (!Curr) return "Out of memory";
	    Curr->Next = NULL;
	    Curr->Offset = ntohl(CorTable[i].Recordnum);
	}
	else
	{
	    Curr->Next = malloc(sizeof(**Matches));
	    Curr = Curr->Next;
	    if (!Curr) return "Out of memory";
	    Curr->Next = NULL;
	    Curr->Offset = ntohl(CorTable[i].Recordnum);
	}
    }

    munmap(Map, Extra * sizeof(struct Correspondance));
    
    return NULL;
}

char * ReindexFTDB(FullTextDB db)
{
    struct OneField * Field;
    char * rc = NULL;

    if (db->ReadOnly) return "Database is open for read only";

    /* Go through the indexes individually and rebuild them */
    
    Field = db->FieldList;
    while (Field)
    {
	if (Field->WriteCache)
	{
	    rc = FlushWriteCache(Field);
	    if (rc) return rc;
	}

	if (Field->DoIndex) rc = ReindexField(db, Field);
	if (rc) return rc;
	Field = Field->Next;
    }
    
    return NULL;
}

static int CorCompare(const void * first, const void * second)
{
    const struct Correspondance * FirstCor = first;
    const struct Correspondance * SecondCor = second;
    unsigned int firstKey = ntohl(FirstCor->CorrKey);
    unsigned int secondKey = ntohl(SecondCor->CorrKey);
    unsigned int firstRec = ntohl(FirstCor->Recordnum);
    unsigned int secondRec = ntohl(SecondCor->Recordnum);

    if (firstKey < secondKey) return -1;
    if (firstKey > secondKey) return 1;
    
    if (firstRec < secondRec) return -1;
    if (firstRec > secondRec) return 1;
    
    return 0;
}

static char * ReindexField(FullTextDB db, struct OneField * Field)
{
    char * Map;
    struct Correspondance * CorTable;
    int newfd;
    unsigned int Filesize;
    char * Filename, * FinalFilename;
    unsigned int NumUnstructured;
    struct PointerTableEntry * EntryTable, EntryHdr;
    char * rc;
    unsigned int * RecList;
    unsigned int NextCor;
    unsigned int i, j;
    unsigned int End;

    if (!Field->DoIndex) return "Field not indexed";
    
    Filename = alloca(strlen(db->Path) + strlen(Field->Name) + 20);
    if (!Filename) return "Out of memory";
    FinalFilename = alloca(strlen(db->Path) + strlen(Field->Name) + 20);
    if (!FinalFilename) return "Out of memory";
    
    sprintf(FinalFilename, "%s/%s.Cor", db->Path, Field->Name);
    sprintf(Filename, "%s/%s.Cor.New", db->Path, Field->Name);
    unlink(Filename);
    newfd = open(Filename, O_RDWR | O_CREAT, 0644);
    if (newfd < 0) return strerror(errno);
    
    Filesize = lseek(Field->cfd, 0, SEEK_END);
    if ((int) Filesize == -1) 
    {
	rc = strerror(errno);
	close(newfd);
	return rc;
    }
    
    Map = (char *) mmap(NULL, Filesize, PROT_READ | PROT_WRITE,
			MAP_FILE | MAP_PRIVATE, Field->cfd, 0);
    if (Map == (char *) -1) 
    {
	rc = strerror(errno);
	unlink(Filename);
	close(newfd);
	return rc;
    }

    if (writeCorHeader(newfd, &Field->CorHeader)) 
    {
	rc = strerror(errno);
	unlink(Filename);
	close(newfd);
	return rc;
    }

    NumUnstructured = ((Filesize - Field->CorHeader.Unstructured) / 
		       sizeof(struct Correspondance));

    CorTable = (void *) ((char *) Map + Field->CorHeader.Unstructured);
    EntryTable = (void *) ((char *) Map + sizeof(struct CorrespondanceHeader));
    RecList = (void *) ((char *) EntryTable + 
			(Field->CorHeader.PointerTableEntries *
			 sizeof(struct PointerTableEntry)));
    
    /* Sort the unstructured elements */
    qsort(CorTable, NumUnstructured, sizeof(struct Correspondance), 
	  CorCompare);

    /* Set aside space for the entry table */
    End = lseek(newfd, (Field->CorHeader.NextPointer
			* sizeof(struct PointerTableEntry)), SEEK_CUR);

    NextCor = 0;

    for (i = 0; i < Field->CorHeader.NextPointer; i++)
    {
	EntryHdr.Offset = htonl(lseek(newfd, End, SEEK_SET));
	EntryHdr.Entries = 0;

	if (i < Field->CorHeader.PointerTableEntries)
	{
	    EntryHdr.Entries = ntohl(EntryTable[i].Entries);

	    if (write(newfd, RecList, EntryHdr.Entries * sizeof(int)) 
		== -1)
	    {
		rc = strerror(errno);
		unlink(Filename);
		close(newfd);
		return rc;
	    }
	    RecList += EntryHdr.Entries;
	    End += EntryHdr.Entries * sizeof(int);
	}

	j = htonl(i);
	while (j == CorTable[NextCor].CorrKey && NextCor < NumUnstructured)
	{
	    EntryHdr.Entries++;
	    if (write(newfd, &CorTable[NextCor].Recordnum, sizeof(int)) == -1)
	    {
		rc = strerror(errno);
		unlink(Filename);
		close(newfd);
		return rc;
	    }
	    End += sizeof(int);
	    NextCor++;
	}
	    
	lseek(newfd, sizeof(struct CorrespondanceHeader) + 
	      i * sizeof(struct PointerTableEntry), SEEK_SET);

	EntryHdr.Entries = htonl(EntryHdr.Entries);

	if (write(newfd, &EntryHdr, sizeof(EntryHdr)) == -1)
	{
	    rc = strerror(errno);
	    unlink(Filename);
	    close(newfd);
	    return rc;
	}
    }

    Field->CorHeader.PointerTableEntries = Field->CorHeader.NextPointer;
    Field->CorHeader.Unstructured = End;
    lseek(newfd, 0, SEEK_SET);
    if (writeCorHeader(newfd, &Field->CorHeader))
    {
	rc = strerror(errno);
	unlink(Filename);
	close(newfd);
	return rc;
    }

    close(newfd);

    munmap(Map, Filesize);
    close(Field->cfd);
    
    if (rename(Filename, FinalFilename) == -1)
    {
	rc = strerror(errno);
	unlink(Filename);
	return rc;
    }    
    
    Field->cfd = open(FinalFilename, O_RDWR);
    if (Field->cfd < 0)
    {
	return strerror(errno);
    }
    
    lseek(Field->cfd, 0, SEEK_SET);
    readCorHeader(Field->cfd, &Field->CorHeader);
    if (Field->CorHeader.Magic != CORRESPONDANCE_MAGIC)
        return "Bad magic number for correspondance file";

    return NULL;
}

char * UnionMatchesFTDB(MatchList One, MatchList Two, MatchList * Result)
{
    struct OneMatch * Place;
    struct OneMatch * TempMatch;
    struct OneMatch * TheResult;

    if (!One)
    {
	*Result = Two;
	return NULL;
    }
    else if (!Two)
    {
	*Result = One;
	return NULL;
    }

    if (One->Offset < Two->Offset)
    {
	TheResult = One;
	One = One->Next;
    }
    else
    {
	TheResult = Two;
	if (One->Offset == Two->Offset) 
	{
	    TempMatch = One;
	    One = One->Next;
	    free(TempMatch);
	}
	Two = Two->Next;
    }
    Place = TheResult;
    
    while (One && Two)
    {
	if (One->Offset < Two->Offset)
	{
	    Place->Next = One;
	    Place = One;
	    One = One->Next;
	}
	else if (One->Offset >= Two->Offset)
	{
	    Place->Next = Two;
	    Place = Two;
	    
	    if (One->Offset == Two->Offset) 
	    {
		TempMatch = One;
		One = One->Next;
		free(TempMatch);
	    }

	    Two = Two->Next;
	}
    }
    
    if (!Two)
    {
	Place->Next = One;
    }
    else
    {
	Place->Next = Two;
    }

    *Result = TheResult;
    
    return NULL;
}

char * IntersectMatchesFTDB(MatchList One, MatchList Two, MatchList * Result)
{
    struct OneMatch * Place, * TempMatch, * TheResult;

    while (One && Two && One->Offset != Two->Offset)
    {
	if (One->Offset < Two->Offset)
	{
	    TempMatch = One;
	    One = One->Next;
	    free(TempMatch);
	}
	else
	{
	    TempMatch = Two;
	    Two = Two->Next;
	    free(TempMatch);
	}
    }
    
    if (!One)
    {
	FreeMatchesFTDB(Two);
	*Result = NULL;
	return NULL;
    }
    else if (!Two)
    {
	FreeMatchesFTDB(One);
	*Result = NULL;
	return NULL;
    }
    
    TheResult = One;
    Place = One;
    One = One->Next;
    TempMatch = Two;
    Two = Two->Next;
    free(TempMatch);
    
    while (One && Two)
    {
	if (One->Offset < Two->Offset)
	{
	    TempMatch = One;
	    One = One->Next;
	    free(TempMatch);
	}
	else if (One->Offset > Two->Offset)
	{
	    TempMatch = Two;
	    Two = Two->Next;
	    free(TempMatch);
	}
	else /* One->Offset == Two->Offset */
	{
	    Place->Next = One;
	    Place = Place->Next;
	    One = One->Next;
	    TempMatch = Two;
	    Two = Two->Next;
	    free(TempMatch);
	}
    }
    
    if (Two)
    {
	FreeMatchesFTDB(Two);
    }
    else if (One)
    {
	FreeMatchesFTDB(One);
    }

    Place->Next = NULL;

    *Result = TheResult;

    return NULL;
}

void FreeMatchesFTDB(MatchList List)
{
    struct OneMatch * Match, * NextMatch;
    
    Match = List;
    while (Match)
    {
	NextMatch = Match->Next;
	free(Match);
	Match = NextMatch;
    }
}

void CloseFTDB(FullTextDB db)
{
    struct OneField * Field, * LastField;

    close(db->fd);
    free(db->Path);

    Field = db->FieldList;
    while (Field)
    {
	free(Field->Name);
	if (Field->DoIndex)
	{
	    if (Field->WriteCache)
	    {
		FlushWriteCache(Field);
	    }
	    CloseTrie(Field->Index);
	    unlockfile(Field->cfd);
	    close(Field->cfd);
	    ClearHashTable(Field->Cache);
	}
	LastField = Field;
	Field = Field->Next;
	free(LastField);
    }

    unlockfile(db->fd);
    close(db->fd);

    free(db);
}

char * StartRecordFTDB(FullTextDB db, Record * inrec)
{
    struct Record * rec;
    
    rec = malloc(sizeof(struct Record));
    if (!(rec)) return "Out of memory";
    *inrec = rec;
    
    rec->Fields = malloc(sizeof(char *) * db->NumFields);
    if (!rec->Fields)
    {
	free(rec);
	return "Out of memory";
    }
    rec->NextFieldNum = 0;
    rec->FreeFirstOnly = 0;

    return NULL;
}


char * AddStringFieldToRecordFTDB(FullTextDB db, Record rec, char * data)
{
    rec->Fields[rec->NextFieldNum] = strdup(data);
    if (!rec->Fields[rec->NextFieldNum]) return "Out of memory";
    rec->NextFieldNum++;
    
    return NULL;
}

char * AddRecordFTDB(FullTextDB db, Record rec)
{
    unsigned int TotalLength = 0;
    struct Data * Data;
    char * chptr;
    char * rc;
    struct OneField * Field;
    int i;
    unsigned int Offset;

    if (db->ReadOnly) return "Database is open for read only";
    if (!rec) return "Rec cannot be NULL in AddRecordFTDB()";

    Field = db->FieldList;
    for (i = 0; i < db->NumFields; i++)
    {
	if (Field->StoreData) 
	    TotalLength += strlen(rec->Fields[i]) + 1;
	else
	    TotalLength++;

	Field = Field->Next;
    }
    
    Data = malloc(TotalLength + sizeof(unsigned int));
    if (!Data) return "Out of memory";
    
    Data->Length = htonl(TotalLength);
    chptr = &Data->Data[0];
    Field = db->FieldList;
    for (i = 0; i < db->NumFields; i++)
    {
	if (Field->StoreData)
	{
	    strcpy(chptr, rec->Fields[i]);
	    chptr += strlen(rec->Fields[i]) + 1;
	}
	else
	{
	    *chptr = '\0';
	    chptr++;
	}
	Field = Field->Next;
    }
    
    if (lseek(db->fd, db->Size, SEEK_SET) == -1)
    {
	free(Data);
	return strerror(errno);
    }
    
    if (write(db->fd, Data, TotalLength + sizeof(unsigned int)) == -1)
    {
	free(Data);
	return strerror(errno);
    }
    Offset = db->Size;
    db->Size += TotalLength + sizeof(unsigned int);
        
    /* Errors in here are bad as we can't go back and erase items from
       indexes! This oughta be fixed *** */
    
    Field = db->FieldList;
    for (i = 0; i < db->NumFields; i++)
    {
	if (Field->DoIndex)
	{
	    rc = DoIndex(Field, rec->Fields[i], Offset);
	    if (rc) return rc;
	}
	Field = Field->Next;
    }

    free(Data);

    return NULL;
}

static char * DoIndex(struct OneField * Field, char * Item, 
		      unsigned int RecOffset)
{
    char * s;
    char * chptr;
    char * start;
    char * rc;
    int last = 0;
    unsigned int CorrKey;
    int ItemLength;
    HashTable WordsIndexed;

    ItemLength = strlen(Item);

    rc = CreateHashTable(&WordsIndexed, ItemLength);

    s = alloca(ItemLength + 1);
    if (!s) return "Out of stack space";

    chptr = s;
    strcpy(s, Item);
    while (*chptr)
    {
	*chptr = tolower(*chptr);
	chptr++;
    }

    chptr = s;
    while (*chptr && !last)
    {
	while (!indexchar(*chptr) && *chptr) chptr++;
	if (!(*chptr)) break;
	
	start = chptr;
	
	while (indexchar(*chptr) && *chptr) chptr++;
	if (!(*chptr)) last = 1;
	*chptr = '\0';

	if (InHashTable(WordsIndexed, start, &CorrKey)) 
	{
	    chptr++;
	    continue;
	}
	
	rc = AddToHashTable(WordsIndexed, start, CorrKey);
	if (rc) 
	    return rc;

	if (!InHashTable(Field->Cache, start, &CorrKey))
	{
	    rc = FindWordInTrie(Field->Index, start, &CorrKey);
	    if (rc) 
	        return rc;

	    if (CorrKey)
	    {
		CorrKey = ntohl(CorrKey);
		rc = AddToHashTable(Field->Cache, start, CorrKey);
		if (rc) 
		    return rc;
	    }
	}

	if (!CorrKey)
	{
	    CorrKey = (Field->CorHeader.NextPointer++) + 1;

	    lseek(Field->cfd, 0, SEEK_SET);
	    writeCorHeader(Field->cfd, &Field->CorHeader);

	    rc = AddWordToTrie(Field->Index, start, htonl(CorrKey));
	    if (rc) 
		return rc;

	    rc = AddToHashTable(Field->Cache, start, CorrKey);
	    if (rc) 
		return rc;
	}

	rc = AppendToCorrespondanceList(Field, CorrKey - 1, RecOffset);
	if (rc) 
	    return rc;
    
	chptr++;
    }

    FreeHashTable(WordsIndexed);

    return NULL;
}

void FreeRecordFTDB(FullTextDB db, Record Rec)
{
    int i;

    if (Rec->FreeFirstOnly)
    {
	free(Rec->Fields[0]);
    }
    else
    {
	for (i = 0; i < db->NumFields; i++) free(Rec->Fields[i]);
    }
    
    free(Rec->Fields);
    free(Rec);
}

char * GetRecordFTDB(FullTextDB db, unsigned int RecordOffset, Record * rec)
{
    char * rc;
    unsigned int Length;
    int i;
    struct OneField * Field;
        
    if (lseek(db->fd, RecordOffset, SEEK_SET) == -1) return strerror(errno);

    if (read(db->fd, &Length, sizeof(Length)) == -1) return strerror(errno);
    Length = ntohl(Length);

    rc = StartRecordFTDB(db, rec);
    if (rc) return rc;
    
    (*rec)->Fields[0] = malloc(Length);
    if (!(*rec)->Fields[0])
    {
	free((*rec)->Fields);
	free((*rec));
	return "Out of memory";
    }

    if (read(db->fd, (*rec)->Fields[0], Length) == -1)
    {
	free((*rec)->Fields[0]);
	free((*rec)->Fields);
	free((*rec));
	return "Out of memory";
    }

    (*rec)->FreeFirstOnly = 1;

    Field = db->FieldList->Next;
    for (i = 1; i < db->NumFields; i++)
    {
	(*rec)->Fields[i] = (*rec)->Fields[i-1] + 
			    strlen((*rec)->Fields[i - 1]) + 1;
	Field = Field->Next;
    }

    (*rec)->NextFieldNum = db->NumFields;
    
    return NULL;
}

static char * CreateHashTable(HashTable * Table, unsigned int Size)
{
    Table->Buckets = malloc(sizeof(struct HashTableBucket) * Size);
    if (!(Table->Buckets)) return "Out of memory";

    memset(Table->Buckets, 0, sizeof(struct HashTableBucket) * Size);
    
    Table->Size = Size;
    Table->Elements = 0;
    
    return NULL;
}

static char * AddToHashTable(HashTable Table, char * Item, unsigned int Data)
{
    unsigned int HashValue;
    unsigned int Tries;
    char * s = Item;

    if ((Table.Size / 3) < Table.Elements) ClearHashTable(Table);
    
    for (HashValue = 0; *s; s++) HashValue = *s + 31 * HashValue;
    HashValue %= Table.Size;
    
    Tries = 0;
    while (Tries < Table.Size && Table.Buckets[HashValue].Key)
    {
	Tries++;
	HashValue = (HashValue + 1) % Table.Size;
    }
    
    if (Tries == Table.Size) 
    {
	ClearHashTable(Table);
        return AddToHashTable(Table, Item, Data);
    }
    
    Table.Buckets[HashValue].Data = Data;
    Table.Buckets[HashValue].Key = strdup(Item);
    if (!Table.Buckets[HashValue].Key)
        return "Out of memory";
    
    return NULL;
}

static int InHashTable(HashTable Table, char * Item, unsigned int * Data)
{
    unsigned int HashValue;
    unsigned int Tries;
    char * s = Item;
    
    for (HashValue = 0; *s; s++) HashValue = *s + 31 * HashValue;
    HashValue %= Table.Size;
    
    Tries = 0;
    while (Tries < Table.Size && Table.Buckets[HashValue].Key)
    {
	if (!strcmp(Table.Buckets[HashValue].Key, Item))
	{
	    *Data = Table.Buckets[HashValue].Data;
	    return 1;
	}
	HashValue = (HashValue + 1) % Table.Size;
	
	Tries++;
    }
    
    return 0;
}    

static void ClearHashTable(HashTable Table)
{
    int i;
    
    for (i = 0; i < Table.Size; i++)
    {
	if (Table.Buckets[i].Key)
	{ 
	    free(Table.Buckets[i].Key);
	    Table.Buckets[i].Key = NULL;
	}
    }
}

static void FreeHashTable(HashTable Table)
{
    ClearHashTable(Table);
    free(Table.Buckets);
}

#ifdef TEST_FTDB

static void die(char * rc);

void die(char * rc)
{
    fprintf(stderr, "%s\n", rc);
    exit(1);
}

void Dump(FullTextDB db, unsigned int Offset, char * label);

void Dump(FullTextDB db, unsigned int Offset, char * label)
{
    Record rec;
    char * rc;

    rc = GetRecordFTDB(db, Offset, &rec);
    if (rc) die(rc);
    printf("%s: %s %s %s %s\n", label, rec->Fields[0], rec->Fields[1],
	   rec->Fields[2], rec->Fields[3]);
    FreeRecordFTDB(db, rec);
}


int main(void)
{
    Record rec;
    FullTextDB db;
    char * rc;
    MatchList Matches;
    MatchList OctaviaMatches;
    MatchList ErikMatches;
    int i;
    struct tests
    {
	char * Name, * Address, * City, * State;
    } ;
    struct tests TestData[] = { 
	{ "Erik Troan", "3232 Octavia Street", "Raleigh", "NC" },
	{ "Alan Hudson", "3234 Octavia Street", "Raleigh", "NC" },
	{ "Erik's Girl", "Highway 54", "Chapel Hill", "NC" },
	{ "Mom", "Old Post Road", "Rhinebeck", "NY" }
    } ;
    
    system("rm -rf testdb; mkdir testdb");
    rc = OpenFTDB("testdb", &db, 0);
    if (rc) die(rc);

    rc = DefineFieldFTDB(db, "Name", FTDB_STRING, 1, 1, 1);
    if (rc) die(rc);
    rc = DefineFieldFTDB(db, "Address", FTDB_STRING, 1, 1, 1);
    if (rc) die(rc);
    rc = DefineFieldFTDB(db, "City", FTDB_STRING, 1, 1, 1);
    if (rc) die(rc);
    rc = DefineFieldFTDB(db, "State", FTDB_STRING, 1, 1, 1);
    if (rc) die(rc);

    for (i = 0; i < sizeof(TestData) / sizeof(struct tests); i++)
    {
	rc = StartRecordFTDB(db, &rec);
	if (rc) die(rc);

	rc = AddStringFieldToRecordFTDB(db, rec, TestData[i].Name);
	if (rc) die(rc);
	rc = AddStringFieldToRecordFTDB(db, rec, TestData[i].Address);
	if (rc) die(rc);
	rc = AddStringFieldToRecordFTDB(db, rec, TestData[i].City);
	if (rc) die(rc);
	rc = AddStringFieldToRecordFTDB(db, rec, TestData[i].State);
	if (rc) die(rc);

	rc = AddRecordFTDB(db, rec);
	if (rc) die(rc);
	FreeRecordFTDB(db, rec);

	if ((i % 3) == 0) ReindexFTDB(db);
    }
    
    rc = SearchFTDB(db, "Erik", "Name", &ErikMatches);
    if (rc) die(rc);
    if (!ErikMatches || !ErikMatches->Next) 
        die("Should have found \"Erik\" and \"Erik's Girl\"\n");
    if (ErikMatches->Next->Next) die("Found too many for \"Erik\"\n"); 

    Dump(db, ErikMatches->Offset, "Erik");
    Dump(db, ErikMatches->Next->Offset, "Erik");

    rc = SearchFTDB(db, "Octavia", "Address", &OctaviaMatches);
    if (rc) die(rc);
    if (!OctaviaMatches || !OctaviaMatches->Next) 
        die("Should have found \"Octavia\"\n");
    if (OctaviaMatches->Next->Next) die("Found too many for \"Octavia\"\n");

    Dump(db, OctaviaMatches->Offset, "Octavia");
    Dump(db, OctaviaMatches->Next->Offset, "Octavia");

    rc = SearchFTDB(db, "NY", "State", &Matches);
    if (rc) die(rc);
    if (!Matches) die("Should have found \"NY\"\n");
    if (Matches->Next) die("Found too many for \"NY\"\n");

    Dump(db, Matches->Offset, "NY");
    FreeMatchesFTDB(Matches);

    rc = SearchFTDB(db, "NC", "State", &Matches);
    if (rc) die(rc);
    if (!Matches || !Matches->Next || !Matches->Next->Next)
        die("Should have found \"NC\"\n");
    if (Matches->Next->Next->Next) die("Found too many for \"NC\"\n");
 
    Dump(db, Matches->Offset, "NC");
    Dump(db, Matches->Next->Offset, "NC");
    Dump(db, Matches->Next->Next->Offset, "NC");
    FreeMatchesFTDB(Matches);

    CloseFTDB(db);
    rc = OpenFTDB("testdb", &db, 0);
    if (rc) die(rc);

    rc = DefineFieldFTDB(db, "Name", FTDB_STRING, 1, 1, 1);
    if (rc) die(rc);
    rc = DefineFieldFTDB(db, "Address", FTDB_STRING, 1, 1, 1);
    if (rc) die(rc);
    rc = DefineFieldFTDB(db, "City", FTDB_STRING, 1, 1, 1);
    if (rc) die(rc);
    rc = DefineFieldFTDB(db, "State", FTDB_STRING, 1, 1, 1);
    if (rc) die(rc);

    UnionMatchesFTDB(ErikMatches, OctaviaMatches, &Matches);
    if (!Matches || !Matches->Next || !Matches->Next->Next)
        die("Union of Erik and Octavia too short\n");
    if (Matches->Next->Next->Next) 
        die("Found too many Erik | Octavia\n");

    Dump(db, Matches->Offset, "Erik|Octavia");
    Dump(db, Matches->Next->Offset, "Erik|Octavia");
    Dump(db, Matches->Next->Next->Offset, "Erik|Octavia");
    
    FreeMatchesFTDB(Matches);

    rc = SearchFTDB(db, "Erik", "Name", &ErikMatches);
    if (rc) die(rc);
    rc = SearchFTDB(db, "Octavia", "Address", &OctaviaMatches);
    if (rc) die(rc);

    IntersectMatchesFTDB(ErikMatches, OctaviaMatches, &Matches);
    if (!Matches) die("Should have found \"Erik\" & \"Octavia\"\n");
    if (Matches->Next) die("Found too many for \"Erik\" & \"Octavia\"\n");

    Dump(db, Matches->Offset, "Erik&Octavia");
    
    FreeMatchesFTDB(Matches);
   
    CloseFTDB(db);

    return 0;
}

#endif
