/*
** obstrat.c - Oblique Strategies
**
** Based on original code by Dave MacLeod <davmac@netcomuk.co.uk>
**
** Copyright 1998, Lonnon Foster <lonnief@pobox.com>
** All rights reserved.
**
** The original Oblique Strategies are Copyright 1975, 78, 79
** by Brian Eno and Peter Schmidt.
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
** General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program (see LICENSE.TXT); if not, write to the
** Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
** USA.
**
** Created 30 June, 1998.  Last edited 14 July, 1998.
*/

#include <System/SysAll.h>
#include <UI/UIAll.h>

#include "obstrat.h"
#include "resource.h"


/* Global variables */
static short        CurrNo = 0;
DmOpenRef           obstratDB;
char                obstratDBName[] = "ObStratDB";


/*
** RandN
**   Returns a random integer in the range 0..(n-1).
** Inputs
**   n     Maximum number to be returned + 1.
** Returns
**   A random integer.
*/
UInt RandN(UInt n)
{
  return SysRandom(0) / (1+sysRandomMax/n);
}


/*
** GetObjPtr
**   Returns the pointer to an object on the active form.
** Inputs
**   objectID   The resource ID of the object.
** Returns
**   A pointer to the specified object.
*/
static VoidPtr GetObjPtr(Int objectID)
{
    FormPtr frm;
    
    frm = FrmGetActiveForm();
    return(FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, objectID)));
}


/*
** DisplayCard
**   Spits the oblique strategy text onto the screen and updates
**   the title with the current card number.
*/
void DisplayCard(void)
{
    FormPtr  frm;
    FieldPtr fld;
    char     Tmp1[4];
    char     Tmp2[26];
    Handle   recHandle;
    UInt     index;
    
    /* Set the current index */
    index = (UInt)CurrNo;

    /* Get pointer to active form */
    frm = FrmGetActiveForm();
    
    /* Assemble the card number string.  Note that we can't handle more  */
    /*  than a 3-digit number here; the title resource for the main form */
    /*  can only fit three characters beyond "Oblique Strategies - #".   */
    StrIToA(Tmp1, CurrNo + 1);
    StrCopy(Tmp2, "Oblique Strategies - #");
    StrCat(Tmp2, Tmp1);
    
    /* Change the title to display the current card number */
    FrmCopyTitle(frm, Tmp2);

    /* Get pointer to the field which will hold the card text */
    fld = (FieldPtr)GetObjPtr(fldID_MainText);
    
    /* Get a handle the the database record */
    recHandle = (Handle)DmGetRecord(obstratDB, index);

    /* Set the text field and repaint it */
    FldSetTextHandle(fld, recHandle);
    FldDrawField(fld);

    /* Release the record handle */
    DmReleaseRecord(obstratDB, index, false);

    /* Clear the handle value in the field, otherwise the handle */
    /*  will be free when the form is disposed of, resulting in  */
    /*  an error.                                                */
    FldSetTextHandle(fld, 0);
}


/*
** AboutViewHandleEvent
**   Event handler for the About form.
** Inputs
**   event    Pointer to an event type structure.
** Returns
**   True if the event has been dealt with.
*/
static Boolean AboutViewHandleEvent(EventPtr event)
{
    Boolean handled = false;
    
    switch (event->eType)
    {
        case frmOpenEvent:
            FrmDrawForm(FrmGetActiveForm());
            handled = true;
            break;
        case ctlSelectEvent:
            if (event->data.popSelect.controlID == btnID_AboutOK)
            {
                FrmReturnToForm(frmID_Main);
                handled = true;
                break;
            }
    }
    return(handled);
}


/*
** PrevRecord
**   Moves to the previous record in the database and displays it.
*/
void PrevRecord(void)
{
    CurrNo = CurrNo--;
    if (CurrNo < 0)
        CurrNo = DmNumRecords(obstratDB) - 1;
    DisplayCard();
}


/*
** NextRecord
**   Moves to the next record in the database and displays it.
*/
void NextRecord(void)
{
    CurrNo = CurrNo++;
    if (CurrNo > DmNumRecords(obstratDB) - 1)
        CurrNo = 0;
    DisplayCard();
}


/*
** MainViewHandleEvent
**   Event handler for the Main form.
** Inputs
**   event    Pointer to an event type structure.
** Returns
**   True if the event has been dealt with.
*/
static Boolean MainViewHandleEvent(EventPtr event)
{
    Boolean handled = false;
    
    switch (event->eType) {
        case frmOpenEvent:
            FrmDrawForm(FrmGetActiveForm());
            if (obstratDB)
                DisplayCard();
            break;
        case ctlSelectEvent:
            if (event->data.ctlEnter.controlID == btnID_MainRandom) {
                if (!obstratDB) {
                    FrmAlert(altID_NoDatabase);
                } else {
                    CurrNo = RandN(DmNumRecords(obstratDB));
                    DisplayCard();
                    handled = true;
                }
            }
            if (event->data.ctlEnter.controlID == btnID_MainPrevious) {
                if (!obstratDB) {
                    FrmAlert(altID_NoDatabase);
                } else {
                    PrevRecord();
                    handled = true;
                }
            }
            if (event->data.ctlEnter.controlID == btnID_MainNext) {
                if (!obstratDB) {
                    FrmAlert(altID_NoDatabase);
                } else {
                    NextRecord();
                    handled = true;
                }
            }
        case menuEvent:
            MenuEraseStatus(0);
            switch (event->data.menu.itemID) {
                case mniID_MainAbout:
                    FrmPopupForm(frmID_About);
                    handled = true;
                    break;
            }
            handled = true;
            break;
        case keyDownEvent:
            if (event->data.keyDown.chr == pageDownChr) {
                /* Scroll down hardware button pressed */
                NextRecord();
                handled = true;
            } else if (event->data.keyDown.chr == pageUpChr) {
                /* Scroll up hardware button pressed */
                PrevRecord();
                handled = true;
            }
    }
    return(handled);
}


/*
** ApplicationHandleEvent
**   Loads a form and sets the event handler for the form.
** Inputs
**   event    Pointer to an event type structure.
** Returns
**   True if the event has been dealt with.
*/
static Boolean ApplicationHandleEvent(EventPtr event)
{
    FormPtr frm;
    Int formId;
    Boolean handled = false;
    
    if (event->eType == frmLoadEvent) {
        formId = event->data.frmLoad.formID;
        frm = FrmInitForm(formId);
        FrmSetActiveForm(frm);
        
        switch (formId) {
            case frmID_Main:
                FrmSetEventHandler(frm, MainViewHandleEvent);
                break;
            case frmID_About:
                FrmSetEventHandler(frm, AboutViewHandleEvent);
                break;
        }
        handled = true;
    }
    return handled;
}


/*
** OpenDatabase
**   Opens the application database.
*/
void OpenDatabase(void)
{
    UInt         index = 0;
    LocalID      dbID;
    UInt         cardNo;
    Word         attributes;
    
    obstratDB = DmOpenDatabaseByTypeCreator(obstratDBType, obstratAppID,
                                            dmModeReadOnly);
    if (!obstratDB) {
        /* We've got no databse; ask the user to install one. */
        FrmAlert(altID_NoDatabase);
    } else {
        /* Flip the backup bit off if it's set.  Due to the way the  */
        /*  databases were created, they have their backup bits set, */
        /*  so we'll turn them off here to prevent the database from */
        /*  unneccessarily hotsyncing.                               */

        DmOpenDatabaseInfo(obstratDB, &dbID, NULL, NULL, &cardNo, NULL);
        DmDatabaseInfo(cardNo, dbID, NULL, &attributes, NULL, NULL,
                       NULL, NULL, NULL, NULL, NULL, NULL, NULL);
        if (attributes & dmHdrAttrBackup) {
            attributes ^= dmHdrAttrBackup;
            DmSetDatabaseInfo(cardNo, dbID, NULL, &attributes, NULL, NULL,
                              NULL, NULL, NULL, NULL, NULL, NULL, NULL);
        }
    }
}


/*
** StartApplication
**   Takes care of things which need to be done when opening the app.
*/
void StartApplication(void)
{
    ObStratPrefs prefs;
    
    /* Open the application database */
    OpenDatabase();
    
    /* Get application preferences */
    if (! PrefGetAppPreferencesV10(obstratAppID, obstratVersion, &prefs,
                                sizeof(ObStratPrefs))) {
        /* Set defaults if no saved prefs exist*/
        prefs.CurrNo = 0;
    }
    CurrNo = prefs.CurrNo;
    /* If the CurrNo is greater than the highest database record, set it to */
    /*  equal the highest record.  This can happen if a large database is   */
    /*  installed, a record near the end of that database is displayed, and */
    /*  then a smaller database is installed.  If the last record displayed */
    /*  from the larger datbase is higher than the last record in the small */
    /*  datbase, an "Index out of range" error occurs.                      */
    if (CurrNo > (DmNumRecords(obstratDB) - 1))
        CurrNo = DmNumRecords(obstratDB) - 1;
    
    /* Fire up the main form */
    FrmGotoForm(frmID_Main);
}


/*
** StopApplication
**   Save application state and prepare for switching to a new app.
*/
void StopApplication(void)
{
    ObStratPrefs prefs;
    
    /* Close the database */
    if (obstratDB)
        DmCloseDatabase(obstratDB);
    
    /* Save prefs so we start again in the same state we left */
    PrefGetAppPreferencesV10(obstratAppID, obstratVersion, &prefs,
                          sizeof(ObStratPrefs));
    prefs.CurrNo = CurrNo;
    PrefSetAppPreferencesV10(obstratAppID, obstratVersion, &prefs,
                          sizeof(ObStratPrefs));

    FrmCloseAllForms();
}


/*
** EventLoop
**   Hands out events from the Event Manager to the system,
**   menu, and application event handlers before passing them
**   to the application for processing.
*/
static void EventLoop(void)
{
    EventType event;
    Word error;
    do
    {
        EvtGetEvent(&event, evtWaitForever);
        if (! SysHandleEvent(&event))
            if (! MenuHandleEvent(NULL, &event, &error))
                if (! ApplicationHandleEvent(&event))
                    FrmDispatchEvent(&event);
    }
    while (event.eType != appStopEvent);
}

/*
** PilotMain
**   Entry point for the app.
** Inputs
**   cmd          command specifying how to launch the app
**   cmdPBP       parameter block for the command
**   launchFlags  flags used to configure the launch
** Returns
**   Error code
*/
DWord PilotMain (Word cmd, Ptr cmdPBP, Word launchFlags)
{
    if (cmd == sysAppLaunchCmdNormalLaunch) {
        StartApplication();
        EventLoop();
        StopApplication();
    }
    return 0;
}
