/*
 * DB: Database Design Form
 * Copyright (C) 1998  by Tom Dyas (tdyas@vger.rutgers.edu)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>

#include "enum.h"
#include "db.h"
#include "callback.h"

#define GetObjectPtr(f,i) FrmGetObjectPtr((f),FrmGetObjectIndex((f),(i)))

#define oldNameColumn    0
#define fieldNameColumn  1
#define fieldTypeColumn  2

static Word TopVisibleField;
static VoidHand * handles;
static Word * oldFieldNums;
static CharPtr * oldFieldNames;
static Boolean DoInsert;
DBInfoType design_info;
Boolean DesignNewDB;

static void
UpdateScrollers(void)
{
    Word upIndex, downIndex, fieldNum;
    Boolean scrollUp, scrollDown;
    SWord row;
    FormPtr form;
    TablePtr table;

    if (DesignNewDB) {
	scrollUp = TopVisibleField != 0;

	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_DesignView_Table);
	row = TblGetLastUsableRow(table);
	if (row != -1) {
	    fieldNum = TblGetRowID(table, row);
	    scrollDown = fieldNum < dbMaxFields - 1;
	}
    } else {
	scrollUp = TopVisibleField != 0 && design_info.numFields > 0;

	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_DesignView_Table);
	row = TblGetLastUsableRow(table);
	if (row != -1 && row != 0) {
	    fieldNum = TblGetRowID(table, row);
	    scrollDown = fieldNum < design_info.numFields - 1;
	} else {
	    scrollDown = false;
	}
    }

    upIndex = FrmGetObjectIndex(form, ctlID_DesignView_UpButton);
    downIndex = FrmGetObjectIndex(form, ctlID_DesignView_DownButton);
    FrmUpdateScrollers(form, upIndex, downIndex, scrollUp, scrollDown);

    if (!DesignNewDB) {
	if (design_info.numFields < dbMaxFields)
	    FrmShowObject(form, FrmGetObjectIndex(form, ctlID_DesignView_AddButton));
	else
	    FrmHideObject(form, FrmGetObjectIndex(form, ctlID_DesignView_AddButton));

	if (design_info.numFields > 0)
	    FrmShowObject(form, FrmGetObjectIndex(form, ctlID_DesignView_DelButton));
	else
	    FrmHideObject(form, FrmGetObjectIndex(form, ctlID_DesignView_DelButton));
    }
}

static Err
LoadData(VoidPtr table, Word row, Word column,
                   Boolean editing, VoidHand * textH, WordPtr textOffset,
                   WordPtr textAllocSize, FieldPtr fld)
{
    Word fieldNum;
    CharPtr p;
    FieldAttrType attr;
    VoidHand handle;

    CALLBACK_PROLOGUE;
    fieldNum = (Word) TblGetRowID(table, row);

    if (MemHandleSize(handles[fieldNum]) < (32+1))
	MemHandleResize(handles[fieldNum], 32+1);

    p = MemHandleLock(handles[fieldNum]);
    StrCopy(p, design_info.fields[fieldNum].name);
    MemPtrUnlock(p);

    *textH = handles[fieldNum];
    *textOffset = 0;
    *textAllocSize = 32+1;

    if (editing && fld) {
	FldGetAttributes(fld, &attr);
	attr.singleLine = 1;
	attr.dynamicSize = 0;
	attr.underlined = 1;
	FldSetAttributes(fld, &attr);
	
	FldSetMaxChars(fld, 32);
    }

    CALLBACK_EPILOGUE;

    return (0);
}           

static Boolean
SaveData(VoidPtr table, Word row, Word col)
{
    Word fieldNum;
    CharPtr p;

    CALLBACK_PROLOGUE;

    fieldNum = TblGetRowID(table, row);
    p = MemHandleLock(handles[fieldNum]);
    StrCopy(design_info.fields[fieldNum].name, p);
    MemPtrUnlock(p);

    CALLBACK_EPILOGUE;
    return false;
}

static void
LoadTable(TablePtr table)
{
    Word numRows, row, fieldNum;
    ListPtr list;
    FormPtr form;

    form = FrmGetActiveForm();
    list = GetObjectPtr(form, ctlID_DesignView_PopupList);

    numRows = TblGetNumberOfRows(table);
    fieldNum = TopVisibleField;
    for (row = 0; row < numRows; row++) {
	if (DesignNewDB ?
	    (fieldNum < dbMaxFields) : (fieldNum < design_info.numFields)) {
	    if (! TblRowUsable(table, row)
		|| TblGetRowID(table, row) != fieldNum) {
		TblSetRowUsable(table, row, true);
		TblMarkRowInvalid(table, row);
		TblSetItemStyle(table, row, oldNameColumn, labelTableItem);
		TblSetItemPtr(table, row, oldNameColumn, oldFieldNames[fieldNum]);
		TblSetItemStyle(table, row, fieldNameColumn, textTableItem);
		TblSetItemStyle(table, row, fieldTypeColumn, popupTriggerTableItem);
		TblSetItemPtr(table, row, fieldTypeColumn, list);
		TblSetItemInt(table, row, fieldTypeColumn,
			      design_info.fields[fieldNum].type);
		TblSetRowID(table, row, fieldNum);
	    }
	} else {
	    TblSetRowUsable(table, row, false);
	}
	fieldNum++;
    }

    UpdateScrollers();
}

static void
SetColumnWidths(TablePtr table)
{
    SWord width;
    SWord i, w, m;
    FontID oldFont;

    oldFont = FntSetFont(stdFont);
    width = FntCharsWidth(oldFieldNames[0], StrLen(oldFieldNames[0]))
	+ FntCharsWidth(":", 1);
    m = DesignNewDB ? dbMaxFields : design_info.numFields;
    for (i = 1; i < m; i++) {
	w = FntCharsWidth(oldFieldNames[i], StrLen(oldFieldNames[i]))
	    + FntCharsWidth(":", 1);
	if (w > width)
	    width = w;
    }
    FntSetFont(oldFont);

    TblSetColumnWidth(table, oldNameColumn, width);
    TblSetColumnWidth(table, fieldNameColumn, 110 - width);
    TblSetColumnWidth(table, fieldTypeColumn, 50);
}

static void
InitTable(TablePtr table)
{
    Word row, numRows;

    numRows = TblGetNumberOfRows(table);
    for (row = 0; row < numRows; row++)
	TblSetRowUsable(table, row, false);

    SetColumnWidths(table);

    TblSetColumnUsable(table, oldNameColumn, true);
    TblSetColumnUsable(table, fieldNameColumn, true);
    TblSetColumnUsable(table, fieldTypeColumn, true);

    TblSetLoadDataProcedure(table, fieldNameColumn, LoadData);
    TblSetSaveDataProcedure(table, fieldNameColumn, SaveData);

    LoadTable(table);
}

static void
Scroll(DirectionType direction)
{
    UInt newTopVisibleField;

    newTopVisibleField = TopVisibleField;

    if (direction == down) {
	newTopVisibleField++;
    } else {
	newTopVisibleField--;
    }

    if (TopVisibleField != newTopVisibleField) {
	FormPtr form;
	TablePtr table;

	TopVisibleField = newTopVisibleField;

	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_DesignView_Table);
	LoadTable(table);
	TblRedrawTable(table);
    }
}

static Boolean
UpdateNewDesign(void)
{
    FormPtr form;
    FieldPtr fld;
    VoidHand name_handle;
    CharPtr name;
    int i;

    /* Update the number of fields by finding the last nonempty field. */
    for (i = dbMaxFields - 1; i >= 0; i--) {
	if (design_info.fields[i].name[0] != '\0')
	    break;
    }
    design_info.numFields = i + 1;
    if (design_info.numFields == 0) {
	FrmAlert(alertID_NoFields);
	return false;
    }

    /* Verify that we can use the database name. */
    form = FrmGetActiveForm();
    fld = GetObjectPtr(form, ctlID_DesignView_NameField);
    name_handle = (VoidHand) FldGetTextHandle(fld);
    name = MemHandleLock(name_handle);
    if (name[0] == '\0' || DmFindDatabase(0, name) != 0) {
	FrmAlert(name[0] == '\0' ? alertID_NameBlank : alertID_NameConflict);
	MemPtrUnlock(name);
	return false;
    }

    /* Create the database. Creation routine will display any errors. */
    CreateDatabase(name, &design_info);
    MemPtrUnlock(name);

    return true;
}

static Boolean
UpdateExistingDesign(void)
{
    FormPtr form;
    FieldPtr fld;
    VoidHand name_handle;
    CharPtr name;
    Char dbname[dmDBNameLength+1];
    UInt cardNo;
    LocalID dbID;
    Boolean updateDesign = false;

    /* Validate any changes to the database name. */
    form = FrmGetActiveForm();
    fld = GetObjectPtr(form, ctlID_DesignView_NameField);
    name_handle = (VoidHand) FldGetTextHandle(fld);
    name = MemHandleLock(name_handle);
    if (name[0] == '\0') {
	FrmAlert(alertID_NameBlank);
	MemPtrUnlock(name);
	return false;
    }

    DmOpenDatabaseInfo(CurrentDB, &dbID, 0, 0, &cardNo, 0);
    DmDatabaseInfo(cardNo, dbID, dbname, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    if (StrCompare(dbname, name) != 0) {
	if (DmFindDatabase(cardNo, name) != 0) {
	    FrmAlert(alertID_NameConflict);
	    MemPtrUnlock(name);
	    return false;
	}
    }

    /* The database design is not allowed to be changed at this time. */
    if (design_info.numFields != dbInfo.numFields) {
	FrmAlert(alertID_DesignChange);
	MemPtrUnlock(name);
	return false;
    } else {
	Word i;

	for (i = 0; i < design_info.numFields; i++) {
	    /* Reordering fields or changing field type is a no-no. */
	    if (oldFieldNums[i] != i
		|| design_info.fields[i].type != dbInfo.fields[i].type) {
		FrmAlert(alertID_DesignChange);
		MemPtrUnlock(name);
		return false;
	    }

	    /* If the field name changed, then we update the design. */
	    if (StrCompare(design_info.fields[i].name,
			   dbInfo.fields[i].name) != 0) {
		updateDesign = true;
	    }
	}
    }

    /* Make the database name change. */
    if (StrCompare(dbname, name) != 0) {
	DmSetDatabaseInfo(cardNo, dbID, name, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
    }

    /* Make any database design changes. */
    if (updateDesign) {
	LocalID appInfoID;
	VoidPtr appInfo;
	UInt attr;

	appInfoID = DmGetAppInfoID(CurrentDB);
	appInfo = MemLocalIDToLockedPtr(appInfoID, cardNo);
	DmWrite(appInfo, 0, &design_info, sizeof(DBInfoType));
	MemPtrUnlock(appInfo);

	DmDatabaseInfo(cardNo, dbID, 0, &attr, 0, 0, 0, 0, 0, 0, 0, 0, 0);
	attr |= dmHdrAttrAppInfoDirty;
	DmSetDatabaseInfo(cardNo, dbID, 0, &attr, 0, 0, 0, 0, 0, 0, 0, 0, 0);

	MemMove(&dbInfo, &design_info, sizeof(DBInfoType));
    }

    MemPtrUnlock(name);
    return true;
}

Boolean
DesignViewHandleEvent(EventPtr event)
{
    FormPtr form;
    FieldPtr fld;
    TablePtr table;
    VoidHand name_handle;
    CharPtr name;
    Word i;

    switch (event->eType) {
    case frmOpenEvent:
	/* Allocate temporary memory space. */
	handles = MemPtrNew(dbMaxFields * sizeof(VoidHand));
	oldFieldNames = MemPtrNew(dbMaxFields * sizeof(CharPtr));
	oldFieldNums = MemPtrNew(dbMaxFields * sizeof(Word));
	for (i = 0; i < dbMaxFields; i++) {
	    handles[i] = MemHandleNew(32+1);
	    oldFieldNames[i] = MemPtrNew(32+1);
	    oldFieldNums[i] = i;
	}

	form = FrmGetActiveForm();
	if (DesignNewDB) {
	    for (i = 0; i < dbMaxFields; i++) {
		design_info.fields[i].name[0] = '\0';
		design_info.fields[i].type = 0;
		design_info.fields[i].colwidth = 80;
		StrPrintF(oldFieldNames[i], "Fld %d", i + 1);
	    }
	    FrmCopyTitle(form, "New Database");
	    FrmHideObject(form, FrmGetObjectIndex(form, ctlID_DesignView_AddButton));
	    FrmHideObject(form, FrmGetObjectIndex(form, ctlID_DesignView_DelButton));

	} else {
	    MemMove(&design_info, &dbInfo, sizeof(DBInfoType));
	    for (i = 0; i < design_info.numFields; i++)
		StrCopy(oldFieldNames[i], design_info.fields[i].name);

	    FrmCopyTitle(form, "DB Design");
	}

	/* Update the name field. */
	fld = GetObjectPtr(form, ctlID_DesignView_NameField);
	name_handle = MemHandleNew(dmDBNameLength+1);
	name = MemHandleLock(name_handle);
	name[0] = '\0';
	if (! DesignNewDB) {
	    Char dbname[dmDBNameLength+1];
	    UInt cardNo;
	    LocalID dbID;

	    DmOpenDatabaseInfo(CurrentDB, &dbID, 0, 0, &cardNo, 0);
	    DmDatabaseInfo(cardNo, dbID, dbname, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
	    StrCopy(name, dbname);
	}
	MemPtrUnlock(name);

	FldSetTextHandle(fld, (Handle) name_handle);
	FldSetMaxChars(fld, dmDBNameLength);

	/* Initialize the table that holds the design information. */
	table = GetObjectPtr(form, ctlID_DesignView_Table);
	InitTable(table);

	/* Draw the form with field #1 appearing at the top. */
	TopVisibleField = 0;
	FrmDrawForm(form);
	return true;

    case frmCloseEvent: {
	CALLBACK_PROLOGUE;
	for (i = 0; i < dbMaxFields; i++) {
	    MemHandleFree(handles[i]);
	    MemPtrFree(oldFieldNames[i]);
	}
	MemPtrFree(handles);
	MemPtrFree(oldFieldNames);
	MemPtrFree(oldFieldNums);
	CALLBACK_EPILOGUE;
	break;
    }

    case tblSelectEvent:
	if (event->data.tblSelect.column == fieldTypeColumn) {
	    Word fieldNum;

	    fieldNum = TblGetRowID(event->data.tblSelect.pTable,
				   event->data.tblSelect.row);
	    design_info.fields[fieldNum].type = TblGetItemInt(event->data.tblSelect.pTable,event->data.tblSelect.row,event->data.tblSelect.column);
	    return true;
	} else if (event->data.tblSelect.column == oldNameColumn) {
	    /* A tap on the label edits the field name. */
	    TblReleaseFocus(event->data.tblSelect.pTable);
	    TblUnhighlightSelection(event->data.tblSelect.pTable);
	    TblGrabFocus(event->data.tblSelect.pTable,
			 event->data.tblSelect.row, fieldNameColumn);
	    fld = TblGetCurrentField(event->data.tblSelect.pTable);
	    if (fld) {
		FldGrabFocus(fld);
		FldMakeFullyVisible(fld);
	    }
	    return true;
	}
	break;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_DesignView_DoneButton:
	    if (DesignNewDB) {
		if (! UpdateNewDesign())
		    return true;
	    } else {
		if (! UpdateExistingDesign())
		    return true;
	    }

	case ctlID_DesignView_CancelButton:
	    if (DesignNewDB)
		FrmGotoForm(formID_Chooser);
	    else
		FrmGotoForm(formID_ListView);

	    return true;

	case ctlID_DesignView_AddButton:
	    DoInsert = true;
	    FrmPopupForm(formID_DesignView_SelectField);
	    return true;

	case ctlID_DesignView_DelButton:
	    DoInsert = false;
	    FrmPopupForm(formID_DesignView_SelectField);
	    return true;
	}
	break;

    case ctlRepeatEvent:
	switch (event->data.ctlRepeat.controlID) {
	case ctlID_DesignView_UpButton:
	    Scroll(up);
	    break;
	case ctlID_DesignView_DownButton:
	    Scroll(down);
	    break;
	}
	break;

    case frmUpdateEvent:
	form = FrmGetActiveForm();
	table = GetObjectPtr(form, ctlID_DesignView_Table);
	InitTable(table);
	TblMarkTableInvalid(table);
	TblDrawTable(table);
	return true;

    case menuEvent:
	return HandleCommonMenuEvent(event->data.menu.itemID);

    }

    return false;
}

static Word WhichField;
static Char delname[32+1];

Boolean
SelectFieldHandleEvent(EventPtr event)
{
    FormPtr form;
    ControlPtr control;
    ListPtr list;
    CharPtr s;
    CharPtr * a;
    Word i;
    RectangleType r;
    SWord j;

    switch (event->eType) {
    case frmOpenEvent:
	WhichField = 0;

	form = FrmGetActiveForm();

	/* Setup the form title and label. */
	if (DoInsert) {
	    FrmCopyTitle(form, "Insert Field");
	    FrmCopyLabel(form, ctlID_DesignViewDel_Label,
			 "Select where to insert field:");
	} else {
	    FrmCopyTitle(form, "Delete Field");
	    FrmCopyLabel(form, ctlID_DesignViewDel_Label,
			 "Select field to delete:");
	}

	/* Update the field popup trigger with the correct label. */
	control = GetObjectPtr(form, ctlID_DesignViewDel_Trigger);
	s = MemHandleLock(handles[WhichField]);
	StrCopy(delname, s);
	MemPtrUnlock(s);
	CtlSetLabel(control, delname);

	FrmDrawForm(form);
	return true;

    case ctlSelectEvent:
	switch (event->data.ctlSelect.controlID) {
	case ctlID_DesignViewDel_OkayButton:
	    if (DoInsert) {
		/* Move the other fields up. */
		design_info.numFields++;
		for (i = WhichField + 1; i < design_info.numFields; i++) {
		    MemMove(&(design_info.fields[i]),
			    &(design_info.fields[i-1]),
			    sizeof(design_info.fields[i]));
		    StrCopy(oldFieldNames[i], oldFieldNames[i-1]);
		    oldFieldNums[i] = oldFieldNums[i-1];
		}

		/* Fill in the information on this field. */
		StrCopy(design_info.fields[WhichField].name, "New Field");
		StrCopy(oldFieldNames[WhichField], "New");
		oldFieldNums[WhichField] = dbMaxFields;
		design_info.fields[WhichField].type = 0;
		design_info.fields[WhichField].colwidth = 80;
	    } else {
		/* Move the other fields down. */
		for (i = WhichField; i < design_info.numFields - 1; i++) {
		    MemMove(&(design_info.fields[i]),
			    &(design_info.fields[i+1]),
			    sizeof(design_info.fields[i]));
		    StrCopy(oldFieldNames[i], oldFieldNames[i+1]);
		    oldFieldNums[i] = oldFieldNums[i+1];
		}
		oldFieldNames[dbMaxFields-1][0] = '\0';
		design_info.numFields--;
	    }
	    FrmReturnToForm(formID_DesignView);
	    FrmUpdateForm(formID_DesignView, 1);
	    return true;

	case ctlID_DesignViewDel_CancelButton:
	    FrmReturnToForm(formID_DesignView);
	    return true;

	case ctlID_DesignViewDel_Trigger:
	    form = FrmGetActiveForm();
	    list = GetObjectPtr(form, ctlID_DesignViewDel_List);
	    LstSetListChoices(list, oldFieldNames, design_info.numFields);
	    LstSetHeight(list, design_info.numFields);
	    FrmGetObjectBounds(form, FrmGetObjectIndex(form, ctlID_DesignViewDel_Trigger), &r);
	    LstSetPosition(list, r.topLeft.x, r.topLeft.y);
	    j = LstPopupList(list);
	    if (j >= 0) {
		if (j != WhichField) {
		    WhichField = j;
		    StrCopy(delname, oldFieldNames[j]);
		    CtlSetLabel(event->data.ctlSelect.pControl, delname);
		}
	    }
	    return true;
	}
	break;
    }

    return false;
}
