#pragma pack(2)

#include <Common.h>
#include <System/SysAll.h>
#include <UI/UIAll.h>

#include "nimid.h"
#include "nim.h"

/* "Preferences" structure:  What we remeber when off */
struct {
	int count, pilesize, goal, skill;	/* configuration stuff */
	char cols[MAXCOUNT];			/* and board position */
} p;

int xmin, xmax, ymin, ymax;	/* calculated display parameters */

int cleanup, cleanhigh;		/* remember to undraw computer moves */
ULong cleantime;

/* [re]calculate the display parameters */
void prefcalc(void)
{
	xmin = 80 - p.count*XSPACE/2;
	xmax = xmin + p.count * XSPACE - 1;
	ymin = BASELINE - p.pilesize * YSPACE;
	ymax = ymin + p.pilesize * YSPACE - 1;
	cleanup = -1;
}

/* Draw block in position x,y as {empty, full, hollow} */
void doblock(int x, int y, int draw)
{
	RectangleType rect;
	Word diam;

	x = x*XSPACE+xmin;
	y = ((p.pilesize-1)-y)*YSPACE+ymin;

	rect.topLeft.x = x+2;
	rect.topLeft.y = y+1;
	rect.extent.x = XSPACE-4;
	rect.extent.y = YSPACE-2;
	diam = 3;
	
	switch (draw) {
		case 0:
			WinEraseRectangle(&rect,diam);
			break;
		case 1:
			WinDrawRectangle(&rect,diam);
			break;
		case 2:
			WinDrawRectangle(&rect,diam);
			rect.topLeft.x += 1;
			rect.topLeft.y += 1;
			rect.extent.x -= 2;
			rect.extent.y -= 2;
			WinEraseRectangle(&rect,diam);
			break;
	}
}

/* Record a column height */
static void setcol(int x, int y)
{
	p.cols[x] = y;
}

/* Draw a column */
void drawcol(int x)
{
	int loop, y;

	y = p.cols[x];
	for (loop=0; loop<p.pilesize; loop++)
	{
		if (loop < y)
			doblock(x,loop,1);
		else
			doblock(x,loop,0);
	}
}

/* Draw all columns */
void drawall(void)
{
	int i;

	for (i=0; i<p.count; i++)
	{
		drawcol(i);
	}
	if (cleanup != -1)
	{
		for (i=p.cols[cleanup]; i<cleanhigh; i++)
		{
			doblock(cleanup,i,2);
		}
	}
}

/* Scramble for a new game */
void randomize(void)
{
	int i;

	for (i=0; i<p.count; i++)
	{
		/* +1 is to prevent empty (boring) piles */
		setcol(i,(SysRandom(0) % p.pilesize)+1);
	}
	for (i=p.count; i < MAXCOUNT; i++)
	{
		setcol(i,0);
	}
}

/* check for empty board */
int board_empty(void)
{
	int loop;

	for (loop=0; loop<p.count; loop++)
	{
		if (p.cols[loop] != 0)
			return(0);
	}
	return(1);
}

/* check for real choices */
int board_nchoice(void)
{
	int loop, ch;

	ch = 0;
	for (loop=0; loop<p.count; loop++)
	{
		if (p.cols[loop] > 1)
			ch++;
	}
	return(ch);
}

void computer_move(int x, int y)
{
	int loop;

	cleanhigh = p.cols[x];
	cleantime = TimGetTicks() + sysTicksPerSecond;
	setcol(x,y);
	drawcol(x);
	for (loop=y; loop<cleanhigh; loop++)
	{
		doblock(x,loop,2);
	}
	cleanup = x;
}

/* make a random computer move, due to either no good move or stupidity */
void move_randomly(void)
{
	int var, loop, col;

	var = SysRandom(0) % p.count;	/* pick a random column */
	for (loop=var; loop<p.count+var; loop++)	/* look for a pile */
	{
		col = loop%p.count;	/* which real column */
		if (p.cols[col] != 0)	/* if not empty */
		{
			computer_move(col, SysRandom(0)%p.cols[col]);
			return;
		}
	}
	/* failed to move -- shouldn't happen */
	cleanup = -1;
}

int skill_exceeded(void) {
	int var, loop;

	switch(p.skill) {
		case 1: /* random */
			return(1);
			break;
		case 2: /* inept */
			var = 0;
			for (loop=0; loop<p.count; loop++)
			{
				var += p.cols[loop];
				if (p.cols[loop] > 1)
					var += p.cols[loop] + 10;
			}
			return (SysRandom(0)%var > 20);
			break;
		case 3: /* good */
			var = 0;
			for (loop=0; loop<p.count; loop++)
			{
				var += p.cols[loop];
				if (p.cols[loop] > 1)
					var += p.cols[loop] + 10;
			}
			return (SysRandom(0)%var > 50);
			break;
		case 4: /* perfect */
		default: /* treat anything else as perfect too */
			return(0);
			break;
	}
}

/* do something when the player has entered a move */
void docol(int x, int y)
{
	int loop, var;

	if (y >= p.cols[x]) /* didn't select a block -- too high */
		return;	/* do nothing */
	if (cleanup != -1)	/* just in case */
		drawcol(cleanup);	/* erase old computer move */
	cleanup = -1;
	setcol(x,y); /* do player move */
	drawcol(x); /* display player move */
	if (board_empty())
	{
		if (p.goal == 1)
			FrmAlert(Player_win); /* You win */
		else
			FrmAlert(Pilot_win); /* I win */
		randomize();
		drawall();
		return;
	}
	/* There are blocks left... */
	/* I'm deliberately not being clear on the computer strategy.  It's
	   more fun to work it out on your own.  But if you really don't want
	   to, consult "Winning Ways" or "The Book of Numbers" (both math books
	   that cover Nim).  Either that or email me. */
	var = 0;
	for (loop=0; loop<p.count; loop++)
	{
		var ^= p.cols[loop];
	}
	if ((var == 0) || skill_exceeded()) {
		move_randomly();
	} else if ((p.goal == 2) && (board_nchoice() == 1)) {
		for (loop=0; loop<p.count; loop++)
		{
			if (p.cols[loop] > 1)
			{
				computer_move(loop, (var+p.cols[loop]+1)%2);
				break;
			}
		}
	} else {
		for (loop=0; loop<p.count; loop++)
		{
			if ((var ^ p.cols[loop]) < p.cols[loop])
			{
				computer_move(loop,var ^ p.cols[loop]);
				break;
			}
		}
	}
	if (board_empty())	/* game may be over */
	{
		if (p.goal == 1)
			FrmAlert(Pilot_win); /* I win */
		else
			FrmAlert(Player_win); /* You win */
		cleanup = -1;
		randomize();
		drawall();
		return;
	}
}

void drawframe(void)
{
	int i;

	if (p.goal == 1)
		WinDrawChars("Take last",9,100,2);
	else
		WinDrawChars("Leave last",10,100,2);
	for (i=1; i<5; i++)
	{
		WinEraseLine(0, ymax+i, xmin-i-1, ymax+i);
		WinDrawLine(xmin-i, ymax+i, xmax+i, ymax+i);
		WinEraseLine(xmax+i+1, ymax+i, 159, ymax+i);
	}
	drawall();
}

void smash_piles(void)
{
	int loop;

	for (loop=0; loop<p.count; loop++)
	{
		if (p.cols[loop] > p.pilesize)
			p.cols[loop] = p.pilesize;
	}
}

DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
	EventType e;
	short err;
	int x, y;
	unsigned long appid = 0x4e696d30;	/* Nim0 */
	Int version = 0;
	ULong temptime;
	long delaytime;
	ControlPtr ctl;
	FormPtr frm;

	if (cmd == sysAppLaunchCmdNormalLaunch)
	{
		if (!PrefGetAppPreferences(appid,version,(void *)&p,sizeof(p)))
		{
			/* If we haven't run before, set up static data */
			p.goal = 1;	/* Normal: take last */
			p.count = 3;	/* 3 columns */
			p.pilesize = 8;	/* 8 high max */
			p.skill = 4;	/* perfect computer */
			/* calculate screen size parameters */
			prefcalc();
			/* set up a game */
			randomize();
		} else {
			/* calculate (unsaved) screen size parameters */
			prefcalc();
		}
		FrmGotoForm(MainForm);
		while(1)
		{
			if (cleanup != -1)	/* if we have cleanup to do */
			{
				temptime = TimGetTicks();
				if (temptime < cleantime)
					delaytime = cleantime - temptime;
				else
					delaytime = 1;	/* short wait */
			} else {
				delaytime = evtWaitForever;	/* whenever */
			}
			EvtGetEvent(&e, delaytime);
			/* system want it? */
			if (SysHandleEvent(&e))
				continue;
			/* menu want it? */
			if (MenuHandleEvent((void *)0, &e, &err))
				continue;
			switch(e.eType)
			{
				case nilEvent:	/* timer expired */
					if (cleanup != -1)
					{
						drawcol(cleanup);
						cleanup = -1;
					}
					break;
				case frmLoadEvent:
					FrmSetActiveForm(FrmInitForm(e.data.frmLoad.formID));
					break;
				case frmOpenEvent:
					frm = FrmGetActiveForm();
					if (FrmGetActiveFormID() == MainForm) {
						FrmDrawForm(frm);
						drawframe();
					} else {
						ctl = (ControlPtr)FrmGetObjectPtr(frm,FrmGetObjectIndex(frm,Nums + p.count));
						CtlSetValue(ctl,1);
						ctl = (ControlPtr)FrmGetObjectPtr(frm,FrmGetObjectIndex(frm,HGR + p.pilesize));
						CtlSetValue(ctl,1);
						ctl = (ControlPtr)FrmGetObjectPtr(frm,FrmGetObjectIndex(frm,Goal + p.goal));
						CtlSetValue(ctl,1);
						ctl = (ControlPtr)FrmGetObjectPtr(frm,FrmGetObjectIndex(frm,Skill + p.skill));
						CtlSetValue(ctl,1);
						FrmDrawForm(frm);
					}
					break;
				case frmUpdateEvent:
					FrmDrawForm(FrmGetActiveForm());
					if (FrmGetActiveFormID() == MainForm)
						drawframe();
					break;
				case frmCloseEvent:
					FrmHandleEvent(FrmGetActiveForm(), &e);
					break;
				case appStopEvent:
					PrefSetAppPreferences(appid,version,(void *)&p,sizeof(p));
					return(0);
				case penDownEvent:
					x = e.screenX;
					y = e.screenY;
					if ((x > xmin) && (x < xmax) &&
					    (FrmGetActiveFormID() == MainForm))
					{
						if ((y > ymin) && (y < ymax))
						{
							x = (x-xmin)/XSPACE;
							y = (p.pilesize-1)-(y-ymin)/YSPACE;
							docol(x,y);
						}
					} else {
						FrmHandleEvent(FrmGetActiveForm(), &e);
					}
					break;
				case ctlSelectEvent:
					switch(e.data.menu.itemID) {
						case CB1:
							p.count = 1;
							break;
						case CB2:
							p.count = 2;
							break;
						case CB3:
							p.count = 3;
							break;
						case CB4:
							p.count = 4;
							break;
						case CB5:
							p.count = 5;
							break;
						case CB6:
							p.count = 6;
							break;
						case CB7:
							p.count = 7;
							break;
						case HB1:
							p.pilesize = 1;
							smash_piles();
							break;
						case HB2:
							p.pilesize = 2;
							smash_piles();
							break;
						case HB3:
							p.pilesize = 3;
							smash_piles();
							break;
						case HB4:
							p.pilesize = 4;
							smash_piles();
							break;
						case HB5:
							p.pilesize = 5;
							smash_piles();
							break;
						case HB6:
							p.pilesize = 6;
							smash_piles();
							break;
						case HB7:
							p.pilesize = 7;
							smash_piles();
							break;
						case HB8:
							p.pilesize = 8;
							smash_piles();
							break;
						case HB9:
							p.pilesize = 9;
							smash_piles();
							break;
						case HB10:
							p.pilesize = 10;
							smash_piles();
							break;
						case GL1:
							p.goal = 1;
							break;
						case GL2:
							p.goal = 2;
							break;
						case SB1:
							p.skill = 1;
							break;
						case SB2:
							p.skill = 2;
							break;
						case SB3:
							p.skill = 3;
							break;
						case SB4:
							p.skill = 4;
							break;
						case PrefOkButton:
							FrmGotoForm(MainForm);
							prefcalc();
							break;
						default:
							break;
					}
					break;
				case menuEvent:
					switch(e.data.menu.itemID) {
						case 5000:	/* prefs */
							FrmGotoForm(PrefForm);
							prefcalc();
							break;
						case 5001:	/* new */
							randomize();
							drawall();
							break;
						case 5002:	/* empty line */
							break;
						case 5003:	/* about */
							FrmAlert(About);
							break;
						default:
							break;
					}
					break;
				default:
					FrmHandleEvent(FrmGetActiveForm(), &e);
			}
		}
	} else {
		return(sysErrParamErr);
	}
	return 0;
}
