%{
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>

#define IN_MAKELEVEL
#include "tiles.h"
#include "makelevel.h"
#include "mulg.h"

#define noDEBUG_CONST

#define MAX_IDENT  256
#define STR_LEN    32

#define MAX_LINES 33
#define MAX_ROWS 37
#define MAX_LEVELS 256

/* Palm<->Unix Time conversion */
#define PALM2UNIX(a)  (a - 2082844800)
#define UNIX2PALM(a)  (a + 2082844800)

char ident_name[MAX_IDENT][STR_LEN];
int  ident_value[MAX_IDENT];

int db_debug=0;

char string_buffer[256];
char dbname[32];

void yyerrorf(int offset, const char *fmt, ...);
void enter_file(char *name);

typedef struct {
  char name[32];
  int width, height;
  unsigned short data[MAX_LINES][MAX_ROWS];
} level;

char doc[256][256];

/* the hole level data */
int   levels=0;
level *level_data[MAX_LEVELS];

/* level buffer */
level level_buffer;

/* line buffer */
int line_len, line_buffer[MAX_ROWS];

int get_ident_value(int *ret, char *name) {
  int i;

  for(i=0;i<MAX_IDENT;i++) {
    if(strncmp(ident_name[i],name,STR_LEN)==0) {
      *ret = ident_value[i];
      return 1;
    }
  }
  *ret=0;
  return 0;
}

int add_ident_value(int value, char *name) {
  int i;

  /* search for existing entry */
  for(i=0;i<MAX_IDENT;i++) {
    if(strncmp(ident_name[i],name,STR_LEN)==0) {
      ident_value[i]=value;
#ifdef DEBUG_CONST
      printf("stored %s=%d\n", name, value);
#endif
      return 1;
    }
  }

  /* create new entry */
  for(i=0;i<MAX_IDENT;i++) {
    if(ident_name[i][0]==0) {
      strncpy(ident_name[i],name,32);
      ident_value[i]=value;
#ifdef DEBUG_CONST
      printf("stored new %s=%d\n", name, value);
#endif
      return 1;
    }
  }
  return 0;
}

%}
%union {
  int val;
  char name[STR_LEN];
}
%left '+' '-' '&' '|' LEFTSHIFT, RIGHTSHIFT
%left '*' '/'
%right '~'
%token IDENTIFIER, INTEGER, DEFINE, EOL, HEXINT, EXTERN, STRING, LEVEL, DOCUMENT
%token BININT, DBNAME, DBDEBUG
%type <name> IDENTIFIER STRING
%type <val> expr tile INTEGER HEXINT attribute constant BININT
%%
file    :
        | preproc file
        | error file
        ;

preproc : define
        | dbname
        | level
        | doc
        | EXTERN
        | EOL
        | DBDEBUG  { db_debug=1; }
        ;

ws      :
        | EOL ;

dbname  :  DBNAME STRING { if(dbname[0]!=0) yyerrorf(0, "database name redefined"); 
                           if(strlen(string_buffer)>32) yyerrorf(0, "database name must have 32 chars or less");
                           strcpy(dbname, string_buffer); }
        ;

doc     :  DOCUMENT expr STRING { if(doc[$2][0]!=0) yyerrorf(0, "document %d redefined", $2); 
                                  strcpy(doc[$2], string_buffer); }
        ;

level   :  LEVEL STRING ws '{' ws lines '}' 
                    { 
		      level_data[levels]=(level*)malloc(sizeof(level));
		      if(level_data[levels]==NULL) { yyerrorf(0,"out of memory"); exit(1); }

		      /* copy level into buffer */
		      memcpy(level_data[levels],&level_buffer,sizeof(level));

		      /* copy name */
		      strcpy(level_data[levels]->name, string_buffer);

		      levels++;
                    };

lines   :  line EOL { level_buffer.width=line_len;
                      level_buffer.height=1;
                      /* copy line to level buffer */
                      { int i; for(i=0;i<line_len;i++) level_buffer.data[0][i]=line_buffer[i]; } 
                    }
        |  lines line EOL {
                      if(line_len!=level_buffer.width) 
                        yyerrorf(-1, "illegal line length of %d objects, must be %d", line_len, level_buffer.width);

                      /* copy line to level buffer */
                      { int i; for(i=0;i<line_len;i++) level_buffer.data[level_buffer.height][i]=line_buffer[i]; }
                      level_buffer.height++;
                    }
        ;

line    :  tile { line_len=0; line_buffer[line_len++]=$1; }
        |  line tile { line_buffer[line_len++]=$2; } 
        ;

tile    :  IDENTIFIER {   if(!get_ident_value(&$$, $1)) { 
                          yyerrorf(0, "undefined constant %s",$1);
                          $$=STONE; } }
        |  IDENTIFIER '.' attribute {
                          if(!get_ident_value(&$$, $1)) { 
                          yyerrorf(0, "undefined constant %s",$1);
                          $$=STONE; }
			  /* add attribute to upper 8 bit */
                          $$ |= ($3<<8);
                      }
        ;

attribute      :  IDENTIFIER
                          { char modname[32];
                            strcpy(modname,"ATTRIB_"); 
                            strcat(modname, $1);

			    if(!get_ident_value(&$$, modname)) { 
                              yyerrorf(0, "undefined attribute %s",modname);
                              $$=0; 
                            } }
               |  constant  { $$ = $1 }
               ;

define  :  DEFINE IDENTIFIER expr EOL { if(!add_ident_value($3, $2)) yyerror(0,"too many identifiers");  }
        ;

expr  	:  '(' expr ')'           { $$ = $2; }
	|  expr '+' expr          { $$ = $1 + $3; }
	|  expr '-' expr          { $$ = $1 - $3; }
	|  expr '*' expr          { $$ = $1 * $3; }
	|  expr '/' expr          { $$ = $1 / $3; }
	|  expr '&' expr          { $$ = $1 & $3; }
	|  expr '|' expr          { $$ = $1 | $3; }
	|  expr LEFTSHIFT  expr   { $$ = $1 << $3; }
	|  expr RIGHTSHIFT expr   { $$ = $1 >> $3; }
	|  '~' expr               { $$ = ~$2; }
	|  constant
	|  IDENTIFIER      { if(!get_ident_value(&$$, $1)) yyerrorf(0,"undefined constant %s",$1); }
	;

constant   :  INTEGER  { $$=$1; }
	   |  HEXINT   { $$=$1; }
           |  BININT   { $$=$1; }

%%
#include "ml_lex.c"

yyerror(char *s) {
  printf("error in file '%s' line %d: %s\n",fname[include_stack_ptr],line[include_stack_ptr],s);
}

void 
yyerrorf(int offset, const char *fmt, ...) {
  char p[128];
  va_list ap;
  va_start(ap, fmt);
  (void) vsnprintf(p, 128, fmt, ap);
  va_end(ap);
  printf("error in file '%s' line %d: %s\n",fname[include_stack_ptr],line[include_stack_ptr]+offset,p);
}

void 
WriteWord(void *addr, uword data) {
  ubyte *to = (ubyte*)addr;
  ubyte *from = (ubyte*)&data;

  to[0] = from[1]; to[1] = from[0];
}

void 
WriteDWord(void *addr, udword data) {
  ubyte *to = (ubyte*)addr;
  ubyte *from = (ubyte*)&data;

  to[0] = from[3]; to[1] = from[2];
  to[2] = from[1]; to[3] = from[0];
}


main(int argc, char **argv) {
  FILE *outfile;
  int i,c,str_size,str_no, offset,x,y;
  time_t tim;
  char *p;
  PdbHeader pdb_header;
  RecHeader rec_header;
  unsigned short us;

  if((argc!=2)&&(argc!=3)) {
    puts("usage: makelevel infile [outfile]");
    exit(1);
  }

  yyin=fopen(argv[1],"r");

  strcpy(fname[0],argv[1]);
  line[0]=1;
  if(yyin==0) {
    fprintf(stderr,"error opening input file '%s'\n", argv[1]);
    exit(1);
  }

  /* clear arrays/strings */
  for(i=0;i<MAX_IDENT;i++)  ident_name[i][0]=0;
  for(i=0;i<256;i++)        doc[i][0]=0;
  dbname[0]=0;

  /* ok, parse the input file */
  yyparse();

  if(dbname[0]==0) {
    fprintf(stderr, "no database name given\n");
    exit(1);
  }

  /* more checks on the maze go here ... */

  if(argc==2) {
    if((p=strrchr(argv[1],'.'))!=NULL) {
      strcpy(p,".pdb");
    } else
      strcat(argv[1],".pdb");    
    outfile=fopen(argv[1],"wb");
  } else
    outfile=fopen(argv[2],"wb");

  if(outfile==NULL) {
    printf("error opening output file\n");
    exit(1);
  }

  tim=UNIX2PALM(time(0));
  
  /* create/write pdb header */
  memset(&pdb_header,0,sizeof(PdbHeader));
  strcpy(pdb_header.name,dbname);
  WriteWord(&(pdb_header.version),0x0001);
  WriteDWord(&(pdb_header.creationDate),tim);
  WriteDWord(&(pdb_header.modificationDate),tim);
  WriteDWord(&(pdb_header.databaseType),DB_TYPE);
  WriteDWord(&(pdb_header.creatorID),CREATOR);
  WriteWord(&(pdb_header.numberOfRecords),levels+2);
  fwrite(&pdb_header,sizeof(PdbHeader),1,outfile);

  /* find no/size of strings */
  for(str_no=0,str_size=0,i=0;i<256;i++) {
    if(doc[i][0]!=0) {
      str_no=i;
      str_size+=strlen(doc[i])+1;
    } 
  }

  offset = sizeof(PdbHeader) + (levels+2)*sizeof(RecHeader);

  /* write record header (0=High-Scores, 1=Strings, 2,... = Levels) */
  for(c = 0; c < levels+2 ;c++){
    WriteDWord(&(rec_header.recordDataOffset), offset);
    
    rec_header.recordAttributes = 0x60;
    rec_header.uniqueID[0] = rec_header.uniqueID[1] = rec_header.uniqueID[2] = 0x00;
    fwrite(&rec_header,sizeof(RecHeader),1,outfile);

    if(c==0) offset += 4 + levels*4; /* space for high scores, current level & max level */
    if(c==1) offset += 2*str_no + str_size;  /* space for doc strings */
    /* space for level data (32 bytes name, 2 bytes size, 2*w*h bytes data */
    if(c>=2) offset += 32 + 2 + (2 * level_data[c-2]->width * level_data[c-2]->height);
  }

  /* now write the records */

  if(db_debug) c=0x0000ffff;  /* all levels can be selected */
  else         c=0x00000000;  /* only first level can be selected */
  fwrite(&c,4,1,outfile);
  
  /* high scores */
  for(c=0xffffffff, i=0; i<levels;i++)  fwrite(&c,4,1,outfile);

  /* doc strings (first the offsets) */
  for(us=0,i=0;i<=str_no;i++) {
    fputc(us>>8, outfile);
    fputc(us&0xff, outfile);

    if(doc[i][0]!=0)
      offset+=strlen(doc[i])+1;
  }

  /* doc strings (data) */
  for(i=0;i<=str_no;i++)
    if(doc[i][0]!=0)
      fwrite(doc[i],strlen(doc[i])+1,1,outfile);

  /* and finally the levels */
  for(i=0;i<levels;i++) {
    fwrite(level_data[i]->name, 32, 1, outfile);
    fputc(level_data[i]->width,  outfile);
    fputc(level_data[i]->height, outfile);

    for(y=0;y<level_data[i]->height;y++) {
      for(x=0;x<level_data[i]->width;x++) {
	fputc((level_data[i]->data[y][x])>>8,   outfile);
	fputc((level_data[i]->data[y][x])&0xff, outfile);
      }
    }
  }

//  printf("%d levels processed\n", levels);

//  for(i=0;i<levels;i++) {
//    printf("Level %d: %s\n", i, level_data[i]->name);
//  }

  fclose(outfile);

  if(db_debug)
    printf("wrote debug version\n");

  return 0;
}













