/* ----------------------------------------------------- */
/* Fret Board Map (fretboard.c)                          */
/* Dave MacLeod <davmac@netcomuk.co.uk>                  */
/*                                                       */
/* This app displays a schematic of the guitar neck, and */
/* maps out notes, scales and chords.                    */
/* ----------------------------------------------------- */
/* v0.1     29/04/97    Creation                         */
/*                                                       */
/* ----------------------------------------------------- */
/* v0.2     17/05/97    Allow setting of preferences for */
/*                      both horizontal and vertical     */
/*                      orientation of the neck          */
/*                                                       */
/*                      Add quartic and quintal chords   */
/*                                                       */
/*                      Change the way notes are played  */
/*                                                       */
/* ----------------------------------------------------- */
/* v0.3     25/05/97    Fix bug in instrument and scale  */
/*                      selection                        */
/*                                                       */
/*                      When nut moves, move the display */
/*                      of the labels                    */
/*                                                       */
/*                      Add DADGAD tuning                */
/*                                                       */
/* ----------------------------------------------------- */
/* v0.4     06/06/97    Amend the way that notes, scales */
/*                      and chords are played.           */
/*                                                       */
/* ----------------------------------------------------- */
/* v0.5     16/06/97    Add window to show chord         */
/*                      spelling.                        */
/*                                                       */
/*                      Amend chord display to highlight */
/*                      the "significant" notes.         */
/*                                                       */
/*                      Add new funky splash/about       */
/*                      window.                          */
/*                                                       */
/*                      Make main window non-modal.      */
/*                                                       */
/* ----------------------------------------------------- */
/* v0.6     18/06/97    Fix bug with note playing        */
/*                                                       */
/* ----------------------------------------------------- */
/* v0.6a    18/06/97    Add Bouzouki                     */
/*                                                       */
/* ----------------------------------------------------- */
/* v0.7     19/11/97    Banjo tunings (G, C, D, Gm).     */
/*                                                       */
/*                      Add more bouzouki tunings (incl  */
/*                      5 string variants).              */
/*                                                       */
/*                      Quick access to chord spelling.  */
/*                                                       */
/* ----------------------------------------------------- */
/* v0.8     24/11/97    Add viola and cello.             */
/*                                                       */
/* ----------------------------------------------------- */
/* v0.9     25/11/97    Add chapman stick, CFAD          */
/*             to       bouzouki, lute, ukulele and      */
/*          01/12/97    renaissance guitar.              */
/* ----------------------------------------------------- */

#pragma pack(2)
#include <System/SysAll.h>
#include <UI/UIAll.h>

#define MainForm     1000
#define TypeLab      1051
#define NoteLab      1050
#define RootListT    1005
#define ModeListT    1008
#define ChrdListT    1011
#define InstListT    1014

#define HelpText     1000

#define RootList     1006
#define ModeList     1009
#define ChrdList     1012
#define InstList     1015

#define NoteBut      1001
#define ChordBut     1002
#define ScaleBut     1003
#define PlayBut      1013

#define FormAbout    1050
#define AboutDone    1051

#define aboutNeck    2000
#define menuNeck     3000
#define menuAbout    3001
#define menuSwapNut  3002
#define menuSwapStr  3003
#define menuHelp     3005

#define fretspace    8
#define numfret      15
#define nutos        2
#define labelos1     4
#define labelos2     15
#define noscales     16
#define numinst      37
#define scaleintvals 8
#define nochords     25
#define chordintvals 6

#define rootnote     1
#define boldnote     2
#define normnote     3


/* ----------------------------------------------------- */
/* Global Variables                                      */
/* ----------------------------------------------------- */

static Boolean PlaySnd;

static short offset[10],
             numstring,
             stringspace,
             neckorigx,
             neckorigy;

static char GChordName[20],
            GChordSpell[20];

typedef struct
{
  Boolean NutAtRight;
  Boolean ThickStringAtTop;
  short DispReqd,
        RootReqd,
        ChrdReqd,
        ModeReqd,
        InstReqd;
}
FretBoardPrefType;
FretBoardPrefType prefs;


/* ----------------------------------------------------- */
/* GetObjPtr                                             */
/*   Return the pointer to a form object                 */
/* Parms                                                 */
/*   objectID - The ID of the object, specified in the   */
/*              .rcp file.                               */
/* Returns                                               */
/*   A pointer to the specified object                   */
/* ----------------------------------------------------- */

static VoidPtr GetObjPtr(Int objectID)
{
  FormPtr frm;

  frm = FrmGetActiveForm();
  return(FrmGetObjectPtr(frm, FrmGetObjectIndex(frm, objectID)));
}

/* ----------------------------------------------------- */
/* DrawNote                                              */
/*   This draws a note on the neck, on the specified     */
/*   string, at the specified fret.                      */
/* Parms                                                 */
/*   StrNo  - The number of the string (0=Thick E)       */
/*   FretNo - The number of the fret (0=nut)             */
/*   Root   - True if the note concerned is the root.    */
/* Returns                                               */
/*   Nowt                                                */
/* ----------------------------------------------------- */

void DrawNote(short StrNo, short FretNo, short NoteType)
{
  RectangleType rect;
  short xpos,
        ypos;

  if (StrNo<numstring)
  {
    /* calculate horizontal screen position of the fret */
    if (prefs.NutAtRight)
      xpos = ((numfret-FretNo)*fretspace)+neckorigx;
    else
      xpos = (FretNo*fretspace)+neckorigx;

    /* calculate vertical screen position of the fret */
    if (prefs.ThickStringAtTop)
      ypos = (StrNo*stringspace)+neckorigy;
    else
      ypos = (((numstring-1)-StrNo)*stringspace)+neckorigy;

    switch (NoteType)
    { 
      case rootnote:
        rect.topLeft.x=xpos-2;
        rect.topLeft.y=ypos-2;
        rect.extent.x=5;
        rect.extent.y=5;
        WinDrawRectangle(&rect,0);
        rect.topLeft.x=xpos-1;
        rect.topLeft.y=ypos-1;
        rect.extent.x=3;
        rect.extent.y=3;
        WinEraseRectangle(&rect,0);
        break;
      case boldnote:
        rect.topLeft.x=xpos-2;
        rect.topLeft.y=ypos-2;
        rect.extent.x=5;
        rect.extent.y=5;
        WinDrawRectangle(&rect,3);
        break;
      case normnote:
        rect.topLeft.x=xpos-1;
        rect.topLeft.y=ypos-1;
        rect.extent.x=3;
        rect.extent.y=3;
        WinDrawRectangle(&rect,0);
        break;
    }
  }
}

/* ----------------------------------------------------- */
/* PlayNote                                              */
/*   This plays the required note                        */
/*   string, at the specified fret.                      */
/* Parms                                                 */
/*   Note - The note to play A=0...G#=11                 */
/* Returns                                               */
/*   Nowt                                                */
/* ----------------------------------------------------- */

void PlayNote(short Note)
{
  SndCommandType snd;
  Word err;
  int Hz[24] = {440,466,494,523,554,587,622,659,698,740,784,831,880,
                932,988,1047,1109,1175,1245,1319,1397,1480,1568,1660};

  if (Note>=100)
    Note = Note-100;
  if (Note<0)
    return;
  if (Note>23)
    return;

  snd.cmd = 1;
  snd.param1 = Hz[Note];
  snd.param2 = 250;
  snd.param3 = sndMaxAmp;
  err = SndDoCmd(0,&snd,0);  
}

/* ----------------------------------------------------- */
/* ShowNotes                                             */
/*   Display the notes as determined by the Root popup.  */
/* Parms                                                 */
/*   None                                                */
/* Returns                                               */
/*   Nowt                                                */
/* ----------------------------------------------------- */

void ShowNotes(void)
{
  short string,
        pos;

  for(string=0;string<=5;string++)
  {
    pos=prefs.RootReqd+offset[string];
    while (pos>=12)
    {
      pos-=12;
    }
    while (pos<=numfret)
    {
      DrawNote(string,pos,boldnote);
      pos+=12;
    }
  }
  if(PlaySnd)
  {
    PlayNote(prefs.RootReqd);
    PlayNote(prefs.RootReqd+12);
    PlayNote(prefs.RootReqd);
    PlayNote(prefs.RootReqd+12);
  }
}

/* ----------------------------------------------------- */
/* ShowChords                                            */
/*   Display the chords as determined by the Root, and   */
/*   Chord, popups.                                      */
/* Parms                                                 */
/*   None                                                */
/* Returns                                               */
/*   Nowt                                                */
/* ----------------------------------------------------- */

void ShowChords(void)
{
  short intval[nochords][chordintvals];
  short string,
        intno,
        pos,
        i;

  Boolean shownorm;

  /* major */
  intval[0][0] = 0;
  intval[0][1] = 104;
  intval[0][2] = 7;
  intval[0][3] = -1;
  intval[0][4] = -1;
  intval[0][5] = -1;

  /* major 7th */
  intval[1][0] = 0;
  intval[1][1] = 4;
  intval[1][2] = 7;
  intval[1][3] = 111;
  intval[1][4] = -1;
  intval[1][5] = -1;

  /* major 9th */
  intval[2][0] = 0;
  intval[2][1] = 102;
  intval[2][2] = 4;
  intval[2][3] = 7;
  intval[2][4] = 111;
  intval[2][5] = -1;

  /* 6th */
  intval[3][0] = 0;
  intval[3][1] = 4;
  intval[3][2] = 7;
  intval[3][3] = 109;
  intval[3][4] = -1;
  intval[3][5] = -1;

  /* 7th */
  intval[4][0] = 0;
  intval[4][1] = 4;
  intval[4][2] = 7;
  intval[4][3] = 110;
  intval[4][4] = -1;
  intval[4][5] = -1;

  /* 9th */
  intval[5][0] = 0;
  intval[5][1] = 102;
  intval[5][2] = 4;
  intval[5][3] = 7;
  intval[5][4] = 10;
  intval[5][5] = -1;

  /* 11th */
  intval[6][0] = 0;
  intval[6][1] = 2;
  intval[6][2] = 104;
  intval[6][3] = 5;
  intval[6][4] = 7;
  intval[6][5] = 10;

  /* 13th */
  intval[7][0] = 0;
  intval[7][1] = 2;
  intval[7][2] = 4;
  intval[7][3] = 7;
  intval[7][4] = 109;
  intval[7][5] = 10;

  /* 6/9 */
  intval[8][0] = 0;
  intval[8][1] = 102;
  intval[8][2] = 4;
  intval[8][3] = 7;
  intval[8][4] = 109;
  intval[8][5] = -1;

  /* minor */
  intval[9][0] = 0;
  intval[9][1] = 103;
  intval[9][2] = 7;
  intval[9][3] = -1;
  intval[9][4] = -1;
  intval[9][5] = -1;

  /* minor 6th */
  intval[10][0] = 0;
  intval[10][1] = 3;
  intval[10][2] = 7;
  intval[10][3] = 9;
  intval[10][4] = -1;
  intval[10][5] = -1;

  /* minor 7th */
  intval[11][0] = 0;
  intval[11][1] = 3;
  intval[11][2] = 7;
  intval[11][3] = 10;
  intval[11][4] = -1;
  intval[11][5] = -1;

  /* minor 9th */
  intval[12][0] = 0;
  intval[12][1] = 2;
  intval[12][2] = 3;
  intval[12][3] = 7;
  intval[12][4] = 110;
  intval[12][5] = -1;

  /* sus4 */
  intval[13][0] = 0;
  intval[13][1] = 105;
  intval[13][2] = 7;
  intval[13][3] = -1;
  intval[13][4] = -1;
  intval[13][5] = -1;

  /* 7th sus4 */
  intval[14][0] = 0;
  intval[14][1] = 105;
  intval[14][2] = 7;
  intval[14][3] = 110;
  intval[14][4] = -1;
  intval[14][5] = -1;

  /* dim */
  intval[15][0] = 0;
  intval[15][1] = 103;
  intval[15][2] = 106;
  intval[15][3] = 109;
  intval[15][4] = -1;
  intval[15][5] = -1;

  /* Dim 5th */
  intval[16][0] = 0;
  intval[16][1] = 4;
  intval[16][2] = 106;
  intval[16][3] = -1;
  intval[16][4] = -1;
  intval[16][5] = -1;

  /* 7th Dim 5th */
  intval[17][0] = 0;
  intval[17][1] = 4;
  intval[17][2] = 106;
  intval[17][3] = 110;
  intval[17][4] = -1;
  intval[17][5] = -1;

  /* 7th Dim 9th */
  intval[18][0] = 0;
  intval[18][1] = 101;
  intval[18][2] = 4;
  intval[18][3] = 7;
  intval[18][4] = 110;
  intval[18][5] = -1;

  /* 9th Dim 5th */
  intval[19][0] = 0;
  intval[19][1] = 102;
  intval[19][2] = 4;
  intval[19][3] = 106;
  intval[19][4] = 10;
  intval[19][5] = -1;

  /* aug */
  intval[20][0] = 0;
  intval[20][1] = 4;
  intval[20][2] = 108;
  intval[20][3] = -1;
  intval[20][4] = -1;
  intval[20][5] = -1;

  /* 7th Aug 5th */
  intval[21][0] = 0;
  intval[21][1] = 4;
  intval[21][2] = 108;
  intval[21][3] = 110;
  intval[21][4] = -1;
  intval[21][5] = -1;

  /* 7th Aug 9th */
  intval[22][0] = 0;
  intval[22][1] = 103;
  intval[22][2] = 4;
  intval[22][3] = 7;
  intval[22][4] = 110;
  intval[22][5] = -1;

  /* Stacked Fourths */
  intval[23][0] = 0;
  intval[23][1] = 105;
  intval[23][2] = 110;
  intval[23][3] = -1;
  intval[23][4] = -1;
  intval[23][5] = -1;

  /* Stacked Fifths */
  intval[24][0] = 0;
  intval[24][1] = 107;
  intval[24][2] = 114;
  intval[24][3] = -1;
  intval[24][4] = -1;
  intval[24][5] = -1;

  /* save the selected chord for use in the chord spelling window */
  for(intno=0;intno<chordintvals;intno++)
  {
    if(intval[prefs.ChrdReqd][intno]>=0)
    {
      pos=prefs.RootReqd+intval[prefs.ChrdReqd][intno];
      if (pos>100)
        pos -= 100;
      while (pos>=12)
        pos-=12;
      if (intno==0)
        StrCopy(GChordSpell,LstGetSelectionText(GetObjPtr(RootList),pos));
      else
      {
        StrCat(GChordSpell,", ");
        StrCat(GChordSpell,LstGetSelectionText(GetObjPtr(RootList),pos));
      }
    }
  }
  StrCopy(GChordName,LstGetSelectionText(GetObjPtr(RootList),prefs.RootReqd));
  StrCat(GChordName," ");
  StrCat(GChordName,LstGetSelectionText(GetObjPtr(ChrdList),prefs.ChrdReqd));

  for(string=0;string<numstring;string++)
    for(intno=0;intno<chordintvals;intno++)
    {
      if(intval[prefs.ChrdReqd][intno]>=0)
      {
        pos=prefs.RootReqd+offset[string]+intval[prefs.ChrdReqd][intno];
        if (pos>100)
        {
          pos -= 100;
          shownorm = false;
        }
        else
          shownorm = true;
        while (pos>=12)
        {
          pos-=12;
        }
        while (pos<=numfret)
        {
          if (intno==0)
            DrawNote(string,pos,rootnote);
          else
            if (shownorm)
              DrawNote(string,pos,normnote);
            else
              DrawNote(string,pos,boldnote);
          pos+=12;
        }
      }
    }
  if(PlaySnd)
  {
    for(intno=0;intno<chordintvals;intno++)
      if(intval[prefs.ChrdReqd][intno]>=0)
      {
        pos = prefs.RootReqd+intval[prefs.ChrdReqd][intno];
        PlayNote(prefs.RootReqd+intval[prefs.ChrdReqd][intno]);
      }
    PlayNote(prefs.RootReqd+12);
  }
}

/* ----------------------------------------------------- */
/* ShowScales                                            */
/*   Display the notes as determined by the Root, and    */
/*   mode popups.                                        */
/* Parms                                                 */
/*   None                                                */
/* Returns                                               */
/*   Nowt                                                */
/* ----------------------------------------------------- */

void ShowScales(void)
{
  short intval[noscales][scaleintvals];
  short string,
        intno,
        pos;

  /* major */
  intval[0][0] = 0;
  intval[0][1] = 2;
  intval[0][2] = 4;
  intval[0][3] = 5;
  intval[0][4] = 7;
  intval[0][5] = 9;
  intval[0][6] = 11;
  intval[0][7] = -1;

  /* major pentatonic */
  intval[1][0] = 0;
  intval[1][1] = 2;
  intval[1][2] = 4;
  intval[1][3] = 7;
  intval[1][4] = 9;
  intval[1][5] = -1;
  intval[1][6] = -1;
  intval[1][7] = -1;

  /* natural minor */
  intval[2][0] = 0;
  intval[2][1] = 2;
  intval[2][2] = 3;
  intval[2][3] = 5;
  intval[2][4] = 7;
  intval[2][5] = 8;
  intval[2][6] = 10;
  intval[2][7] = -1; 

  /* minor pentatonic */
  intval[3][0] = 0;
  intval[3][1] = 3;
  intval[3][2] = 5;
  intval[3][3] = 7;
  intval[3][4] = 10;
  intval[3][5] = -1;
  intval[3][6] = -1;
  intval[3][7] = -1;

  /* harmonic minor */
  intval[4][0] = 0;
  intval[4][1] = 2;
  intval[4][2] = 3;
  intval[4][3] = 5;
  intval[4][4] = 7;
  intval[4][5] = 8;
  intval[4][6] = 11;
  intval[4][7] = -1;

  /* melodic minor */
  intval[5][0] = 0;
  intval[5][1] = 2;
  intval[5][2] = 3;
  intval[5][3] = 5;
  intval[5][4] = 7;
  intval[5][5] = 9;
  intval[5][6] = 11;
  intval[5][7] = -1;

  /* blues */
  intval[6][0] = 0;
  intval[6][1] = 3;
  intval[6][2] = 5;
  intval[6][3] = 6;
  intval[6][4] = 7;
  intval[6][5] = 10;
  intval[6][6] = -1;
  intval[6][7] = -1;

  /* major blues */
  intval[7][0] = 0;
  intval[7][1] = 2;
  intval[7][2] = 3;
  intval[7][3] = 4;
  intval[7][4] = 7;
  intval[7][5] = 9;
  intval[7][6] = -1;
  intval[7][7] = -1;

  /* diminished */
  intval[8][0] = 0;
  intval[8][1] = 2;
  intval[8][2] = 3;
  intval[8][3] = 5;
  intval[8][4] = 6;
  intval[8][5] = 8;
  intval[8][6] = 9;
  intval[8][7] = 11;

  /* whole tone */
  intval[9][0] = 0;
  intval[9][1] = 2;
  intval[9][2] = 4;
  intval[9][3] = 6;
  intval[9][4] = 8;
  intval[9][5] = 10;
  intval[9][6] = -1;
  intval[9][7] = -1;

  /* dorian */
  intval[10][0] = 0;
  intval[10][1] = 2;
  intval[10][2] = 3;
  intval[10][3] = 5;
  intval[10][4] = 7;
  intval[10][5] = 9;
  intval[10][6] = 10;
  intval[10][7] = -1;

  /* phrygian */
  intval[11][0] = 0;
  intval[11][1] = 1;
  intval[11][2] = 3;
  intval[11][3] = 5;
  intval[11][4] = 7;
  intval[11][5] = 8;
  intval[11][6] = 10;
  intval[11][7] = -1;

  /* lydian */
  intval[12][0] = 0;
  intval[12][1] = 2;
  intval[12][2] = 4;
  intval[12][3] = 6;
  intval[12][4] = 7;
  intval[12][5] = 9;
  intval[12][6] = 11;
  intval[12][7] = -1;

  /* mixolydian */
  intval[13][0] = 0;
  intval[13][1] = 2;
  intval[13][2] = 4;
  intval[13][3] = 5;
  intval[13][4] = 7;
  intval[13][5] = 9;
  intval[13][6] = 10;
  intval[13][7] = -1;

  /* aolian */
  intval[14][0] = 0;
  intval[14][1] = 2;
  intval[14][2] = 3;
  intval[14][3] = 5;
  intval[14][4] = 7;
  intval[14][5] = 8;
  intval[14][6] = 10;
  intval[14][7] = -1;

  /* locrian */
  intval[15][0] = 0;
  intval[15][1] = 1;
  intval[15][2] = 3;
  intval[15][3] = 5;
  intval[15][4] = 6;
  intval[15][5] = 8;
  intval[15][6] = 10;
  intval[15][7] = -1;

  for(string=0;string<numstring;string++)
    for(intno=0;intno<scaleintvals;intno++)
    {
      if(intval[prefs.ModeReqd][intno]>=0)
      {
        pos=prefs.RootReqd+offset[string]+intval[prefs.ModeReqd][intno];
        while (pos>=12)
        {
          pos-=12;
        }
        while (pos<=numfret)
        {
          if (intno==0)
            DrawNote(string,pos,rootnote);
          else
            DrawNote(string,pos,boldnote);
          pos+=12;
        }
      }
    }
  if(PlaySnd)
  {
    for(intno=0;intno<scaleintvals;intno++)
      if(intval[prefs.ModeReqd][intno]>=0)
        PlayNote(prefs.RootReqd+intval[prefs.ModeReqd][intno]);
    PlayNote(prefs.RootReqd+12);
  }
}

/* ----------------------------------------------------- */
/* LabelFret                                             */
/*   This draws a fret label next to the neck.           */
/* Parms                                                 */
/*   FretNo - The number of the fret (0=nut)             */
/* Returns                                               */
/*   Nowt                                                */
/* ----------------------------------------------------- */

void LabelFret(short FretNo)
{
  char FretT[3];
  short xpos,
        ypos,
        charW;

  StrIToA(FretT,FretNo);
  charW = FntCharsWidth(FretT,StrLen(FretT));
  if (prefs.NutAtRight)
    xpos = (((numfret-FretNo)*fretspace)+neckorigx)-(charW/2);
  else
    xpos = ((FretNo*fretspace)+neckorigx)-(charW/2);
  ypos = neckorigy-15;
  WinDrawChars(FretT, StrLen(FretT),xpos,ypos);
}

/* ----------------------------------------------------- */
/* DisplayNeck                                           */
/*   This function paints the neck.                      */
/* Parms                                                 */
/*   None                                                */
/* Returns                                               */
/*   Nowt                                                */
/* ----------------------------------------------------- */

void DisplayNeck(void)
{
  RectangleType rect;
  short labos,
        StrNo,
        i,
        xpos,
        ypos;
  Char StrLabel[6];
  Word NotePos;

  /* Clear the neck diagram */
  rect.topLeft.x=0;
  rect.topLeft.y=15;
  rect.extent.x=160;
  rect.extent.y=69;
  WinEraseRectangle(&rect,0);

  /* Draw the strings */
  for(i=neckorigy;i<(neckorigy+(numstring*stringspace));i+=stringspace)
  {
    WinDrawLine(neckorigx,i,neckorigx+((numfret)*fretspace),i);
  }

  /* label the strings */
  StrNo = 0;
  labos = 0;
  for(i=0;i<numstring;i++)
  {
    if (offset[StrNo]==0)
      NotePos = 0;
    else
      NotePos = 12 - offset[StrNo];
    StrCopy(StrLabel,LstGetSelectionText(GetObjPtr(RootList),NotePos));
    if (labos==labelos1)
      labos = labelos2;
    else
      labos = labelos1;

   if (prefs.NutAtRight)
      xpos = labos+neckorigx+nutos+((numfret)*fretspace);
   else
      xpos = neckorigx-labos-FntCharsWidth("E",StrLen("E"))-3;

    if (prefs.ThickStringAtTop)
      ypos = (stringspace*i)+neckorigy;
    else
      ypos = (((numstring-1)-i)*stringspace)+neckorigy;
    ypos-=FntCharHeight()/2;
    WinDrawChars(StrLabel,StrLen(StrLabel),xpos,ypos);
    StrNo++;
  }

  /* Draw the frets */
  for(i=neckorigx;i<(neckorigx+((numfret+1)*fretspace));i+=fretspace)
    WinDrawLine(i,neckorigy,i,neckorigy+((numstring-1)*stringspace));

  /* Draw the nut */
  if (prefs.NutAtRight)
    WinDrawLine((neckorigx+(numfret*fretspace))+nutos,neckorigy,(neckorigx+(numfret*fretspace))+nutos,neckorigy+((numstring-1)*stringspace));
  else
    WinDrawLine(neckorigx-nutos,neckorigy,neckorigx-nutos,neckorigy+((numstring-1)*stringspace));

  /* Draw markers at frets 0, 3, 5, 7, 10, 12 */
  FntSetFont (stdFont);
  LabelFret(0);
  LabelFret(3);
  LabelFret(5);
  LabelFret(7);
  LabelFret(10);
  LabelFret(12);
}

/* ----------------------------------------------------- */
/* ChangeLabel                                           */
/*   Amend the text of a label.                          */
/* Parms                                                 */
/*   LabID - The ref no of the label                     */
/*   LabTx - The text to display                         */
/* Returns                                               */
/*   Nowt                                                */
/* ----------------------------------------------------- */

void ChangeLabel(Word LabID, CharPtr LabTx)
{
  FormPtr frm;
  Word obj;
  
  frm = FrmGetActiveForm();
  obj = FrmGetObjectIndex(frm,LabID);
  FrmHideObject(frm,obj);
  FrmCopyLabel(frm,LabID,LabTx);
  FrmShowObject(frm,obj);
}

/* ----------------------------------------------------- */
/* SetInstrument                                         */
/*   When This function loads a form and set the event   */
/*   handler for the form.                               */
/* Parms                                                 */
/*   event - Pointer to an event type structure          */
/* Returns                                               */
/*   True - if the event has been dealt with             */
/* ----------------------------------------------------- */

static void SetInstrument(void)
{
  offset[0] = -1;
  offset[1] = -1;
  offset[2] = -1;
  offset[3] = -1;
  offset[4] = -1;
  offset[5] = -1;
  offset[6] = -1;
  offset[7] = -1;
  offset[8] = -1;
  offset[9] = -1;
  neckorigy = 35;
  stringspace = 6;
  switch (prefs.InstReqd)
  {
    case 0: /* Guitar - Standard */
      numstring = 6;
      offset[0] = 5;
      offset[1] = 0;
      offset[2] = 7;
      offset[3] = 2;
      offset[4] = 10;
      offset[5] = 5;
      break;
    case 1: /* Guitar - Open C */
      numstring = 6;
      offset[0] = 9;
      offset[1] = 2;
      offset[2] = 9;
      offset[3] = 2;
      offset[4] = 9;
      offset[5] = 5;
      break;
    case 2: /* Guitar - Open D */
      numstring = 6;
      offset[0] = 7;
      offset[1] = 0;
      offset[2] = 7;
      offset[3] = 3;
      offset[4] = 0;
      offset[5] = 7;
      break;
    case 3: /* Guitar - Open E */
      numstring = 6;
      offset[0] = 5;
      offset[1] = 10;
      offset[2] = 5;
      offset[3] = 1;
      offset[4] = 10;
      offset[5] = 5;
      break;
    case 4: /* Guitar - Open G */
      numstring = 6;
      offset[0] = 7;
      offset[1] = 2;
      offset[2] = 7;
      offset[3] = 2;
      offset[4] = 10;
      offset[5] = 7;
      break;
    case 5: /* Guitar - Crossnote D */
      numstring = 6;
      offset[0] = 7;
      offset[1] = 0;
      offset[2] = 7;
      offset[3] = 4;
      offset[4] = 0;
      offset[5] = 7;
      break;
    case 6: /* Guitar - Crossnote E */
      numstring = 6;
      offset[0] = 5;
      offset[1] = 10;
      offset[2] = 5;
      offset[3] = 2;
      offset[4] = 10;
      offset[5] = 5;
      break;
    case 7: /* Guitar - Modal D */
      numstring = 6;
      offset[0] = 7;
      offset[1] = 0;
      offset[2] = 7;
      offset[3] = 2;
      offset[4] = 10;
      offset[5] = 5;
      break;
    case 8: /* Guitar - Modal G */
      numstring = 6;
      offset[0] = 7;
      offset[1] = 2;
      offset[2] = 7;
      offset[3] = 2;
      offset[4] = 9;
      offset[5] = 7;
      break;
    case 9: /* Guitar - DADGAD */
      numstring = 6;
      offset[0] = 7;
      offset[1] = 0;
      offset[2] = 7;
      offset[3] = 2;
      offset[4] = 0;
      offset[5] = 7;
      break;
    case 10: /* Banjo - G */
      numstring = 4;
      offset[0] = 7;
      offset[1] = 2;
      offset[2] = 10;
      offset[3] = 7;
      break;
    case 11: /* Banjo - C */
      numstring = 4;
      offset[0] = 9;
      offset[1] = 2;
      offset[2] = 10;
      offset[3] = 7;
      break;
    case 12: /* Banjo - D */
      numstring = 4;
      offset[0] = 7;
      offset[1] = 0;
      offset[2] = 3;
      offset[3] = 7;
      break;
    case 13: /* Banjo - Gm */
      numstring = 4;
      offset[0] = 7;
      offset[1] = 2;
      offset[2] = 1;
      offset[3] = 7;
      break;
    case 14: /* 4 String Bass */
      numstring = 4;
      offset[0] = 5;
      offset[1] = 0;
      offset[2] = 7;
      offset[3] = 2;
      break;
    case 15: /* 5 String Bass */
      numstring = 5;
      offset[0] = 10;
      offset[1] = 5;
      offset[2] = 0;
      offset[3] = 7;
      offset[4] = 2;
      break;
    case 16: /* 6 String Bass */
      numstring = 6;
      offset[0] = 10;
      offset[1] = 5;
      offset[2] = 0;
      offset[3] = 7;
      offset[4] = 2;
      offset[5] = 9;
      break;
    case 17: /* Bouzouki - GDAD */
      numstring = 4;
      offset[0] = 2;
      offset[1] = 7;
      offset[2] = 0;
      offset[3] = 7;
      break;
    case 18: /* Bouzouki - ADAD */
      numstring = 4;
      offset[0] = 0;
      offset[1] = 7;
      offset[2] = 0;
      offset[3] = 7;
      break;
    case 19: /* Bouzouki - CFAD */
      numstring = 4;
      offset[0] = 9;
      offset[1] = 4;
      offset[2] = 0;
      offset[3] = 7;
      break;
    case 20: /* Bouzouki - CGDAD */
      numstring = 5;
      offset[0] = 9;
      offset[1] = 2;
      offset[2] = 7;
      offset[3] = 0;
      offset[4] = 7;
      break;
    case 21: /* Bouzouki - DGDAD */
      numstring = 5;
      offset[0] = 7;
      offset[1] = 2;
      offset[2] = 7;
      offset[3] = 0;
      offset[4] = 7;
      break;
    case 22: /* Bouzouki - GDAEA */
      numstring = 5;
      offset[0] = 2;
      offset[1] = 7;
      offset[2] = 0;
      offset[3] = 5;
      offset[4] = 0;
      break;
    case 23: /* Bouzouki - CGDAE */
      numstring = 5;
      offset[0] = 9;
      offset[1] = 2;
      offset[2] = 7;
      offset[3] = 0;
      offset[4] = 5;
      break;
    case 24: /* Bouzouki - DGDAE */
      numstring = 5;
      offset[0] = 7;
      offset[1] = 2;
      offset[2] = 7;
      offset[3] = 0;
      offset[4] = 5;
      break;
    case 25: /* Bouzouki - GDGDG */
      numstring = 5;
      offset[0] = 2;
      offset[1] = 7;
      offset[2] = 2;
      offset[3] = 7;
      offset[4] = 2;
      break;
    case 26: /* Bouzouki - GCGCG */
      numstring = 5;
      offset[0] = 2;
      offset[1] = 9;
      offset[2] = 2;
      offset[3] = 9;
      offset[4] = 2;
      break;
    case 27: /* Bouzouki - ADADA */
      numstring = 5;
      offset[0] = 0;
      offset[1] = 7;
      offset[2] = 0;
      offset[3] = 7;
      offset[4] = 0;
      break;
    case 28: /* Bouzouki - GCGDG */
      numstring = 5;
      offset[0] = 2;
      offset[1] = 9;
      offset[2] = 2;
      offset[3] = 7;
      offset[4] = 2;
      break;
    case 29: /* Dulcimer */
      numstring = 4;
      offset[0] = 7;
      offset[1] = 0;
      offset[2] = 7;
      offset[3] = 7;
      break;
    case 30: /* Mandolin */
      numstring = 4;
      offset[0] = 2;
      offset[1] = 7;
      offset[2] = 0;
      offset[3] = 5;
      break;
    case 31: /* Violin */
      numstring = 4;
      offset[0] = 2;
      offset[1] = 7;
      offset[2] = 0;
      offset[3] = 5;
      break;
    case 32: /* Viola/Cello */
      numstring = 4;
      offset[0] = 9;
      offset[1] = 2;
      offset[2] = 7;
      offset[3] = 0;
      break;
    case 33: /* Chapman Stick */
      numstring = 10;
      offset[0] = 7;
      offset[1] = 0;
      offset[2] = 5;
      offset[3] = 10;
      offset[4] = 3;
      offset[5] = 9;
      offset[6] = 2;
      offset[7] = 7;
      offset[8] = 0;
      offset[9] = 5;
      neckorigy = 30;
      stringspace = 5;
      break;
    case 34: /* Lute */
      numstring = 6;
      offset[0] = 2;
      offset[1] = 9;
      offset[2] = 4;
      offset[3] = 0;
      offset[4] = 7;
      offset[5] = 2;
      break;
    case 35: /* Ukulele */
      numstring = 4;
      offset[0] = 2;
      offset[1] = 9;
      offset[2] = 5;
      offset[3] = 0;
      break;
    case 36: /* Renaissance Guitar */
      numstring = 4;
      offset[0] = 2;
      offset[1] = 9;
      offset[2] = 5;
      offset[3] = 0;
      break;
  }
}

/* ----------------------------------------------------- */
/* RedrawScreen                                          */
/*   This gets the information from the popup lists and  */
/*   uses this data to generate the correct pattern of   */
/*   notes on the neck.                                  */
/* Parms                                                 */
/*   None                                                */
/* Returns                                               */
/*   Nowt                                                */
/* ----------------------------------------------------- */

void RedrawScreen(void)
{
  switch(prefs.DispReqd)
  {
    case 0:
      CtlHideControl(GetObjPtr(ModeListT));
      CtlHideControl(GetObjPtr(ChrdListT));
      ChangeLabel(NoteLab,"Note:");
      ChangeLabel(TypeLab," ");
      break;
    case 1:
      CtlHideControl(GetObjPtr(ModeListT));
      CtlShowControl(GetObjPtr(ChrdListT));
      ChangeLabel(NoteLab,"Root:");
      ChangeLabel(TypeLab,"Chord:");
      break;
    case 2:
      CtlHideControl(GetObjPtr(ChrdListT));
      CtlShowControl(GetObjPtr(ModeListT));
      ChangeLabel(NoteLab,"Root:");
      ChangeLabel(TypeLab,"Scale:");
      break;
  }
  SetInstrument();
  DisplayNeck();
  switch(prefs.DispReqd)
  {
    case 0:
      ShowNotes();
      break;
    case 1:
      ShowChords();
      break;
    case 2:
      ShowScales();
  }
  PlaySnd = false;
}

/* ----------------------------------------------------- */
/* AboutViewHandleEvent                                  */
/*   Handle something happening on the about form        */
/* Parms                                                 */
/*   event - Pointer to an event type structure          */
/* Returns                                               */
/*   True - if the event has been dealt with             */
/* ----------------------------------------------------- */

static Boolean AboutViewHandleEvent(EventPtr event)
{
  Boolean handled = false;
  FormType frm;

  switch (event->eType)
  {
    case frmOpenEvent:
      FrmDrawForm(FrmGetActiveForm());
      handled = true;
      break;
    case ctlEnterEvent:
      if (event->data.popSelect.controlID==AboutDone)
      {
        FrmGotoForm(MainForm);
        handled = true;
        break;
      }
  }
  return(handled);
}

/* ----------------------------------------------------- */
/* MainViewHandleEvent                                   */
/*   This function loads a form and set the event        */
/*   handler for the form.                               */
/* Parms                                                 */
/*   event - Pointer to an event type structure          */
/* Returns                                               */
/*   True - if the event has been dealt with             */
/* ----------------------------------------------------- */

static Boolean MainViewHandleEvent(EventPtr event)
{
  Boolean handled = false;
  FormType frm;
  RectangleType rect;

  switch (event->eType)
  {
    case frmOpenEvent:
      FrmDrawForm(FrmGetActiveForm());
      CtlSetLabel(GetObjPtr(InstListT),LstGetSelectionText(GetObjPtr(InstList),prefs.InstReqd));
      CtlSetLabel(GetObjPtr(RootListT),LstGetSelectionText(GetObjPtr(RootList),prefs.RootReqd));
      CtlSetLabel(GetObjPtr(ChrdListT),LstGetSelectionText(GetObjPtr(ChrdList),prefs.ChrdReqd));
      CtlSetLabel(GetObjPtr(ModeListT),LstGetSelectionText(GetObjPtr(ModeList),prefs.ModeReqd));
      switch(prefs.DispReqd)
      {
        case 0:
          CtlSetValue(GetObjPtr(NoteBut),1);
          break;
        case 1:
          CtlSetValue(GetObjPtr(ChordBut),1);
          break;
        case 2:
          CtlSetValue(GetObjPtr(ScaleBut),1);
          break;
      }
      RedrawScreen();
      handled = true;
      break;
    case penDownEvent:
      if (prefs.DispReqd==1)
      {
        if (event->screenY<=(neckorigy+(stringspace*numstring)))
        {
          FntSetFont (largeFont);
          rect.topLeft.x = event->screenX - FntCharsWidth(GChordSpell,StrLen(GChordSpell)) - 4;
          rect.topLeft.y = event->screenY - FntCharHeight() - 4;
          if (rect.topLeft.x < 3)
            rect.topLeft.x = 3;
          if (rect.topLeft.y < 18)
            rect.topLeft.y = 18;
          rect.extent.x=FntCharsWidth(GChordSpell,StrLen(GChordSpell)) + 4;
          rect.extent.y=FntCharHeight() + 4;
          WinDrawRectangle(&rect,0);

          rect.topLeft.x-=3;
          rect.topLeft.y-=3;
          WinDrawRectangle(&rect,0);

          rect.topLeft.x+=1;
          rect.topLeft.y+=1;
          rect.extent.x-=2;
          rect.extent.y-=2;
          WinEraseRectangle(&rect,0);
          WinDrawChars(GChordSpell,StrLen(GChordSpell),rect.topLeft.x+1,rect.topLeft.y+1);
          FntSetFont (stdFont);
        }
      }
      handled = false;
      break;
    case penUpEvent:
      RedrawScreen();
      handled = false;
      break;
    case ctlSelectEvent:
      PlaySnd = false;
      switch(event->data.popSelect.controlID)
      {
        case NoteBut:
          prefs.DispReqd=0;
          break;
        case ChordBut:
          prefs.DispReqd=1;
          break;
        case ScaleBut:
          prefs.DispReqd=2;
          break;
        case PlayBut:
          PlaySnd = true;
          break;
      }
      RedrawScreen();
      break;
    case popSelectEvent:
      switch(event->data.popSelect.listID)
      {
        case RootList:
          prefs.RootReqd = event->data.popSelect.selection;
          CtlSetLabel(GetObjPtr(RootListT),LstGetSelectionText(GetObjPtr(RootList),LstGetSelection(GetObjPtr(RootList))));
          break;
        case ChrdList:
          prefs.ChrdReqd = event->data.popSelect.selection;
          CtlSetLabel(GetObjPtr(ChrdListT),LstGetSelectionText(GetObjPtr(ChrdList),LstGetSelection(GetObjPtr(ChrdList))));
          break;
        case ModeList:
          prefs.ModeReqd = event->data.popSelect.selection;
          CtlSetLabel(GetObjPtr(ModeListT),LstGetSelectionText(GetObjPtr(ModeList),LstGetSelection(GetObjPtr(ModeList))));
          break;
        case InstList:
          prefs.InstReqd = event->data.popSelect.selection;
          CtlSetLabel(GetObjPtr(InstListT),LstGetSelectionText(GetObjPtr(InstList),LstGetSelection(GetObjPtr(InstList))));
      }
      RedrawScreen();
      handled = true;
      break;
    case menuEvent:
      MenuEraseStatus(0);
      switch (event->data.menu.itemID)
      {
        case menuHelp:
          FrmHelp(HelpText);
          handled = true;
          break;
        case menuAbout:
          FrmPopupForm(FormAbout);
          handled = true;
          break;
        case menuSwapNut:
          if (prefs.NutAtRight)
          {
            prefs.NutAtRight = false;
            neckorigx = 27;
          }
          else
          {
            prefs.NutAtRight = true;
            neckorigx = 10;
          }
          handled=true;
          break;
        case menuSwapStr:
          if (prefs.ThickStringAtTop)
            prefs.ThickStringAtTop = false;
          else
            prefs.ThickStringAtTop = true;
          handled = true;
          break;
      }
      RedrawScreen();
      handled = true;
      break;
  }
  return(handled);
}

/* ----------------------------------------------------- */
/* ApplicationHandleEvent                                */
/*   This function loads a form and set the event        */
/*   handler for the form.                               */
/* Parms                                                 */
/*   event - Pointer to an event type structure          */
/* Returns                                               */
/*   True - if the event has been dealt with             */
/* ----------------------------------------------------- */

static Boolean ApplicationHandleEvent(EventPtr event)
{
  FormPtr frm;
  Int formId;
  Boolean handled = false;

  if (event->eType == frmLoadEvent)
  {
    formId = event->data.frmLoad.formID;
    frm = FrmInitForm(formId);
    FrmSetActiveForm(frm);

    switch (formId)
    {
      case MainForm:
        FrmSetEventHandler(frm, MainViewHandleEvent);
        break;
      case FormAbout:
        FrmSetEventHandler(frm, AboutViewHandleEvent);
        break;
    }
    handled = true;
  }
  return handled;
}

/* ----------------------------------------------------- */
/* InitApplication                                       */
/*   Get everything ready.                               */
/* Parms                                                 */
/*   None                                                */
/* Returns                                               */
/*   Nowt                                                */
/* ----------------------------------------------------- */

static void InitApplication(void)
{
  FrmGotoForm(MainForm);
  if (PrefGetAppPreferencesV10('DM04', 1, &prefs, sizeof(FretBoardPrefType)) == NULL)
  {
    prefs.NutAtRight = true;
    prefs.ThickStringAtTop = true;
    prefs.DispReqd = 0;
    prefs.RootReqd = 0;
    prefs.ChrdReqd = 0;
    prefs.ModeReqd = 0;
    prefs.InstReqd = 0;
  }
  if (prefs.NutAtRight)
    neckorigx = 10;
  else
    neckorigx = 27;
  /* FrmPopupForm(FormAbout); */

}

/* ----------------------------------------------------- */
/* StopApplication                                       */
/*   Like my Mum used to say "Tidy up after yourself".   */
/* Parms                                                 */
/*   None                                                */
/* Returns                                               */
/*   Nowt                                                */
/* ----------------------------------------------------- */

static void StopApplication(void)
{
  PrefSetAppPreferencesV10('DM04', 1, &prefs, sizeof(FretBoardPrefType));
  FrmCloseAllForms ();
}

/* ----------------------------------------------------- */
/* EventLoop                                             */
/*   This function gets events from the Event Manager,   */
/*   passes them to the system, menu and application     */
/*   event handlers before passing them onto our own     */
/*   processing.                                         */
/* Parms                                                 */
/*   None                                                */
/* Returns                                               */
/*   Nowt                                                */
/* ----------------------------------------------------- */

static void EventLoop(void)
{
  EventType event;
  Word error;
  do
  {
    EvtGetEvent(&event, evtWaitForever);
    if (! SysHandleEvent(&event))
      if (! MenuHandleEvent(NULL, &event, &error))
        if (!ApplicationHandleEvent(&event))
          FrmDispatchEvent(&event);
  }
  while (event.eType != appStopEvent);
}

/* ----------------------------------------------------- */
/* PilotMain                                             */
/*   Equivalent of the normal C Main(), it starts the    */
/*   execution of the application.                       */
/* Parms                                                 */
/*   cmd - command specifying how to launch the app      */
/*   cmdPBP - parameter block for the command            */
/*   launchFlags - flags used to config the launch       */
/* Returns                                               */
/*   An applicable error code                            */
/* ----------------------------------------------------- */

DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
  Int err;

  if (cmd == sysAppLaunchCmdNormalLaunch)
  {
    InitApplication();
    EventLoop();
    StopApplication();
  }
  return 0;
}
