/*
 * init.c
 * 
 * This is the install type init 
 *
 * Erik Troan (ewt@redhat.com)
 *
 * Copyright 1996 Red Hat Software 
 *
 * This software may be freely redistributed under the terms of the GNU
 * public license.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef USE_MINILIBC
#define myisspace(a) (a == ' ' || a == '\t')
void reboot();
#define reboot unused_reboot
#else
#define myisspace isspace
#endif

#include <ctype.h>
#include <fcntl.h>
#include <net/if.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/klog.h>
#include <sys/mount.h>
#include <sys/swap.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/un.h>
#include <sys/wait.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/reboot.h>
#include <linux/loop.h>
#include <termios.h>
#include <syslog.h>

#ifdef USE_MINILIBC
#undef reboot
#endif

#define exit error_dont_use_exit__use_return_instead

#define KICK_FLOPPY     1
#define KICK_BOOTP	2

char * env[] = {
    "PATH=/usr/bin:/bin:/sbin:/usr/sbin:/mnt/sbin:/mnt/usr/sbin:"
		   "/mnt/bin:/mnt/usr/bin",
    "LD_LIBRARY_PATH=/lib:/usr/lib:/mnt/lib:/mnt/usr/lib",
    "HOME=/",
    "TERM=linux",
    NULL
};

struct filesystem {
  char * dev;
  char * name;
  char * fs;
  int mounted;
};

/* 
 * this needs to handle the following cases:
 *
 *	1) run from a CD root filesystem
 *	2) run from a read only nfs rooted filesystem
 *      3) run from a floppy
 *	4) run from a floppy that's been loaded into a ramdisk 
 *
 */

int testing;
int klog_pid;

void fatal_error(int usePerror) {
/* FIXME */
#if 0
    if (usePerror) 
	perror("failed:");
    else
#endif
	printf("failed.\n");

    printf("\nI can't recover from this.\n");
    while (1) ;
}

void doklog(char * fn) {
    fd_set readset, unixs;
    int in, out, i;
    int log;
    int s;
    int sock = -1;
    struct sockaddr_un sockaddr;
    char buf[1024];
    int readfd;

    in = open("/proc/kmsg", O_RDONLY,0);
    if (in < 0) {
	/* FIXME: was perror */
	printf("open /proc/kmsg");
	return;
    }

    out = open(fn, O_WRONLY, 0);
    if (out < 0) 
	printf("couldn't open %s for syslog -- still using /tmp/syslog\n", fn);

    log = open("/tmp/syslog", O_WRONLY | O_CREAT, 0644);
    if (log < 0) {
	/* FIXME: was perror */
	printf("error opening /tmp/syslog");
	sleep(5);
	
	close(in);
	return;
    }

    /* if we get this far, we should be in good shape */

    if ((klog_pid = fork())) {
	/* parent */
	close(in);
	close(out);
	close(log);
	return;
    }
    close(0); 
    close(1);
    close(2);

    dup2(log, 1);

#if defined(USE_LOGDEV)
    /* now open the syslog socket */
    sockaddr.sun_family = AF_UNIX;
    strcpy(sockaddr.sun_path, "/dev/log");
    sock = socket(AF_UNIX, SOCK_STREAM, 0);
    if (sock < 0) {
        printf("error creating socket\n");
	sleep(5);
    }
    printf("got socket\n");
    if (bind(sock, (struct sockaddr *) &sockaddr, sizeof(sockaddr.sun_family) + 
			strlen(sockaddr.sun_path))) {
        printf("bind error\n");
	sleep(5);
    }
    printf("bound socket\n");
    chmod("/dev/log", 0666);
    if (listen(sock, 5)) {
	printf("listen error\n");
	sleep(5);
    }
#endif

    syslog(8, NULL, 1);

    FD_ZERO(&unixs);
    while (1) {
	memcpy(&readset, &unixs, sizeof(unixs));

	if (sock >= 0) FD_SET(sock, &readset);
	FD_SET(in, &readset);

	i = select(20, &readset, NULL, NULL, NULL);
	if (i <= 0) continue;

	if (FD_ISSET(in, &readset)) {
	    i = read(in, buf, sizeof(buf));
	    if (i > 0) {
		if (out >= 0) write(out, buf, i);
		write(log, buf, i);
	    }
	} 

	for (readfd = 0; readfd < 20; ++readfd) {
	    if (FD_ISSET(readfd, &readset) && FD_ISSET(readfd, &unixs)) {
		i = read(readfd, buf, sizeof(buf));
		if (i > 0) {
		    if (out >= 0) {
			write(out, buf, i);
			write(out, "\n", 1);
		    }

		    write(log, buf, i);
		    write(log, "\n", 1);
		} else if (i == 0) {
		    /* socket closed */
		    close(readfd);
		    FD_CLR(readfd, &unixs);
		}
	    }
	}

	if (sock >= 0 && FD_ISSET(sock, &readset)) {
	    s = sizeof(sockaddr);
	    readfd = accept(sock, (struct sockaddr *) &sockaddr, &s);
	    if (readfd < 0) {
		if (out >= 0) write(out, "error in accept\n", 16);
		write(log, "error in accept\n", 16);
		close(sock);
		sock = -1;
	    } else {
		FD_SET(readfd, &unixs);
	    }
	}
    }    
}

int setupTerminal(int fd) {
    struct winsize winsize;

    if (ioctl(fd, TIOCGWINSZ, &winsize)) {
	printf("failed to get winsize");
	fatal_error(1);
    }

    winsize.ws_row = 24;
    winsize.ws_col = 80;

    if (ioctl(fd, TIOCSWINSZ, &winsize)) {
	printf("failed to set winsize");
	fatal_error(1);
    }

    env[3] = "TERM=vt100";

    return 0;
}

void del_loop(char *device) {
    int fd;
    if ((fd = open(device, O_RDONLY)) < 0) {
      printf("del_loop open failed\n");
      return;
    }
    if (ioctl(fd, LOOP_CLR_FD, 0) < 0) {
      printf("del_loop ioctl failed");
      return;
    }
    close(fd);
}


void unmountFilesystems(void) {
    int fd, size;
    char buf[65535];			/* this should be big enough */
    char *p;
    struct filesystem fs[500];
    int numfs = 0;
    int i, nb;

    fd = open("/proc/mounts", O_RDONLY, 0);
    if (fd < 1) {
	/* FIXME: was perror */
	printf("failed to open /proc/mounts");
	sleep(2);
	return;
    }

    size = read(fd, buf, sizeof(buf) - 1);
    buf[size] = '\0';

    close(fd);

    p = buf;
    while (*p) {
        fs[numfs].mounted = 1;
	fs[numfs].dev = p;
	while (*p != ' ') p++;
	*p++ = '\0';
	fs[numfs].name = p;
	while (*p != ' ') p++;
	*p++ = '\0';
	fs[numfs].fs = p;
	while (*p != ' ') p++;
	*p++ = '\0';
	while (*p != '\n') p++;
	p++;
	if (strcmp(fs[numfs].name, "/") != 0) numfs++; /* skip if root, no need to take initrd root in account */
    }

    do {
      nb = 0;
      for (i = 0; i < numfs; i++) {
	/*printf("trying with %s\n", fs[i].name);*/
	if (fs[i].mounted && umount(fs[i].name) == 0) { 
	  if (strncmp(fs[i].dev + sizeof("/dev/") - 1, "loop", sizeof("loop") - 1) == 0)
	    del_loop(fs[i].dev);

	  printf("\t%s\n", fs[i].name);
	  fs[i].mounted = 0;
	  nb++;
	}
      }
    } while (nb);

    for (i = nb = 0; i < numfs; i++)
      if (fs[i].mounted) {
	printf("\t%s umount failed\n", fs[i].name);
	if (strcmp(fs[i].fs, "ext2") == 0) nb++; /* don't count not-ext2 umount failed */
      }
    if (nb) {
      printf("failed to umount some filesystems\n");
      while (1);
    }
}

void unmountSwaps(void) {
    int fd, size;
    char buf[65535];			/* this should be big enough */
    char *p, *file;

    fd = open("/proc/swaps", O_RDONLY, 0);
    if (fd < 1) {
	printf("failed to open /proc/swaps");
	sleep(2);
	return;
    }
    size = read(fd, buf, sizeof(buf) - 1);
    buf[size] = '\0';
    close(fd);

    p = buf;

    /* skip first line */
    while (*p != '\n') p++;
    p++;

    while (*p) {
        file = p;
	while (*p != ' ' && *p != '\t') p++;
	*p++ = '\0';

	if (swapoff(file) != 0) printf("failed removing swap %s\n", file);

	while (*p != '\n') p++;
	p++;
    }
}

void readargs(int * isRescue, int * isSerial, int * isExpert, int * isKick,
	      int * isNfs, int * isFtp, int * isCdrom, int * isHd,
	      int * isPcmcia) {
    char buf[512];
    char * arg;
    int fd;
    int size;
    int done = 0;
    char * start, * end;

    printf("opening /proc/cmdline... ");

    if ((fd = open("/proc/cmdline", O_RDONLY, 0)) < 0) fatal_error(1);

    size = read(fd, buf, sizeof(buf) - 1);
    buf[size] = '\0';
    close(fd);

    printf("done\n");

    printf("checking command line arguments... ");
    start = buf;

    while (!done) {
	while (*start && myisspace(*start)) start++;
	if (!(*start)) break;
	
	end = start;
	while (*end && !myisspace(*end) && *end != '\n') end++;

	if (!*end) done = 1;
	*end = '\0';

	if (!strcmp(start, "rescue")) {
	    printf("\n\trescue disk mode enabled");
	    *isRescue = 1;
	} else if (!strcmp(start, "serial")) {
	    printf("\n\tserial mode enabled");
	    *isSerial = 1;
	} else if (!strcmp(start, "expert")) {
	    printf("\n\texpert mode enabled");
	    *isExpert = 1;
	} else if (!strcmp(start, "local")) {
	    *isCdrom = *isHd = 1;
	} else if (!strcmp(start, "cdrom")) {
	    *isCdrom = 1;
	} else if (!strcmp(start, "hd")) {
	    *isHd = 1;
	} else if (!strcmp(start, "network")) {
	    *isFtp = *isNfs = 1;
	} else if (!strcmp(start, "nfs")) {
	    *isNfs = 1;
	} else if (!strcmp(start, "ftp")) {
	    *isFtp = 1;
	} else if (!strcmp(start, "pcmcia")) {
	    *isPcmcia = *isCdrom = *isHd = *isFtp = *isNfs = 1;
	} else if (!strncmp(start, "kickstart", 9) || 
                   !strncmp(start, "ks", 2)) {
	    arg = strchr(start, '=');

	    if (arg) {
		*isKick = KICK_FLOPPY;
		printf("\n\ta floppy kickstart install will be performed");
	    } else {
		*isKick = KICK_BOOTP;
		printf("\n\ta bootp kickstart install will be performed");
	    }
	}
	start = end + 1;
    }
    printf("done\n");
}

int 
main (void)
{
  pid_t installpid, childpid;
  int waitStatus;
  int fd;
  int isRescue = 0;
  int isSerial = 0;
  int isExpert = 0;
  int isKick = 0;
  int isNfs = 0;
  int isFtp = 0;
  int isCdrom = 0;
  int isHd = 0;
  int isPcmcia = 0;
  int ok = 0;
  int doShutdown = 0;
  char * argv[15];
  char ** argvp = argv;

    /* getpid() != 1 should work, by linuxrc tends to get a larger pid */
    testing = (getpid() > 50);

    if (!testing) {
	/* turn off screen blanking */
	printf("\033[9;0]");
	printf("\033[8]");
    }

    printf("Linux-Mandrake install init version %s starting\n", VERSION);

    printf("mounting /proc filesystem... "); 
    if (!testing) {
	if (mount("/proc", "/proc", "proc", 0, NULL))
	    fatal_error(1);
    }
    printf("done\n");

    readargs(&isRescue, &isSerial, &isExpert, &isKick, &isNfs, &isFtp, &isCdrom, &isHd, &isPcmcia);
    
    if (isSerial) {
	printf("Red Hat install init version %s using a serial console\n", 
		VERSION);

	printf("remember, cereal is an important part of a nutritionally "
	       "balanced breakfast.\n\n");

	fd = open("/dev/ttyS0", O_RDWR, 0);
	if (fd < 0) {
	    printf("failed to open /dev/ttyS0");
	    fatal_error(1);
	}

	setupTerminal(fd);

	close(fd);
    } else {
	fd = open("/dev/tty1", O_RDWR, 0);
	if (fd < 0) {
	    printf("failed to open /dev/tty1");
	    fatal_error(1);
	}
    }

    dup2(fd, 0);
    dup2(fd, 1);
    dup2(fd, 2);
    close(fd);

    if (!testing) {
	sethostname("localhost.localdomain", 21);
	/* the default domainname (as of 2.0.35) is "(none)", which confuses 
	   glibc */
	setdomainname("", 0);
    }

    /* Now we have some /tmp space set up, and /etc and /dev point to
       it. We should be in pretty good shape. */

    if (!testing) doklog("/dev/tty4");

    /* Go into normal init mode - keep going, and then do a orderly shutdown
       when:

	1) /bin/install exits
	2) we receive a SIGHUP 
    */
    setsid();
    if (ioctl(0, TIOCSCTTY, NULL)) {
	printf("could not set new controlling tty");
    }

    printf("running install...\n"); 

    if (!(installpid = fork())) {
	/* child */

        *argvp++ = "/bin/install";

	if (isRescue) 
	    *argvp++ = "--rescue";

	if (isExpert) *argvp++ = "--expert";
	if (isKick == KICK_FLOPPY) 
	    *argvp++ = "--ks=floppy";
	else if (isKick)
	    *argvp++ = "--ks=bootp";

	if (isPcmcia)
  	  *argvp++ = "--pcmcia";
#ifdef __i386__
  	if (isCdrom)
  	  *argvp++ = "--cdrom";
  	if (isHd)
  	  *argvp++ = "--hd";
  	if (isNfs)
  	  *argvp++ = "--nfs";
  	if (isFtp)
  	  *argvp++ = "--ftp";
#endif
	
	*argvp++ = NULL;

	printf("executing %s\n", argv[0]);
	execve(argv[0], argv, env);
	printf("unable to execute %s???\n", argv[0]);
	
	return 0;
    }

    while (!doShutdown) {
	childpid = wait4(-1, &waitStatus, 0, NULL);

	if (childpid == installpid) 
	    doShutdown = 1;
    }

    if (!WIFEXITED(waitStatus) || WEXITSTATUS(waitStatus)) {
	printf("install exited abnormally ");
	if (WIFSIGNALED(waitStatus)) {
	    printf("-- recieved signal %d", WTERMSIG(waitStatus));
	}
	printf("\n");
    } else {
        if (isRescue) {
	    kill(klog_pid, 9);
            umount("/proc");
            printf("exiting init -> giving hand to rescue\n");
            return 0;
        }
	ok = 1;
    }

    if (testing) return 0;

    sync(); sync();

    if (!testing) {
	printf("sending termination signals...");
	kill(-1, 15);
	sleep(2);
	printf("done\n");

	printf("sending kill signals...");
	kill(-1, 9);
	sleep(2);
	printf("done\n");
    }

    unmountSwaps();

    printf("unmounting filesystems...\n"); 
    unmountFilesystems();

    if (ok) {
	printf("rebooting system\n");
	sleep(2);

	#ifdef USE_MINILIBC
	    reboot(0xfee1dead, 672274793, 0x1234567);
	#else
	    reboot(RB_AUTOBOOT);
	#endif
    } else {
	printf("you may safely reboot your system\n");
	while (1);
    }

    return 0;
}
