unit OTFEUnified_U;
// Description: Delphi Unified OTFE Component
// By Sarah Dean
// Email: sdean12@mailcity.com
// WWW:   http://www.fortunecity.com/skyscraper/true/882/
//
// This component combines functionality of OTFEBestCrypt, OTFEE4M, OTFEPGPDisk
// and OTFEScramDisk
//
// -----------------------------------------------------------------------------
// Version history:
// 0.9.0 - 6th November 1999
//         Initial beta release
//


interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  OTFE_U,
  OTFEConsts_U,
  OTFEBestCrypt_U,
  OTFEE4M_U,
  OTFEPGPDisk_U,
  OTFEScramDisk_U;

type
  TOTFESystem = (otfesBestCrypt,
                 otfesPGPDisk,
                 otfesE4M,              // Note: E4M must be immediatly before
                                        //       ScramDisk, just in case
                                        //       ScramDisk support is enabled
                 otfesScramDisk);       // Note: ScramDisk MUST be last, as
                                        //       ScramDisk volumes don't have
                                        //       any signature

const
  // Display names for each of the different types of OTF crypto system
  // Note: Each display name must be unique
  OTFESDispNames: array [TOTFESystem] of string = ('BestCrypt',
                                                   'PGPDisk',
                                                   'E4M',
                                                   'ScramDisk');

const
  // "Internal" names for each of the different types of OTF crypto system
  // Note: Each internal name must be unique, and contain no spaces or
  // punctuation.
  // This is intended that the user can use this where a display name would be
  // useless (e.g. if the corrosponding display name contains spaces, etc)
  OTFESInternalNames: array [TOTFESystem] of string = ('BESTCRYPT',
                                                       'PGPDISK',
                                                       'E4M',
                                                       'SCRAMDISK');

type
  TOTFEUnified = class(TOTFE)
  private
    { private declarations here}
  protected
    // Set the component active/inactive
    procedure SetActive(status: Boolean); override;
  public
    OTFComponents: array [TOTFESystem] of TOTFE;
    OTFEnabledComponents: array [TOTFESystem] of boolean;

    constructor Create(AOwner : TComponent); override;
    destructor  Destroy(); override;
    function  Mount(volumeFilename: string; readonly: boolean = FALSE): char; overload; override;
    function  Mount(volumeFilenames: TStringList; var mountedAs: string; readonly: boolean = FALSE): boolean; overload; override;
    function  Dismount(volumeFilename: string; emergency: boolean = FALSE): boolean; overload; override;
    function  Dismount(driveLetter: char; emergency: boolean = FALSE): boolean; overload; override;
    function  IsDriverInstalled(): boolean; overload; override;
    function  IsDriverInstalled(otfeSystem: TOTFESystem): boolean; overload;
    function  Version(): cardinal; overload; override;
    function  VersionStr(): string; overload; override;
    function  IsEncryptedVolFile(volumeFilename: string): boolean; override;
    function  DrivesMounted(): string; overload; override;
    function  GetVolFileForDrive(driveLetter: char): string; override;
    function  GetDriveForVolFile(volumeFilename: string): char; override;
    function  GetMainExe(): string; override;

    function  TypeOfEncryptedVolFile(volumeFilename: string; var typeOfFile: TOTFESystem): boolean;
    function  TypeOfEncryptedDrive(driveLetter: char; var typeOfFile: TOTFESystem): boolean;

    // OTF Crypto "internal" name to system type
    function  OTFSystemFromDispName(dispName: string; var OTFSystem: TOTFESystem): boolean;
    // OTF Crypto display name to system type
    function  OTFSystemFromInternalName(internalName: string; var OTFSystem: TOTFESystem): boolean;

    // For a *mounted* volume, sets OTFSystem to the OTF crypto system used for
    // that volume
    function  OTFSystemForDrive(driveLetter: char; var OTFSystem: TOTFESystem): boolean;
    // For a *mounted* volume, sets OTFSystem to the OTF crypto system used for
    // that volume
    function  OTFSystemForVolFile(volumeFilename: string; var OTFSystem: TOTFESystem): boolean;

  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('OTFE', [TOTFEUnified]);
end;


constructor TOTFEUnified.Create(AOwner : TComponent);
var
  OTFSystemLoop: TOTFESystem;
begin
  inherited;

  OTFComponents[otfesBestCrypt] := TOTFEBestCrypt.Create(nil);
  OTFComponents[otfesE4M      ] := TOTFEE4M.Create(nil);
  OTFComponents[otfesPGPDisk  ] := TOTFEPGPDisk.Create(nil);
  OTFComponents[otfesScramDisk] := TOTFEScramDisk.Create(nil);

  for OTFSystemLoop:=low(OTFComponents) to high(OTFComponents) do
    begin
    OTFEnabledComponents[OTFSystemLoop] := TRUE;
    end;

end;

destructor TOTFEUnified.Destroy();
var
  OTFSystemLoop: TOTFESystem;
begin
  for OTFSystemLoop:=low(OTFComponents) to high(OTFComponents) do
    begin
    OTFComponents[OTFSystemLoop].Free();
    end;

  inherited;

end;

procedure TOTFEUnified.SetActive(status: Boolean);
var
  OTFSystemLoop: TOTFESystem;
begin
  inherited;

  for OTFSystemLoop:=low(OTFComponents) to high(OTFComponents) do
    begin
    if OTFEnabledComponents[OTFSystemLoop] then
      begin
      try
        OTFComponents[OTFSystemLoop].Active := status;
      except
        // Damn. Oh well, better disable this component
        if OTFComponents[OTFSystemLoop].Active<>status then
          begin
          OTFEnabledComponents[OTFSystemLoop] := FALSE;
          FLastErrCode := OTFComponents[OTFSystemLoop].LastErrorCode;
          end;
      end;
      end;
    end;

  FActive := FALSE;
  for OTFSystemLoop:=low(OTFComponents) to high(OTFComponents) do
    begin
    if OTFComponents[OTFSystemLoop].Active then
      begin
      FActive := TRUE;
      break;
      end;
    end;

end;

function TOTFEUnified.Mount(volumeFilename: string; readonly: boolean = FALSE): char;
var
  stlTemp: TStringList;
  mountedAs: string;
begin
  stlTemp := TStringList.Create();
  try
    stlTemp.Add(volumeFilename);
    if Mount(stlTemp, mountedAs, readonly) then
      begin
      Result := mountedAs[1];
      end
    else
      begin
      Result := #0;
      end;
  finally
    stlTemp.Free();
  end;

end;


function TOTFEUnified.Mount(volumeFilenames: TStringList; var mountedAs: string; readonly: boolean = FALSE): boolean;
var
  i: integer;
  useOTFSystem: TOTFESystem;
begin
  Result := FALSE;

  for i:=0 to (volumeFilenames.count-1) do
    begin
    if TypeOfEncryptedVolFile(volumeFilenames[i], useOTFSystem) then
      begin
      Result := OTFComponents[useOTFSystem].Mount(volumeFilenames, mountedAs, readonly);
      FLastErrCode := OTFComponents[useOTFSystem].LastErrorCode;
      break;
      end;
    end;

end;

function TOTFEUnified.Dismount(volumeFilename: string; emergency: boolean = FALSE): boolean;
var
  OTFSystem: TOTFESystem;
begin
  Result := FALSE;

  if OTFSystemForVolFile(volumeFilename, OTFSystem) then
    begin
    Result := OTFComponents[OTFSystem].Dismount(volumeFilename, emergency);
    FLastErrCode := OTFComponents[OTFSystem].LastErrorCode;
    end;

end;


function TOTFEUnified.Dismount(driveLetter: char; emergency: boolean = FALSE): boolean;
var
  OTFSystem: TOTFESystem;
begin
  Result := FALSE;

  if OTFSystemForDrive(driveLetter, OTFSystem) then
    begin
    Result := OTFComponents[OTFSystem].Dismount(driveLetter, emergency);
    FLastErrCode := OTFComponents[OTFSystem].LastErrorCode;
    end;

end;


function TOTFEUnified.IsDriverInstalled(): boolean;
var
  retVal: boolean;
  OTFSystemLoop: TOTFESystem;
begin
  retVal := FALSE;

  for OTFSystemLoop:=low(OTFComponents) to high(OTFComponents) do
    begin
    if OTFComponents[OTFSystemLoop].Active then
      begin
      retVal := retVal OR OTFComponents[OTFSystemLoop].IsDriverInstalled();
      if retVal then
        begin
        break;
        end;
      end;
    end;

  Result := retVal;

end;



// Determine if a specific OTFE driver is installed
function TOTFEUnified.IsDriverInstalled(otfeSystem: TOTFESystem): boolean;
begin
  Result := OTFComponents[otfeSystem].IsDriverInstalled();

end;


function TOTFEUnified.Version(): cardinal;
begin
  Result := $FFFFFFFF;
end;

function TOTFEUnified.VersionStr(): string;
begin
  Result := '';
end;

function TOTFEUnified.IsEncryptedVolFile(volumeFilename: string): boolean;
var
  junkOTFSystem: TOTFESystem;
begin
  Result := TypeOfEncryptedVolFile(volumeFilename, junkOTFSystem);

end;

function TOTFEUnified.TypeOfEncryptedVolFile(volumeFilename: string; var typeOfFile: TOTFESystem): boolean;
var
  retVal: boolean;
  OTFSystemLoop: TOTFESystem;
begin
  retVal := FALSE;

  for OTFSystemLoop:=low(OTFComponents) to high(OTFComponents) do
    begin
    if OTFComponents[OTFSystemLoop].Active then
      begin
      retVal := retVal OR OTFComponents[OTFSystemLoop].IsEncryptedVolFile(volumeFilename);
      if retVal then
        begin
        typeOfFile := OTFSystemLoop;
        break;
        end;
      end;
    end;

  Result := retVal;

end;

function TOTFEUnified.TypeOfEncryptedDrive(driveLetter: char; var typeOfFile: TOTFESystem): boolean;
var
  OTFSystemLoop: TOTFESystem;
begin
  Result := FALSE;
  driveLetter := (uppercase(driveLetter))[1];

  for OTFSystemLoop:=low(OTFComponents) to high(OTFComponents) do
    begin
    if OTFComponents[OTFSystemLoop].Active then
      begin
      if (Pos(driveLetter, OTFComponents[OTFSystemLoop].DrivesMounted())>0) then
        begin
        typeOfFile := OTFSystemLoop;
        Result := TRUE;
        break;
        end;
      end;
    end;

end;


function TOTFEUnified.DrivesMounted(): string;
var
  retVal: string;
  OTFSystemLoop: TOTFESystem;
begin
  retVal := '';

  for OTFSystemLoop:=low(OTFComponents) to high(OTFComponents) do
    begin
    if OTFComponents[OTFSystemLoop].Active then
      begin
      retVal := retVal + OTFComponents[OTFSystemLoop].DrivesMounted();
      end;
    end;

  Result := SortString(retVal);

end;

function TOTFEUnified.GetVolFileForDrive(driveLetter: char): string;
var
  retVal: string;
  OTFSystemLoop: TOTFESystem;
begin
  retVal := '';

  for OTFSystemLoop:=low(OTFComponents) to high(OTFComponents) do
    begin
    if OTFComponents[OTFSystemLoop].Active then
      begin
      retVal := OTFComponents[OTFSystemLoop].GetVolFileForDrive(driveLetter);
      if retVal<>'' then
        begin
        break;
        end;
      end;
    end;

  Result := retVal;

end;


function TOTFEUnified.GetDriveForVolFile(volumeFilename: string): char;
var
  retVal: char;
  OTFSystemLoop: TOTFESystem;
begin
  retVal := #0;

  for OTFSystemLoop:=low(OTFComponents) to high(OTFComponents) do
    begin
    if OTFComponents[OTFSystemLoop].Active then
      begin
      retVal := OTFComponents[OTFSystemLoop].GetDriveForVolFile(volumeFilename);
      if retVal<>#0 then
        begin
        break;
        end;
      end;
    end;

  Result := retVal;

end;

function TOTFEUnified.OTFSystemFromDispName(dispName: string; var OTFSystem: TOTFESystem): boolean;
var
  OTFSystemLoop: TOTFESystem;
begin
  Result := FALSE;

  for OTFSystemLoop:=low(OTFComponents) to high(OTFComponents) do
    begin
    if dispName=OTFESDispNames[OTFsystem] then
      begin
      OTFSystem := OTFSystemLoop;
      Result := TRUE;
      break;
      end;
    end;

end;

function TOTFEUnified.OTFSystemFromInternalName(internalName: string; var OTFSystem: TOTFESystem): boolean;
var
  OTFSystemLoop: TOTFESystem;
begin
  Result := FALSE;

  for OTFSystemLoop:=low(OTFComponents) to high(OTFComponents) do
    begin
    if internalName=OTFESInternalNames[OTFsystem] then
      begin
      OTFSystem := OTFSystemLoop;
      Result := TRUE;
      break;
      end;
    end;

end;

function TOTFEUnified.OTFSystemForDrive(driveLetter: char; var OTFSystem: TOTFESystem): boolean;
var
  OTFSystemLoop: TOTFESystem;
begin
  Result := FALSE;

  driveLetter := (uppercase(driveLetter))[1];
  for OTFSystemLoop:=low(OTFComponents) to high(OTFComponents) do
    begin
    if OTFComponents[OTFSystemLoop].Active then
      begin
      if pos(driveLetter, OTFComponents[OTFSystemLoop].DrivesMounted())>0 then
        begin
        OTFSystem := OTFSystemLoop;
        Result := TRUE;
        break;
        end;
      end;
    end;

end;


function TOTFEUnified.OTFSystemForVolFile(volumeFilename: string; var OTFSystem: TOTFESystem): boolean;
var
  OTFSystemLoop: TOTFESystem;
begin
  Result := FALSE;

  for OTFSystemLoop:=low(OTFComponents) to high(OTFComponents) do
    begin
    if OTFComponents[OTFSystemLoop].Active then
      begin
      if GetDriveForVolFile(volumeFilename)<>#0 then
        begin
        OTFSystem := OTFSystemLoop;
        Result := TRUE;
        break;
        end;
      end;
    end;

end;

function TOTFEUnified.GetMainExe(): string;
begin
  FLastErrCode:= OTFE_ERR_UNABLE_TO_LOCATE_FILE;
  Result := '';

end;


END.


