/*
 * kpvm_demo.cpp
 *
 * Copyright (c) 1998 Michael Kropfberger (michael.kropfberger@gmx.net)
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <unistd.h>

#include <qdict.h>
#include <qapp.h>

#include "kpvm_demo.h"
#include "kpvm_demo.moc"


/****************************************************************************/
/************************    CLASS   Person    ******************************/
/****************************************************************************/
Person::Person(Serialize *parent, const char *name)
  :Serialize(parent,name)
{ 
  _firstName="";
  _surName="";
  _persNr=0;
  _money=0.0;

};

/****************************************************************************/
void Person::setFirstName(QString fname) {
  _firstName=fname;
  emit firstNameChanged();
};

/****************************************************************************/
void Person::setSurName(QString sname) {
  _surName=sname;
  emit surNameChanged();
};

/****************************************************************************/
void Person::setPersNr(int pnr) {
  _persNr=pnr;
  emit persNrChanged();
};

/****************************************************************************/
void Person::setMoney(float m) {
  _money=m;
  emit moneyChanged();
};


/****************************************************************************/
QList<VarTuple> Person::objState() const
{
  QList<VarTuple> vtl;
  vtl.append( new VarTuple("_firstName",_firstName) );
  vtl.append( new VarTuple("_surName",_surName) );
  vtl.append( new VarTuple("_persNr",_persNr) );
  vtl.append( new VarTuple("_money",_money) );
  return vtl;
}; //objState


/****************************************************************************/
void Person::setFixedOrderObjState(QList<VarTuple> state)
{ 
  // we know the sequence and can be sure, that all variables are there
  unsigned int nrOfVarTuples = objState().count();
  if (state.count() != nrOfVarTuples) 
     fatal("setFixedOrderObjState didn't get correct number of varTuples"
           " (got: %d, should be %d)",state.count(),nrOfVarTuples);
  state.first()->storeValueTo(_firstName);
  state.next()->storeValueTo(_surName);
  state.next()->storeValueTo(_persNr);
  state.next()->storeValueTo(_money);
}; //setFixedOrderObjState 


/****************************************************************************/
void Person::setObjState(QList<VarTuple> state)
{ 
  // we don't know the sequence, so we build a hashtable (QDict)
  QDict<VarTuple> dict;
  VarTuple* vt;
  for(vt=state.first();vt != NULL;vt=state.next()) {
     // every varTuple has to have a descriptive (nonEmpty) tag
     if ( vt->tag().isEmpty() )
        fatal("got an VarTuple with an empty VarTag-Field!");
     // double occurancies are overwritten
     dict.replace(vt->tag(),state.current());
  }
  // we can't be sure, if the variables are really in the QDict
  if ( (vt=dict["_firstName"]) ) vt->storeValueTo(_firstName);
  if ( (vt=dict["_surName"]) ) vt->storeValueTo(_surName);
  if ( (vt=dict["_persNr"]) ) vt->storeValueTo(_persNr);
  if ( (vt=dict["_money"]) ) vt->storeValueTo(_money);
}; //setObjState
 


/****************************************************************************/
/************************    CLASS   MyKPvm    ******************************/
/****************************************************************************/
MyKPvm::MyKPvm(KPvm *parent=0, const char *name=0)
       : KPvm(parent,name) 
{ 
  //debug(" MyKPvm::Constructor");
  _persons.setAutoDelete(TRUE); //removal of list entries also frees memory
  output = new QMultiLineEdit(this);
  output->setGeometry(0,0,this->width(),this->height());
  output->setReadOnly(true);
  output->setCursor(waitCursor);  // to symbolize the running process

  if (this->hasPvmParent()) {  //PvmChild
    outputS.sprintf("Child t%x on [%s]",this->tid(),this->hostName().data() );
    this->setCaption( outputS );
    output->setBackgroundColor(blue);
    this->show();
    KPvmParent *myPvmParent=this->pvmParent(); CHECK_PTR(myPvmParent);
    outputS.sprintf("Hi! I'm t%x and I was started from t%x\n",
           this->tid(), myPvmParent->tid() );
    output->append( outputS );

    // every received data is processed in our childHandleReceivedData
    connect(this, SIGNAL(dataReceived(QString, QList<VarTuple>
                        , int, KPvmEntity*))
           , this, SLOT(childHandleReceivedData(QString, QList<VarTuple>
                       , int, KPvmEntity*)) );

  } else { //is a PvmParent
 
    outputS.sprintf("Parent t%x on [%s]",this->tid(),this->hostName().data() );
    this->setCaption( outputS );
    output->setBackgroundColor(red);
    this->show();
    outputS.sprintf("Hi! I'm the parent t%x!\n", this->tid());
    output->append(outputS);
    Person *person;

    person = new Person();
    person->setFirstName("Peter");  person->setSurName("Pan");
    person->setPersNr(9123872);     person->setMoney(55.3);
    _persons.append(person);

    person = new Person();
    person->setFirstName("Pater");  person->setSurName("Pin");
    person->setPersNr(9320453);     person->setMoney(87.1); 
    _persons.append(person);
   
    //QString hs=this->hostName()+":0.0";
    //char *env[] = {  "-display", hs.data(), '\0'  };

    // spawns children and puts the handles to them (KPvmChild) in a list
    QList<KPvmChild> children=this->spawn("kpvm_demo",environ
                                        ,PvmTaskDefault,"",NUMPROCS); 
    for (KPvmChild* child=children.first();child!=0;child=children.next()) {
        outputS.sprintf("started child t%x on host [%s]"
                      ,child->tid(),(const char *)child->hostName() );
        output->append(outputS);
    }
    
    // now iterate through the list and send all persons to child.
    // after sending a person it is immediately deleted on parent
    outputS.sprintf("sending all persons to all children");
    output->append(outputS);
    Person *pers=_persons.first();
    while(pers!=0) {
       //send Person to all children
       for (KPvmChild* child=children.first();child!=0;child=children.next())
           // sends the actual person netEfficiently to the children
           this->send(child, _persons.first(),55,true);
       pers=_persons.next();
       _persons.remove(_persons.first()); //delete person totally
    };

    output->append("OK... now sending the finishing signal...");
    TransferFinished tf;
    for (KPvmChild* child=children.first();child!=0;child=children.next())
       this->send(child, &tf,99,true);

    output->setCursor(arrowCursor);  // to symbolize the finished process
    output->append("\n\nDONE! Close the window to exit!");
  };//PvmParent
};


/****************************************************************************/
void MyKPvm::childHandleReceivedData(QString className
                      , QList<VarTuple> objState
                      , int msgId, KPvmEntity *entity)
{
    outputS.sprintf("got a [%s] with msgid %i from t%x",className.data()
                     ,msgId, entity->tid() );
    output->append(outputS);
    if (className=="Person") {
       Person *p=new Person();
       p->setFixedOrderObjState(objState);
       outputS.sprintf("Person was %s %s, # %d, with %6.2f $"
                       ,p->firstName().data(), p->surName().data()
                       , p->persNr(), p->money() );
        output->append(outputS);
       _persons.append(p);
       return;
    }//if person

    if (className=="TransferFinished") {
        output->setCursor(arrowCursor);  // to symbolize the finished process
        output->append("\n\nDONE! Close the window to exit!");
       return;
    }//if TransferFinished
};

/****************************************************************************/
void MyKPvm::closeEvent( QCloseEvent *) {
  // ALT-C or close-Window etc.
  this->exit();
};

/****************************************************************************/
int main(int argc, char **argv) 
{
  QApplication qapp(argc, argv);
  MyKPvm myKPvm;
  qapp.setMainWidget(&myKPvm);
  return qapp.exec();
};
