/* This file is part of the KDE libraries
    Copyright (c) 1998 Emmeran Seehuber (the_emmy@hotmail.com)
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

/////////////////////////////////////////////////////////////////////
// Group baseclass
/////////////////////////////////////////////////////////////////////

#include "klgroup.h"
#include "klgrouppriv.h"
#include "stdlib.h"
#include "kltools.h"
#include "kllabel.h"
#include "kldesigner.h"

IMPLEMENT_KLMETA_STANDALONE(KLGroup,KLChild,"Group");

const int KLGroup::DefaultSpacing = 5;

const KLGroup::KLChildList &KLGroup::childList() const
{
  return childs;
}

KLGroup::KLGroup()
{
  a_sameSize = false;
  a_xSpacing = DefaultSpacing;
  a_ySpacing = DefaultSpacing;
  a_spaceBorder = false; 
  spacingOverwritten = false; 
}

KLGroup::~KLGroup()
{
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) 
    delete it.current();
}

void KLGroup::setupGrap()
{
  KLChild::setupGrap();
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) 
    it.current()->setupGrap();
}


void KLGroup::cleanupGrap()
{
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) 
    it.current()->cleanupGrap();
  KLChild::cleanupGrap();
}


void KLGroup::addDropMarks( KLDropMarkList * dml ) const
{
  KLChild::addDropMarks(dml);
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    if( !it.current()->hidden() )
      it.current()->addDropMarks(dml);
  }
}

bool KLGroup::isAChild( KLChild *_child ) const
{
  if( KLChild::isAChild( _child ) )
    return true;
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    if( it.current()->isAChild(_child) )
      return true;
  }
  return false;
}

ulong KLGroup::countShowedChilds()
{
  ulong count = 0;
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    if( !it.current()->hidden() )
      count++;
  }
  return count;
}

bool KLGroup::klSetup( KLSetupInfo *setupInfo )
{
  // Do baseclass setup
  if( !KLChild::klSetup( setupInfo ) ) 
    return false;

  // Read settings (not yet)
  if( !spacingOverwritten ) {
    // a_xSpacing = setupInfo->config->xspacing
    // a_ySpacing = setupinfo->config->yspacing
  }

  // Setup all childs
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    KLChild *child = it.current();

    // Not setet up yet ?
    if( child->a_state == SI_None ) {
      // Just setup it !
      if( !child->klSetup(setupInfo) ) {
        // Failed !? -> Cleanup ourselff and the baseclass
        this->KLGroup::klCleanup();
        return false;
      }
    } // if child->a_state == SI_None

  } // for
  return true;
}


void KLGroup::klHide()
{
  // Hide everything
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    KLChild *child = it.current();

    // Not hidden yet !?
    if( child->a_state == SI_Showed )
      child->klHide(); // Just hide it !

  } // for

  KLChild::klHide();
}

void KLGroup::klCleanup()
{
  // Hide everything
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    KLChild *child = it.current();

    // Not hidden yet !?
    if( child->a_state == SI_SetupDone )
      child->klCleanup(); // Just hide it !

  } // for

  KLChild::klCleanup();
}

KLChild *KLGroup::findChild( ulong x, ulong y ) const
{
  // Do we match ?
  if( !KLChild::findChild(x,y) )
    return 0; // No !
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    KLChild *child = it.current()->findChild(x,y);
    if( child )
      return child;
  }
  return (KLGroup *)this;
}

void KLGroup::setSameSize(bool samesize)
{
  if( samesize != a_sameSize ) {
    a_sameSize = samesize;
    doRelayout(false);
  }
}

void KLGroup::setXSpacing( ulong x )
{
  if( a_xSpacing != x ) {
    a_xSpacing = x;
    doRelayout(false);
  }
}


void KLGroup::setYSpacing( ulong y )
{
  if( a_ySpacing != y ) {
    a_ySpacing = y;
    doRelayout(false);
  }
}


void KLGroup::setSpaceBorder( bool spaceBorder )
{
  if( a_spaceBorder != spaceBorder ) {
    a_spaceBorder = spaceBorder;
    doRelayout(false);
  }
}


bool KLGroup::sameSize() const
{
  return a_sameSize;
}

ulong KLGroup::xSpacing() const
{
  return a_xSpacing;
}

ulong KLGroup::ySpacing() const
{
  return a_ySpacing;
}


bool KLGroup::spaceBorder() const
{
  return a_spaceBorder;
}


KLGroup &KLGroup::operator<<(KLChild *child)
{
  addChild(child);
  return *this;
}

KLGroup &KLGroup::operator<<(KLChild &child)
{
  addChild(&child);
  return *this;
}

KLGroup &KLGroup::operator<<(KLGroup::_grp_change_func func)
{
  func(*this);
  return *this;
}

KLGroup &KLGroup::operator<<(KLChild::_child_change_func func)
{
  func(*this);
  return *this;
}

KLChild::KLStreamHelp &KLGroup::operator<<(KLChild::_child_change_p1_func func)
{
  static KLStreamHelp sh;
  sh.child = this;
  sh.func = func;
  return sh;
}


bool KLGroup::addChild( KLChild *child, KLChild *toAddBefore )
{
  HASSERT( child );

  // Set parent
  child->parentChild = this;

  // Setup the new member if needed
  switch( a_state ) { // Are we setted up ?
    case SI_Showed :
    case SI_SetupDone : if( !child->klSetup(&a_setupInfo) ) {
                          WARN( "Cant setup new member !\n" );
                          return false;
                        }
    case SI_None: break; // Nothing to be donee
  }

  // Add child to list
  if( !toAddBefore ) 
    childs.append( child );
  else 
    childs.insert( childs.findRef(toAddBefore), child );

  // Perform a relayout 
  doRelayout(false);

  return true;
}


void KLGroup::remChild( KLChild *child )
{
  HASSERT( child );
  HASSERT( childs.containsRef(child) == 1 ); // Must just only one time exists !!!

  // Hide/Cleanup child
  if( child->a_state == SI_Showed )
    child->klHide();
  if( child->a_state == SI_SetupDone )
    child->klCleanup();

  childs.removeRef(child);

  // Perform a relayout
  doRelayout(false);
}

bool KLGroup::MyChildMeta::dumpObject(KLChild *_child, KLDumpDevice& dev) const
{
  QString helpStr;
  KLGroup *child = (KLGroup *)_child;
  dev.writeEntry("samesize",child->sameSize() ? "1" : "0" );
  dev.writeEntry("xspacing",helpStr.setNum(child->xSpacing()));
  dev.writeEntry("yspacing",helpStr.setNum(child->ySpacing()));
  dev.writeEntry("spaceborder",child->spaceBorder() ? "1" : "0" );
  
  // Dump all childs
  KLChildListIt it(child->children());
  for( ; it.current(); ++it ) {
    if( !KLChildMeta::dumpCompleteObject(it.current(), dev) ) 
      return false;
  }
  return true; 
}


void KLGroup::MyChildMeta::restoreObject(KLChild *_child, KLDumpDevice& dev) const
{
  QString helpStr;
  KLGroup *child = (KLGroup *)_child;
  child->setSameSize(atol(dev.readEntry("samesize",child->sameSize() ? "1" : "0" )) == 1);
  child->setXSpacing(atol(dev.readEntry("xspacing",helpStr.setNum(child->xSpacing()))));
  child->setYSpacing(atol(dev.readEntry("yspacing",helpStr.setNum(child->ySpacing()))));
  child->setSpaceBorder(atol(dev.readEntry("spaceborder",child->spaceBorder() ? "1" : "0" )) == 1);

  // Restore all childs
  QString linestr;
  ulong origlinestart = dev.linestart;
  ulong line = origlinestart, level = 0;
  while( dev.readLine(linestr,line,level) ) {
    if( (level == 2) && (linestr[0] == '{') ) {
      dev.linestart = line - 1;
      KLChild *_child = KLChildMeta::restoreCompleteObject(dev);
      if( _child ) 
        child->addChild( _child );
    }
  }
  dev.linestart = origlinestart;
}


KLChild *KLGroup::MyChildMeta::createObjectEdit(KLChild *child,bool) const
{
  HASSERT(child->metaInherits("KLGroup"));
  KLGroup *group = (KLGroup *)child;
  KLGroupEdit *edit = new KLGroupEdit();
  edit->group = group;
  KLLineEdit *xspacing, *yspacing;
  KLCheckBox *samesize, *spaceborder;
  KLGridGroup *gridgroup = (KLGridGroup *)KLChildMeta::createObjectEdit(child,false);
  *gridgroup << new KLHVSpace()
             << (samesize = new KLCheckBox("Same size"))
             << (*(new KLLabel("X Spacing")) << ::setWeight << 1)
             << (xspacing = edit)
             << (*(new KLLabel("Y Spacing")) << ::setWeight << 1)
             << (yspacing = new KLLineEdit())
             << new KLHVSpace()
             << (spaceborder = new KLCheckBox("Space border"));
  
  QObject::connect(samesize,SIGNAL(toggled(bool)),edit,SLOT(onSameSize(bool)));
  QObject::connect(xspacing,SIGNAL(textChanged(const char*)),edit,SLOT(onXSpacing(const char *)));
  QObject::connect(yspacing,SIGNAL(textChanged(const char*)),edit,SLOT(onYSpacing(const char *)));
  QObject::connect(spaceborder,SIGNAL(toggled(bool)),edit,SLOT(onSpaceborder(bool)));

  QString helpStr;
  samesize->setChecked(group->sameSize());
  xspacing->setText(helpStr.setNum(group->xSpacing()));
  yspacing->setText(helpStr.setNum(group->ySpacing()));
  spaceborder->setChecked(group->spaceBorder());

  return gridgroup;
}


void KLGroupEdit::onSameSize(bool samesize)
{
  group->setSameSize( samesize );
}

void KLGroupEdit::onXSpacing(const char *text)
{
  group->setXSpacing(atol(text));
}

void KLGroupEdit::onYSpacing(const char *text)
{
  group->setYSpacing(atol(text));
}

void KLGroupEdit::onSpaceborder(bool border)
{
  group->setSpaceBorder(border);
}


/////////////////////////////////////////////////////////////////////
// HV Group 
/////////////////////////////////////////////////////////////////////


void KLHVGroup::setHoriz(bool horiz)
{
  if( a_horiz != horiz ) {
    a_horiz = horiz;
    doRelayout(false);
  }
}

void KLHVGroup::addDropMarks( KLDropMarkList * dml ) const
{
  KLGroup::addDropMarks(dml);
  HASSERT( a_state == SI_Showed );
  ulong x = a_showInfo.x;
  ulong y = a_showInfo.y;
  ulong xSize;
  ulong ySize;

  if( a_horiz ) {
    xSize = a_showInfo.xSize;
    ySize = 1;
    y += (spaceBorder() ? ySpacing() : 0 );
  }
  else {
    xSize = 1;
    ySize = a_showInfo.ySize;
    x += (spaceBorder() ? xSpacing() : 0 );
  }

  bool first = true;
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    KLChild *child = it.current();
    if( !child->hidden() ) {
      if( first ) {
        dml->addDropMark(x,y,xSize,ySize,KLDropMark::DA_AddBefore,(KLGroup *)this,child,getNextNotHidden(child),0);
        first = false;
      }
      if( a_horiz ) {
        y += child->showInfo().ySize;
      }
      else {
        x += child->showInfo().xSize;
      }
      ulong xa = 0;
      ulong ya = 0;
      KLChild *nextChild;
      if( (nextChild = getNextNotHidden(child)) ) {
        if( !a_horiz )
          xa += xSpacing() / 2;
        else
          ya += ySpacing() / 2;
        dml->addDropMark(x+xa,y+ya,xSize,ySize,KLDropMark::DA_AddBefore,(KLGroup *)this,nextChild,nextChild,child);
      }
      else
        dml->addDropMark(x+xa,y+ya,xSize,ySize,KLDropMark::DA_Append,(KLGroup *)this,0,child,0);
      if( a_horiz ) {
        y += ySpacing();
      }
      else {
        x += xSpacing();
      }
    } // if( !child->hidden() )
  } // for
}


KLChild *KLHVGroup::getNextNotHidden(KLChild *child)  const
{
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    if( child ==  it.current() ) {
      ++it;
      while( it.current() && it.current()->hidden() )
        ++it;
      return it.current();
    }
  }
  return 0;
}


KLChild *KLHVGroup::getPrevNotHidden(KLChild *child)  const
{
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    if( child ==  it.current() ) {
      --it;
      while( it.current() && it.current()->hidden() )
        --it;
      return it.current();
    }
  }
  return 0;
}


bool KLHVGroup::horiz()
{
  return a_horiz;
}

void KLHVGroup::sumWeight( ulong &xWeight, ulong &yWeight )
{
  xWeight = yWeight = 0;

  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    KLChild *child = it.current();
    if( !child->layoutFinal && !child->hidden() ) {
      xWeight += child->xWeight();
      yWeight += child->yWeight();
    }
  }
}


bool KLHVGroup::klAskMinMax(KLMinMaxSizes *minMaxSizes)
{
  // Call baseclass
  if( !KLGroup::klAskMinMax(minMaxSizes) )
    return false;

  // Calculate sizes
  if( a_sameSize ) 
    askMinMaxSS(minMaxSizes);
  else
    askMinMaxN(minMaxSizes);
  minMaxSizes->validate();

  // Everything is done well
  return true;
}

void KLHVGroup::askMinMaxN(KLMinMaxSizes *mms)
{
  // Get the sizes of all childs and calculate our size
  ulong min = 0, max = 0;
  ulong defX = 0, defY = 0;

  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    KLChild *child = it.current();

    if( !child->hidden() ) {
      // Assert right state (askMinMax is possible if setet up and if showed)
      HASSERT( child->a_state == SI_SetupDone || child->a_state == SI_Showed );
      
      // Get sizes of the child
      KLChild::KLMinMaxSizes &minMaxSizes = child->minMaxSizes();
      minMaxSizes.init();
      child->klDoAskMinMax( &minMaxSizes );

      // Geth the default sizes 
      ulong childDefX =  minMaxSizes.defX * child->xWeight() / KLDefWeight;
      ulong childDefY =  minMaxSizes.defY * child->yWeight() / KLDefWeight;

      // Calc our sizes (algo stolen from MUI. See MUI/Developer.guide at www.sasg.com )
      // Remeber: x = row, y = coloumn
      if( !a_horiz ) {

        // Horiz summing
        // Add horiz
        mms->minX += minMaxSizes.minX + a_xSpacing;
        mms->maxX += minMaxSizes.maxX + a_xSpacing;

        // Maxmize/minimize vert
        min = KLMax(min, minMaxSizes.minY);
        max = KLNonNullMin(max,minMaxSizes.maxY);

        // Sum default
        defX += childDefX + a_xSpacing;
        defY = KLMax(defY,childDefY);

      }
      else {
        // Vert summing
        // Add vert
        mms->minY += minMaxSizes.minY + a_ySpacing;
        mms->maxY += minMaxSizes.maxY + a_ySpacing;

        // Maxmize/minimize horiz
        min = KLMax(min, minMaxSizes.minX);
        max = KLNonNullMin(max,minMaxSizes.maxX);

        // Sum default
        defX = KLMax(defX,childDefX);
        defY += childDefY + a_ySpacing;
      }

    } // if !child->a_hidden

  } // for 


  // Now add/sub borders and spaces
  ulong addXSpace = a_spaceBorder ? a_xSpacing * 2 : 0;
  ulong addYSpace = a_spaceBorder ? a_ySpacing * 2 : 0;
  if( !a_horiz ) {
    mms->minY += min + addYSpace;
    mms->maxY += max + addYSpace;
    mms->minX += addXSpace - a_xSpacing;
    mms->maxX += addXSpace - a_xSpacing;
    defX += addXSpace - a_xSpacing;
    defY += addYSpace;
  }
  else {
    mms->minX += min + addXSpace;
    mms->maxX += max + addXSpace;
    mms->minY += addYSpace - a_ySpacing;
    mms->maxY += addYSpace - a_ySpacing;
    defY += addYSpace - a_ySpacing;
    defX += addXSpace;
  }

  // Set Def sizes
  mms->defX += defX;
  mms->defY += defY;
}

void KLHVGroup::askMinMaxSS(KLMinMaxSizes *mms)
{
  // Get the sizes of the childs
  ulong minX = 0, minY = 0;
  ulong maxX = 0, maxY = 0;
  ulong count = 0;
  ulong defX = 0, defY = 0;

  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    KLChild *child = it.current();

    if( !child->hidden() ) {
      // Assert right state (askMinMax is possible if setet up and if showed)
      HASSERT( child->a_state == SI_SetupDone || child->a_state == SI_Showed );
      count++;
      
      // Get sizes of the child
      KLChild::KLMinMaxSizes &minMaxSizes = child->minMaxSizes();
      minMaxSizes.init();
      child->klDoAskMinMax( &minMaxSizes );

      minX = KLMax(minX,minMaxSizes.minX);
      minY = KLMax(minY,minMaxSizes.minY);
      if( !a_horiz ) {
        maxX = KLMax(maxX,minMaxSizes.maxX);
        maxY = KLNonNullMin(maxY,minMaxSizes.maxY);
      }
      else {
        maxX = KLNonNullMin(maxX,minMaxSizes.maxX);
        maxY = KLMax(maxY,minMaxSizes.maxY);
      }
      defX = KLMax(defX,minMaxSizes.defX);
      defY = KLMax(defY,minMaxSizes.defY);

    } // if !child->a_hidden

  } // for 



  // Now add/sub borders and spaces
  ulong addXSpace = a_spaceBorder ? a_xSpacing * 2 : 0;
  ulong addYSpace = a_spaceBorder ? a_ySpacing * 2 : 0;
  if( !a_horiz ) {
    minX = (minX * count) + (a_xSpacing * (count - 1) ) + addXSpace;
    maxX = (maxX * count) + (a_xSpacing * (count - 1) ) + addXSpace;
    defX = (defX * count) + (a_xSpacing * (count - 1) ) + addXSpace;
    minY += addYSpace;
    maxY += addYSpace;
    defY += addYSpace;
  }
  else {
    minY = (minY * count) + (a_ySpacing * (count - 1) ) + addYSpace;
    maxY = (maxY * count) + (a_ySpacing * (count - 1) ) + addYSpace;
    defY = (defY * count) + (a_ySpacing * (count - 1) ) + addYSpace;
    minX += addXSpace;
    maxX += addXSpace;
    defX += addXSpace;
  }


  // Add sizes 
  mms->minX += minX;
  mms->minY += minY;
  mms->maxX += maxX;
  mms->maxY += maxY;
  mms->defX += defX;
  mms->defY += defY;
}


bool KLHVGroup::klShow(KLShowInfo *showInfo)
{
  if( !KLGroup::klShow(showInfo) ) 
    return false;

  ulong childCount = countShowedChilds();
  if( !childCount ) // No visible member to show ?
    return true;

  ulong availXSpace = showInfo->xSize - ( a_spaceBorder ? a_xSpacing * 2 : 0 );
  ulong availYSpace = showInfo->ySize - ( a_spaceBorder ? a_ySpacing * 2 : 0 );
  if( !a_horiz )
    availXSpace -= (childCount - 1 ) * a_xSpacing;
  else
    availYSpace -= (childCount - 1 ) * a_ySpacing;


  // Get the cursor to begin with showing controls
  ulong actX = showInfo->x + ( a_spaceBorder ? a_xSpacing : 0 );
  ulong actY = showInfo->y + ( a_spaceBorder ? a_ySpacing : 0 );


  // Assert for correct space
  HASSERT( availXSpace && (availXSpace <= KLMaxSize) );
  HASSERT( availYSpace && (availYSpace <= KLMaxSize) );

  // Call layout routines
  if( a_sameSize ) 
    return showSS( actX, actY, availXSpace, availYSpace );
  else
    return showN( actX, actY, availXSpace, availYSpace );
}


bool KLHVGroup::showN( ulong actX, ulong actY, ulong compAvailXSpace, ulong compAvailYSpace )
{
  // Clone sizes
  ulong availXSpace = compAvailXSpace;
  ulong availYSpace = compAvailYSpace;

  // Pass 1: Give the object the sizes

  // Pass 1.1:
  // Sub the minimum sizes of the object from the avail size.
  // Also unset the layouted flag (or set it for hidden childs)

  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    KLChild *child = it.current();

    if( !child->hidden() ) {
      // Sub the minimum
      if( !a_horiz )
        availXSpace -= child->minMaxSizes().minX;
      else
        availYSpace -= child->minMaxSizes().minY;
      child->layoutFinal = false;
    } // if !child->a_hidden
    else 
      child->layoutFinal = true; // Mark as layouted
  } // for 

  // Get the weight
  ulong xWeight, yWeight;
  sumWeight(xWeight,yWeight);

  // Now the availXSpace/availYSpace containts the extra space avaible  wich could
  // be spreaded under the childs

  // Save the orginal avail space
  ulong orgXAvail = availXSpace, orgYAvail = availYSpace;
  // Avaible weigh (only for determinateing the end of the layout)
  ulong axWeight = xWeight, ayWeight = yWeight;
  xWeight = xWeight == 0 ? 1 : xWeight;
  yWeight = yWeight == 0 ? 1 : yWeight;

  // Pass 1.2:
  // Repeat until the rest of the avaible space is given to the childs
  bool cont = true;
  while( cont ) {

    // Go, and give the members there space (under using there weight)
    for( it.toFirst(); it.current(); ++it ) {
      KLChild *child = it.current();

      // If not final layout reached, (re)layout the child
      if( !child->layoutFinal ) {
        KLShowInfo &showInfo = child->showInfo();
        ulong minX = child->minMaxSizes().minX;
        ulong minY = child->minMaxSizes().minY;

        if( !a_horiz ) {
          // Calcluate the "should" sizes using the weight
          // This means "Weight says, the child should get this sizes"
          HASSERT( xWeight );
          ulong xSize = (orgXAvail * child->xWeight()) / xWeight;
          showInfo.xSize = minX + xSize;
          showInfo.ySize = orgYAvail;
          availXSpace -= xSize;

          // Check the sizes

          // This means "If the given space is to big for the child,
          //             we just give it the maximal posible space, and
          //             spread the rest with all other (if they want
          //             some size)"
          HASSERT( showInfo.xSize >= minX );
          HASSERT( showInfo.ySize >= minY );
          if( showInfo.xSize > child->minMaxSizes().maxX ) {
            availXSpace += xSize;
            showInfo.xSize = child->minMaxSizes().maxX;
            availXSpace -= showInfo.xSize - minX;
            child->layoutFinal = true;
          }
          else
            axWeight -= child->xWeight();

          // Check the y size
          if( showInfo.ySize > child->minMaxSizes().maxY )
            showInfo.ySize = child->minMaxSizes().maxY;
        }
        else {
          // Calcluate the "should" sizes using the weight
          // This means "Weight says, the child should get this sizes"
          HASSERT( yWeight );
          ulong ySize = (orgYAvail * child->yWeight()) / yWeight;
          showInfo.ySize = minY + ySize;
          showInfo.xSize = orgXAvail;
          availYSpace -= ySize;

          // Check the sizes

          // This means "If the given space is to big for the child,
          //             we just give it the maximal posible space, and
          //             spread the rest with all other (if they want
          //             some size)"
          HASSERT( showInfo.xSize >= minX );
          HASSERT( showInfo.ySize >= minY );
          if( showInfo.ySize > child->minMaxSizes().maxY ) {
            availYSpace += ySize;
            showInfo.ySize = child->minMaxSizes().maxY;
            availYSpace -= showInfo.ySize - minY;
            child->layoutFinal = true;
          }
          else
            ayWeight -= child->yWeight();

          // Check the x size
          if( showInfo.xSize > child->minMaxSizes().maxX )
            showInfo.xSize = child->minMaxSizes().maxX;
        } // if a_horiz : else

      } // if( !child->layoutFinal ) 
    } // for it.current


    // All spaces given to the members ? (Indicated by weight)
    if( (!a_horiz ? axWeight : ayWeight) == 0 ) 
      cont = false; // Yes stop this loop, and start showing
    else {
      // Not yet ! Get the now avaible weight for all non "final layouted" childs
      sumWeight(xWeight,yWeight);
      axWeight = xWeight;
      ayWeight = yWeight;
      xWeight = xWeight == 0 ? 1 : xWeight;
      yWeight = yWeight == 0 ? 1 : yWeight;

      // Get the avaible space for all non "final layouted" childs
      
      // Reset the space to the orginal avaible space
      availXSpace = orgXAvail;
      availYSpace = orgYAvail;

      // Sub the space of all final layouted childs
      bool final = true;
      for( it.toFirst(); it.current(); ++it ) {
        KLChild *child = it.current();
        if( child->layoutFinal && !child->hidden() ) {
          // Sub the size
          if( !a_horiz ) 
            availXSpace -= child->showInfo().xSize - child->minMaxSizes().minX;
          else
            availYSpace -= child->showInfo().ySize - child->minMaxSizes().minY;
        }
        else
          final = false;
      }

      // Check for "all childs are final layouted" special
      if( final ) {
        cont = false; // Dont contiune calc

        // Change the actX/Y, so that the childs will be centered.
        HASSERT( availXSpace );
        HASSERT( availYSpace );
        if( !a_horiz )
          actX += availXSpace / 2;
        else
          actY += availYSpace / 2;
        availXSpace = availYSpace = 0;  // Do diff to be spreaded
      }

    } // if : else

  } // while cont


  // Pass 1.3: Spread round diffs
  spreadRoundDiffs( availXSpace, availYSpace );

  // Pass 2: Show the objects
  return doShowChilds( actX, actY, compAvailXSpace, compAvailYSpace );
}

bool KLHVGroup::showSS(ulong actX, ulong actY, ulong compAvailXSpace, ulong compAvailYSpace )
{
  ulong availXSpace = compAvailXSpace;
  ulong availYSpace = compAvailYSpace;

  // Pass 1: Give the childs the size

  // Pass 1.1: Give the basic size
  ulong memberCount = countShowedChilds();
  HASSERT( memberCount );
  ulong xSize = availXSpace;
  ulong ySize = availYSpace;
  if( !a_horiz ) 
    xSize /= memberCount;
  else
    ySize /= memberCount;

  // Pass 1.2: Give the childs there space 
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    KLChild *child = it.current();
    child->showInfo().xSize = xSize;
    child->showInfo().ySize = ySize;
  } // for 

  if( !a_horiz ) 
    availXSpace -= xSize * memberCount;
  else
    availYSpace -= ySize * memberCount;

  // Pass 1.3: Spread round diffs
  spreadRoundDiffs( availXSpace, availYSpace );

  // Pass 2: Show the childs
  return doShowChilds( actX, actY, compAvailXSpace, compAvailYSpace );
}


bool KLHVGroup::doShowChilds(ulong actX, ulong actY, ulong xSize, ulong ySize )
{
  // Let`s show the childs
  KLChildListIt it(childs);
  for( ; it.current(); ++it ) {
    KLChild *child = it.current();

    if( !child->hidden() ) {
      // Set the cursor
      KLShowInfo &showInfo = child->showInfo();
      ulong compXSize = showInfo.xSize; // This is the complete size for this object
      ulong compYSize = showInfo.ySize; // If it is bigger than maximum, center the object in it
      showInfo.x = actX;
      showInfo.y = actY;
      if( !a_horiz ) {
        actX += showInfo.xSize + a_xSpacing;
        compYSize = ySize;
      }
      else {
        actY += showInfo.ySize + a_ySpacing;
        compXSize = xSize;
      }

      // Check minmax (normal layout should not need this, but 
      //               samesize layout needs this)
      KLMinMaxSizes &minMax = child->minMaxSizes();
      HASSERT( showInfo.xSize >= minMax.minX );
      HASSERT( showInfo.ySize >= minMax.minY );
      if( showInfo.xSize > minMax.maxX )
        showInfo.xSize = minMax.maxX;
      if( showInfo.ySize > minMax.maxY )
        showInfo.ySize = minMax.maxY;

      // Center the object in its room, if needed
      if( showInfo.xSize < compXSize ) 
        showInfo.x += (compXSize - showInfo.xSize) / 2;
      if( showInfo.ySize < compYSize ) 
        showInfo.y += (compYSize - showInfo.ySize) / 2;

      // Now show the control
      if( !child->klShow(&showInfo) ) {
        // What ?! Show failed -> Hide everything
        KLHVGroup::klHide();
        return false;
      }
    } // if !child->a_hidden
  } // for 
  return true;
}


void KLHVGroup::spreadRoundDiffs( ulong xDiff, ulong yDiff )
{
  // Spread round differences on the childs
  KLChildListIt it(childs);
  it.toLast(); ++it; // it.current() is now 0

  bool final = false;
  while( !a_horiz ? xDiff : yDiff ) {

    // Get next child
    if( it.current() )
      ++it;
    else {
      if( final ) 
        break;
      final = true;
      it.toFirst();
    }
    
    KLChild *child = it.current();
    if( !child || child->hidden() ) 
      continue;
    
    // Spread round diffs
    if( !a_horiz ) {
      if( child->showInfo().xSize + 1 <= child->minMaxSizes().maxX ) {
        child->showInfo().xSize++;
        xDiff--;
        final = false;
      }
    }
    else {
      if( child->showInfo().ySize + 1 <= child->minMaxSizes().maxY ) {
        child->showInfo().ySize++;
        yDiff--;
        final = false;
      }
    }

  } // while
}

KLHVGroup::KLHVGroup(bool horiz)
{
  setHoriz(horiz);
}

IMPLEMENT_KLMETA_STANDALONE(KLHVGroup,KLGroup,"H/V Group");

bool KLHVGroup::MyChildMeta::dumpObject(KLChild *_child, KLDumpDevice& dev) const
{
  QString helpStr;
  KLHVGroup *child = (KLHVGroup *)_child;
  dev.writeEntry("horiz",child->horiz() ? "1" : "0" );
  return true; 
}

void KLHVGroup::MyChildMeta::restoreObject(KLChild *_child, KLDumpDevice& dev) const
{
  QString helpStr;
  KLHVGroup *child = (KLHVGroup *)_child;
  child->setHoriz(atol(dev.readEntry("horiz",child->horiz() ? "1" : "0" )) == 1);
}

KLChild *KLHVGroup::MyChildMeta::createObjectEdit(KLChild *child,bool) const
{
  HASSERT(child->metaInherits("KLHVGroup"));
  KLHVGroup *group = (KLHVGroup *)child;
  KLHVGroupEdit *edit = new KLHVGroupEdit("horionzontal");
  edit->group = group;
  KLGridGroup *gridgroup = (KLGridGroup *)KLChildMeta::createObjectEdit(child,false);
  *gridgroup << new KLHVSpace()
             << edit;
  
  QObject::connect(edit,SIGNAL(toggled(bool)),edit,SLOT(onHoriz(bool)));
  edit->setChecked(group->horiz());

  return gridgroup;
}

void KLHVGroupEdit::onHoriz(bool horiz)
{
  group->setHoriz(horiz);
}


/////////////////////////////////////////////////////////////////////
// PageGroup
/////////////////////////////////////////////////////////////////////

KLPageGroup::KLPageGroup( CalcMode calcMode )
{
  a_calcMode = calcMode;
  a_actPage = 0;
}

KLPageGroup::CalcMode KLPageGroup::calcMode()
{
  return a_calcMode;
}

void KLPageGroup::setCalcMode( CalcMode calcMode )
{
  a_calcMode = calcMode;
  doRelayout(false);
}


bool KLPageGroup::klAskMinMax(KLMinMaxSizes *minMaxSizes)
{
  if( !KLGroup::klAskMinMax(minMaxSizes) ) 
    return false;

  KLChild *child = childs.at(a_actPage);
  HASSERT(child);
  ulong minX = 0, minY = 0;
  ulong maxX = 0, maxY = 0;
  ulong defX = 0, defY = 0;
  switch( a_calcMode ) {
    case MaxMax: {
      KLChild *currentchild;
      KLChildListIt it(childs);
      for( ; it.current(); ++it ) {
        currentchild = it.current();
        if( currentchild == child )
          currentchild->setHidden(false);
        else
          currentchild->setHidden(true);

        // Assert the right states
        HASSERT(currentchild->a_state == SI_SetupDone ||
                currentchild->a_state == SI_Showed );

        // Get the sizes of this childs
        KLMinMaxSizes &mms = currentchild->minMaxSizes();
        mms.init();
        currentchild->klDoAskMinMax(&mms);

        // Calc our sizes
        minX = KLMax(minX,mms.minX);
        minY = KLMax(minY,mms.minY);
        maxX = KLMax(maxX,mms.maxX);
        maxY = KLMax(maxY,mms.maxY);
      } // for
    }; break;
    case ActMax: {
      // Get the sizes of this childs
      KLMinMaxSizes &mms = child->minMaxSizes();
      mms.init();
      child->klDoAskMinMax(&mms);

      // Calc our sizes
      minX = mms.minX;
      minY = mms.minY;
      maxX = mms.maxX;
      maxY = mms.maxY;
    }; break; 
  } // switch calcMode


  //
  // Add borders
  //
  ulong addXSpace = a_spaceBorder ? a_xSpacing * 2 : 0;
  ulong addYSpace = a_spaceBorder ? a_ySpacing * 2 : 0;
  
  minMaxSizes->maxX += maxX + addXSpace;
  minMaxSizes->maxY += maxY + addYSpace;
  minMaxSizes->minX += minX + addXSpace;
  minMaxSizes->minY += minY + addYSpace;
  minMaxSizes->defX += defX + addXSpace;
  minMaxSizes->defY += defY + addYSpace;
  minMaxSizes->validate();

  return true;
}


bool KLPageGroup::klShow(KLShowInfo *showInfo)
{
  if( !KLGroup::klShow(showInfo) )
    return false;
  KLChild *child = childs.at(a_actPage);
  HASSERT(child);
  HASSERT(!child->hidden());
  KLShowInfo &si = child->showInfo();
  si = *showInfo;
  if( a_spaceBorder ) {
    si.xSize -= 2 * a_xSpacing;
    si.ySize -= 2 * a_ySpacing;
  }

  // Check MinMax
  KLMinMaxSizes &minMax = child->minMaxSizes();
  HASSERT( si.xSize >= minMax.minX );
  HASSERT( si.ySize >= minMax.minY );
  ulong compXSize = si.xSize;
  ulong compYSize = si.ySize;
  if( si.xSize > minMax.maxX ) 
    si.xSize = minMax.maxX;
  if( si.ySize > minMax.maxY ) 
    si.ySize = minMax.maxY;
  
  if( si.xSize < compXSize ) 
    si.x += (compXSize - si.xSize) / 2;
  if( si.ySize < compYSize ) 
    si.y += (compYSize - si.ySize) / 2;
  
  if( !child->klShow(&si) ){
    KLPageGroup::klHide();
    return false;
  }
  return true;
}

void KLPageGroup::setActPage(ulong page)
{
  if( page != a_actPage ) {
    HASSERT( page < childs.count() ) ;
    if( page >= childs.count() )
      return;
    a_actPage = page;
    doRelayout(false);
  }
}


ulong KLPageGroup::actPage()
{
  return a_actPage;
}

bool KLPageGroup::MyChildMeta::dumpObject(KLChild *, KLDumpDevice& ) const
{
  return true; 
}

void KLPageGroup::MyChildMeta::restoreObject(KLChild *, KLDumpDevice& ) const
{
}

KLChild *KLPageGroup::MyChildMeta::createObjectEdit(KLChild *child,bool) const
{
  // Not more (yet)
  return KLChildMeta::createObjectEdit(child,false);
}

IMPLEMENT_KLMETA_STANDALONE(KLPageGroup,KLGroup,"Page Group");

/////////////////////////////////////////////////////////////////////
// KLGridGroup
/////////////////////////////////////////////////////////////////////

void KLGridGroup::setRowGrid(bool row)
{
  if( row != a_rowGrid ) {
    a_rowGrid = row;
    doRelayout(false);
  }
}

void KLGridGroup::setRowColCount(ulong rowColCount)
{
  if( rowColCount != a_rowColCount ) {
    a_rowColCount = rowColCount;
    doRelayout(false);
  }
}

KLGridGroup::KLGridGroup(ulong rowColCount, bool rowGrid)
{
  a_rowGrid = rowGrid;
  a_rowColCount = rowColCount; 
  xList.setAutoDelete(true);
  yList.setAutoDelete(true);
}


void KLGridGroup::calcAndSetDim()
{
  // Calculate dimensions
  ulong count = countShowedChilds();
  HASSERT( count % a_rowColCount == 0 ); // Read the docs! 
  if( count % a_rowColCount ) 
    return;

  if( a_rowGrid )  {
    xDim = a_rowColCount;
    yDim = count / a_rowColCount;
  }
  else {
    xDim = count / a_rowColCount;
    yDim = a_rowColCount;
  }

  HASSERT( xDim && yDim ); // There must be elements in this group !!!
  
  // Set the demension (Create the minmaxcontainers)
  xList.clear();
  yList.clear();
  ulong i;
  // Create containter for X(rows)
  for( i = 0; i < xDim; i++ ) 
    xList.append(new KLSizeNode());

  // Create containter for X(rows)
  for( i = 0; i < yDim; i++ ) 
    yList.append(new KLSizeNode());

  return;
}


KLChild *KLGridGroup::getChild(ulong x, ulong y)
{
  ulong i;
  HASSERT( xDim && yDim );

  if( a_rowGrid ) 
    i = (y - 1) * xDim + x;
  else
    i = (x - 1) * yDim + y;

  
  KLChildListIt it(childs);
  for( ; i && it.current(); ++it ) {
    if( !it.current()->hidden() ) {
      i--;
      if( i == 0 )
        return it.current();
    }
  }

  return 0;
}


bool KLGridGroup::klAskMinMax(KLMinMaxSizes *minMaxSizes)
{
  // Call superclass
  if( !KLGroup::klAskMinMax(minMaxSizes) )
    return false;

  this->calcAndSetDim();

  // Pass 1: Let th elements calc there minmaxsizes.
  askMinMax_InitSizes();

  // Pass 2: Get minmax for this group
  if( a_sameSize ) 
    askMinMaxSS( minMaxSizes );
  else
    askMinMaxN( minMaxSizes );

  // Pass 3: Correct borders
  askMinMax_CorrectBorder( minMaxSizes );
  return true;
}

void KLGridGroup::askMinMax_InitSizes()
{
  // Get the minMaxSizes for the rows and cols
  ulong x, y;
  for( x = xDim; x; x-- ) {
    KLSizeNode *xNode = xList.at(x-1);
    for( y = yDim; y; y-- ) {
      // Get the child at (x,y) and the row and cols for this child
      KLChild *child  = getChild(x,y);
      KLSizeNode *yNode = yList.at(y-1);
      HASSERT( child && xNode && yNode );
      KLMinMaxSizes &mms = child->minMaxSizes();
      mms.init();
      child->klDoAskMinMax(&mms);
      
      // Sum the sizes
      xNode->minX = KLMax( xNode->minX, mms.minX );
      xNode->maxX = KLNonNullMin( xNode->maxX, mms.maxX );
      xNode->defX = KLMax( xNode->defX, mms.minX );
      yNode->minY = KLMax( yNode->minY, mms.minY );
      yNode->maxY = KLNonNullMin( yNode->maxY, mms.maxY );
      yNode->defY = KLMax( yNode->defY, mms.defY );
    } // for y
  } // for x
}

void KLGridGroup::askMinMaxN( KLMinMaxSizes *minMaxSizes )
{
  HASSERT( minMaxSizes );

  // Get the minmax for this group in normalmode
  ulong x, y;
  for( x = xDim; x; x-- ) {
    KLSizeNode *xNode = xList.at( x - 1 );
    HASSERT( xNode );
    xNode->validate();
    minMaxSizes->minX += a_xSpacing + xNode->minX;
    minMaxSizes->maxX += a_xSpacing + xNode->maxX;
    minMaxSizes->defX += a_xSpacing + xNode->defX;
  }
  for( y = yDim; y; y-- ) {
    KLSizeNode *yNode = yList.at( y - 1 );
    HASSERT( yNode );
    yNode->validate();
    minMaxSizes->minY += a_ySpacing + yNode->minY;
    minMaxSizes->maxY += a_ySpacing + yNode->maxY;
    minMaxSizes->defY += a_ySpacing + yNode->defY;
  }
}

void KLGridGroup::askMinMaxSS( KLMinMaxSizes *minMaxSizes )
{
  HASSERT( minMaxSizes );
  // Get the minmax for this group in smamesize mode
  ulong x, y;
  ulong minX = 0, maxX = 0, defX = 0;
  ulong minY = 0, maxY = 0, defY = 0;
  for( x = xDim; x; x-- ) {
    KLSizeNode *xNode = xList.at( x -1 );
    HASSERT( xNode );
    xNode->validate();
    minX = KLMax( minX, xNode->minX );
    maxX = KLMax( maxX, xNode->maxX );
    defX = KLMax( defX, xNode->defX );
  } // for x
  for( y = yDim; y; y-- ) {
    KLSizeNode *yNode = yList.at( y -1 );
    HASSERT( yNode );
    yNode->validate();
    minY = KLMax( minY, yNode->minY );
    maxY = KLMax( maxY, yNode->maxY );
    defY = KLMax( defY, yNode->defY );
  } // for y

  // Set the minmax size
  minMaxSizes->minX += ( minX + a_xSpacing ) * xDim;
  minMaxSizes->maxX += ( maxX + a_xSpacing ) * xDim;
  minMaxSizes->defX += ( defX + a_xSpacing ) * xDim;
  minMaxSizes->minY += ( minY + a_ySpacing ) * yDim;
  minMaxSizes->maxY += ( maxY + a_ySpacing ) * yDim;
  minMaxSizes->defY += ( defY + a_ySpacing ) * yDim;
}


void KLGridGroup::askMinMax_CorrectBorder( KLMinMaxSizes *minMaxSizes )
{
  // Weve just added one spacing to much. For spaceborder just add on,
  // for no spaceborder, just sub one
  ulong xAdd;
  ulong yAdd;

  if( a_spaceBorder ) {
    xAdd = a_xSpacing;
    yAdd = a_ySpacing;
  }
  else {
    xAdd = -a_xSpacing;
    yAdd = -a_ySpacing;
  }

  // Add the border
  minMaxSizes->minX += xAdd;
  minMaxSizes->maxX += xAdd;
  minMaxSizes->defX += xAdd;
  minMaxSizes->minY += yAdd;
  minMaxSizes->maxY += yAdd;
  minMaxSizes->defY += yAdd;
}

bool KLGridGroup::klShow(KLShowInfo *showInfo)
{
  if( !KLGroup::klShow( showInfo ) )
    return false;

  ulong availXSpace = showInfo->xSize - ( a_spaceBorder ? a_xSpacing * 2: 0 );
  ulong availYSpace = showInfo->ySize - ( a_spaceBorder ? a_ySpacing * 2: 0 );
  availXSpace = availXSpace - ( ( xDim - 1 ) * a_xSpacing );
  availYSpace = availYSpace - ( ( yDim - 1 ) * a_ySpacing );

  // Some asserts
  HASSERT( availXSpace && availXSpace <= KLMaxSize );
  HASSERT( availYSpace && availYSpace <= KLMaxSize );

  // Get the cursor to begin with showing controls
  ulong actX = showInfo->x + ( a_spaceBorder ? a_xSpacing : 0 );
  ulong actY = showInfo->y + ( a_spaceBorder ? a_ySpacing : 0 );

  if( a_sameSize ) 
    showSS(availXSpace, availYSpace );
  else
    showN(availXSpace, availYSpace );

  return doShowChilds( showInfo, actX, actY );
}

void KLGridGroup::showSS(ulong availXSpace, ulong availYSpace )
{
  // Pass 1: Give the obects the size
  // Pass 1.1: Give the size
  ulong xSize = availXSpace / xDim;
  ulong ySize = availYSpace / yDim;

  // Write the sizes into the row/col nodes
  ulong i = 0;
  for( i = xDim; i; i-- ) {
    KLSizeNode *xNode = xList.at( i - 1 );
    HASSERT( xNode );
    xNode->size = xSize;
  }
  for( i = yDim; i; i-- ) {
    KLSizeNode *yNode = yList.at( i - 1 );
    HASSERT( yNode );
    yNode->size = ySize;
  }

  availXSpace -= xSize * xDim;
  availYSpace -= ySize * yDim;

  // Pass 1.2: Spread round diffs
  spreadRoundDiffs( availXSpace, availYSpace );
}

void KLGridGroup::showN( ulong availXSpace, ulong availYSpace )
{
  // Pass 1: Give the objects the size
  // Pass 1.1: Sub the minimum sizes of the object from
  // the avail size
  ulong i;
  for( i = xDim; i; i-- ) {
    KLSizeNode *xNode = xList.at( i - 1 );
    HASSERT( xNode );
    availXSpace -= xNode->minX;
    xNode->size = xNode->minX;
  }
  for( i = yDim; i; i-- ) {
    KLSizeNode *yNode = yList.at( i - 1 );
    HASSERT( yNode );
    availYSpace -= yNode->minY;
    yNode->size = yNode->minY;
  }

  // Pass 1.2: Spread the rest of the avail space
  ulong xSize = availXSpace / xDim;
  ulong ySize = availYSpace / yDim;
  for( i = xDim; i; i-- ) {
    KLSizeNode *xNode = xList.at( i - 1 );
    HASSERT( xNode );
    ulong size = xSize;
    if( xNode->size + size > xNode->maxX )
      size = xNode->maxX - xNode->size;
    availXSpace -= size;
    xNode->size += size;
  }
  for( i = yDim; i; i-- ) {
    KLSizeNode *yNode = yList.at( i - 1 );
    HASSERT( yNode );
    ulong size = ySize;
    if( yNode->size + size > yNode->maxY )
      size = yNode->maxY - yNode->size;
    availYSpace -= size;
    yNode->size += size;
  }

  // Pass 1.3: Spread round diffs
  spreadRoundDiffs( availXSpace, availYSpace );
}

void KLGridGroup::spreadRoundDiffs( ulong availXSpace, ulong availYSpace )
{
  // Spread round differences on the childs

  // First, spread the avail x space
  ulong x = 0, y = 0;
  bool final = false;
  while( availXSpace ) {

    // Get next child to give the space
    if( !x ) {
      if( final ) 
        break;
      final = true;
      x = 1;
    }
    else {
      x++;
      x = x > xDim ? 0 : x;
      if( !x ) 
        continue;
    }

    KLSizeNode *xNode = xList.at(x - 1);
    HASSERT( xNode );

    // Spread the round diffs
    if( xNode->size + 1 <= xNode->maxX ) {
      xNode->size++;
      availXSpace--;
      final = false;
    }
  } // for

  final = false;
  while( availYSpace ) {

    // Get next child to give the space
    if( !y ) {
      if( final ) 
        break;
      final = true;
      y = 1;
    }
    else {
      y++;
      y = y > yDim ? 0 : y;
      if( !y ) 
        continue;
    }

    KLSizeNode *yNode = yList.at(y - 1);
    HASSERT( yNode );

    // Spread the round diffs
    if( yNode->size + 1 <= yNode->maxY ) {
      yNode->size++;
      availYSpace--;
      final = false;
    }
  } // for
}

bool KLGridGroup::doShowChilds( KLShowInfo *showInfo, ulong actX, ulong actY )
{
  // Lets show the childs
  ulong orgY = actY;
  ulong i, j;
  for( i = 1; i <= xDim; i++ ) {
    KLSizeNode *xNode = xList.at( i - 1 );
    HASSERT( xNode );
    ulong compXSize = xNode->size;
    actY = orgY;

    for( j = 1; j <= yDim; j++ ) {
      KLSizeNode *yNode = yList.at( j - 1 );
      HASSERT( yNode );
      KLChild *child = getChild( i, j );
      HASSERT( child );
      ulong compYSize = yNode->size;

      // Set the cursor
      KLShowInfo &si = child->showInfo();
      si = *showInfo;
      si.x = actX;
      si.y = actY;
      si.xSize = compXSize;
      si.ySize = compYSize;

      // Check minmax (normal layout should not need this, but 
      //               samesize layout needs this)
      KLMinMaxSizes &minMax = child->minMaxSizes();
      HASSERT( si.xSize >= minMax.minX );
      HASSERT( si.ySize >= minMax.minY );
      if( si.xSize > minMax.maxX )
        si.xSize = minMax.maxX;
      if( si.ySize > minMax.maxY )
        si.ySize = minMax.maxY;

      // Center the object in its room, if needed
      if( si.xSize < compXSize ) 
        si.x += (compXSize - si.xSize) / 2;
      if( si.ySize < compYSize ) 
        si.y += (compYSize - si.ySize) / 2;

      // Now show the control
      if( !child->klShow(&si) ) {
        // What ?! Show failed -> Hide everything
        KLGridGroup::klHide();
        return false;
      }
      actY += a_ySpacing + yNode->size;
    } // for j

    actX += a_xSpacing + xNode->size;
  } // for i 
  return true;
}

void KLGridGroup::validateGrid()
{
  // Fill up
  ulong count = childs.count() % rowColCount();
  while( count-- ) 
    addChild( new KLReplaceMe() );

  // Remove Replacemes wich are to much
  ulong repatend = 0;
  while( true ) {
    if( childs.at(childs.count() - 1 - repatend)->metaIsA("KLReplaceMe") ) 
      repatend++;
    else 
      return;
    if( (repatend == rowColCount()) && (childs.count() != repatend) ) {
      // Only remove them, if the group gets not empty
      while( repatend-- ) {
        KLChild *child = childs.at(childs.count() - 1);
        remChild(child);
      }
    }
  } // while
}


bool KLGridGroup::MyChildMeta::dumpObject(KLChild *_child, KLDumpDevice& dev) const
{
  QString helpStr;
  KLGridGroup *child = (KLGridGroup *)_child;
  dev.writeEntry("rowgrid",child->rowGrid() ? "1" : "0" );
  dev.writeEntry("rowcolcount",helpStr.setNum(child->rowColCount()));
  return true; 
}

void KLGridGroup::MyChildMeta::restoreObject(KLChild *_child, KLDumpDevice& dev) const
{
  QString helpStr;
  KLGridGroup *child = (KLGridGroup *)_child;
  child->setRowGrid(atol(dev.readEntry("rowgrid",child->rowGrid() ? "1" : "0" )) == 1);
  child->setRowColCount(atol(dev.readEntry("rowcolcount",helpStr.setNum(child->rowColCount()))));
}

KLChild *KLGridGroup::MyChildMeta::createObjectEdit(KLChild *child,bool) const
{
  HASSERT(child->metaInherits("KLGridGroup"));
  KLGridGroup *group = (KLGridGroup *)child;
  KLGridGroupEdit *edit = new KLGridGroupEdit("row grid");
  edit->group = group;
  KLLineEdit *rowcol;
  KLGridGroup *gridgroup = (KLGridGroup *)KLChildMeta::createObjectEdit(child,false);
  *gridgroup << new KLHVSpace()
             << edit
             << (*(new KLLabel("Row/Col count")) << ::setWeight << 1)
             << (rowcol = new KLLineEdit());
  
  QObject::connect(edit,SIGNAL(toggled(bool)),edit,SLOT(onRowCol(bool)));
  QObject::connect(rowcol,SIGNAL(textChanged(const char*)),edit,SLOT(onRowColCount(const char *)));

  QString helpStr;
  edit->setChecked(group->rowGrid());
  rowcol->setText(helpStr.setNum(group->rowColCount()));

  return gridgroup;
}

void KLGridGroupEdit::onRowGrid(bool rowgrid)
{
  group->setRowGrid(rowgrid);
}

void KLGridGroupEdit::onRowColCount(const char *text)
{
  group->setRowColCount(atol(text));
}

IMPLEMENT_KLMETA_STANDALONE(KLGridGroup,KLGroup,"Grid Group");


#include "klgroup.moc"
// -- EOF -
