/*
 * This file is part of telepathy-presence-applet
 *
 * Copyright (C) 2011 Francesco Nwokeka <francesco.nwokeka@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

import Qt 4.7
import org.kde.plasma.core 0.1 as PlasmaCore
import org.kde.plasma.graphicswidgets 0.1 as PlasmaWidgets

Item {
    id: panel;

    // fetch info from data engine (dario's presence applet data-engine)
    PlasmaCore.DataSource {
        id: presenceSource;
        engine: "presence";
        interval: 0;

        onSourceAdded: {
            // don't add global presence to account list
            if (source != "GlobalPresence") {
                connectSource(source);
            }
        }

        Component.onCompleted: {
            connectedSources = sources;
        }
    }

    Component.onCompleted: {
        checkPresenceDataEngine();
    }

    ListView {
        id: accountList;
        interactive: false;

        anchors {
            left: parent.left;
            top: parent.top;
            bottom: parent.bottom;
            right: scrollBar.left;
        }

        clip: true;

        model: PlasmaCore.DataModel {
            dataSource: presenceSource;
        }

        onHeightChanged: {
            // adapt scrollbar
            calculateScrollBarValues();
        }

        delegate: AccountDelegate {
            // set info retrieved from DataEngine
            avatarSource: AccountAvatar;
            accountIconName: AccountIcon;
            displayName: DisplayName;
            enabled: Enabled;
            nickName: Nickname;
            presenceType: PresenceType;
            presenceStatus: PresenceStatus;
            presenceTypeId: PresenceTypeID;

            onPresenceChanged: {
                // Do the dataengine magic!
                var found = false;

                for (var i = 0; i < presenceSource.sources.length && !found; ++i) {
                    var sourceName = presenceSource.sources[i];
                    if (presenceSource.data[sourceName]["DisplayName"] == displayName) {
                        // change status
                        var operation = presenceSource.serviceForSource(sourceName).operationDescription("setPresence");
                        operation.type_id = presenceTypeId; // set new id
                        presenceSource.serviceForSource(sourceName).startOperationCall(operation);
                        found = true;
                    }
                }
            }

            onAvatarChanged: {
                var found = false;

                for (var i = 0; i < presenceSource.sources.length && !found; ++i) {
                    var sourceName = presenceSource.sources[i];
                    if (presenceSource.data[sourceName]["DisplayName"] == displayName) {
                        // change avatar
                        var operation = presenceSource.serviceForSource(sourceName).operationDescription("setAvatar");
                        operation.url = "file://" + avatarSource; // set new avatar via url taken from file dialog
                        presenceSource.serviceForSource(sourceName).startOperationCall(operation);
                        found = true;
                    }
                }
            }
        }
    }

    ErrorPanel {
        id: errorPanel;
        errorStr: i18n("The presence data engine is missing. Please install the 'telepathy-presence-dataengine' to use this plasmoid");

        anchors {
            left: parent.left;
            top: parent.top;
            bottom: parent.bottom;
            right: scrollBar.left;
        }
    }

    ScrollBar {
        id: scrollBar;

        anchors {
            right: parent.right;
            top: parent.top;
            bottom: parent.bottom;
            leftMargin: 5;
            topMargin: 5;
            bottomMargin: 5;
        }

        Component.onCompleted: {
            calculateScrollBarValues();
        }

        onWrapperValueChanged: {
            accountList.contentY = newValue;
        }
    }

    // panel states
    states: [
        State {
            name: "valid";
            PropertyChanges {
                target: accountList;
                opacity: 1;
            }
            PropertyChanges {
                target: errorPanel;
                opacity: 0;
            }
        },
        State {
            name: "invalid";
            PropertyChanges {
                target: accountList;
                opacity: 0;
            }
            PropertyChanges {
                target: errorPanel;
                opacity: 1;
            }
        }
    ]

    transitions: [
        Transition {
            NumberAnimation {
                properties: "opacity";
                easing.type: Easing.Linear;
                duration: 500;
            }
        }
    ]


    // set status message from global config to all accounts
    function setGlobalStatusMessage(currentPresenceId, globalStatus)
    {
        for (var i = 0; i < presenceSource.sources.length; ++i) {
            var sourceName = presenceSource.sources[i];
            var operation = presenceSource.serviceForSource(sourceName).operationDescription("setPresence");
            operation.type_id = currentPresenceId;

            // check if presence message isn't the default "Set personal status here"
            if (globalStatus == "Set personal status here") {
                operation.status_message = "";
            } else {
                operation.status_message = globalStatus;
            }

            presenceSource.serviceForSource(sourceName).startOperationCall(operation);
        }
    }

    // set nick from global config to all accounts
    function setGlobalNick(globalNick)
    {
        for (var i = 0; i < presenceSource.sources.length; ++i) {
            var sourceName = presenceSource.sources[i];
            var operation = presenceSource.serviceForSource(sourceName).operationDescription("setNickname");
            operation.nickname = globalNick;
            presenceSource.serviceForSource(sourceName).startOperationCall(operation);
        }
    }

    // set avatar from global config to all accounts
    function setGlobalAvatar(globalAvatar)
    {
        for (var i = 0; i < presenceSource.sources.length; ++i) {
            var sourceName = presenceSource.sources[i];
            var operation = presenceSource.serviceForSource(sourceName).operationDescription("setAvatar");
            operation.url = globalAvatar;
            presenceSource.serviceForSource(sourceName).startOperationCall(operation);
        }
    }

    // set presence from global config to all accounts
    function setGlobalPresence(globalPresence)
    {
        var found = false;

        for (var i = 0; i < presenceSource.sources.length && !found; ++i) {
            if (presenceSource.sources[i] == "GlobalPresence") {
                var operation = presenceSource.serviceForSource("GlobalPresence").operationDescription("setGlobalPresence");
                operation.presence = globalPresence;
                presenceSource.serviceForSource("GlobalPresence").startOperationCall(operation);
                found = true;
            }
        }
    }

    // calculate scrollbar values
    function calculateScrollBarValues()
    {
        scrollBar.setMinimum(0);
        scrollBar.setPageStep(accountList.height);

        if (accountList.contentHeight != -1) {
            scrollBar.setMaximum(accountList.contentHeight - accountList.height);

            if (accountList.height >= accountList.contentHeight) {
                scrollBar.state = "hidden";
            } else {
                scrollBar.state = "visible";
            }
        }
    }

    // resets and checks the presence of the plasmoid dataengine
    function checkPresenceDataEngine()
    {
        if (presenceSource.valid) {
            panel.state = "valid";      // show list
        } else {
            panel.state = "invalid";    // show error pane
        }
    }
}
