/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * This file defines classes SKGObjectBase.
 *
 * @author Stephane MANKOWSKI / Guillaume DE BURE
 */
#include "skgobjectbase.h"
#include "skgtraces.h"
#include "skgdocument.h"
#include "skgnamedobject.h"

#include <klocale.h>

#include <QSqlDatabase>
#include <kstringhandler.h>
/**
 * This private class of SKGObjectBase
 */
class SKGObjectBasePrivate
{
public:
    /**
     * internal id
     */
    int id;

    /**
     * internal table
     */
    QString table;

    /**
     * internal document
     */
    SKGDocument* document;

    /**
     * internal attributes
     */
    SKGQStringQStringMap attributes;

    /**
     * objects to save
     */
    SKGObjectBase::SKGListSKGObjectBase objects;
};

SKGObjectBase::SKGObjectBase(SKGDocument* iDocument, const QString& iTable, int iID)
    : QObject(), d(new SKGObjectBasePrivate)
{
    d->id = iID;
    d->table = iTable;
    d->document = iDocument;
    if (d->id != 0) load();
}

SKGObjectBase::~SKGObjectBase()
{
    delete d;
}

SKGObjectBase::SKGObjectBase(const SKGObjectBase& iObject)
    : QObject(), d(new SKGObjectBasePrivate)
{
    copyFrom(iObject);
}

const SKGObjectBase& SKGObjectBase::operator= (const SKGObjectBase& iObject)
{
    copyFrom(iObject);
    return *this;
}

bool SKGObjectBase::operator==(const SKGObjectBase& iObject) const
{
    return (getUniqueID() == iObject.getUniqueID());
}

bool SKGObjectBase::operator!=(const SKGObjectBase& iObject) const
{
    return !(*this == iObject);
}

bool SKGObjectBase::operator<(const SKGObjectBase& iObject) const
{
    double d1 = SKGServices::stringToDouble(getAttribute("f_sortorder"));
    double d2 = SKGServices::stringToDouble(iObject.getAttribute("f_sortorder"));
    return (d1 < d2);
}

bool SKGObjectBase::operator>(const SKGObjectBase& iObject) const
{
    double d1 = SKGServices::stringToDouble(getAttribute("f_sortorder"));
    double d2 = SKGServices::stringToDouble(iObject.getAttribute("f_sortorder"));
    return (d1 > d2);
}

QString SKGObjectBase::getUniqueID() const
{
    return SKGServices::intToString(d->id) % '-' % getRealTable();
}

int SKGObjectBase::getID() const
{
    return d->id;
}

QString SKGObjectBase::getDisplayName() const
{
    QString output;

    SKGStringListList result;
    QString wc = getWhereclauseId();
    if (wc.isEmpty()) wc = "id=" % SKGServices::intToString(d->id);
    QString sql = "SELECT t_displayname FROM v_" % getRealTable() % "_displayname WHERE " % wc;
    if (getDocument()) getDocument()->executeSelectSqliteOrder(sql, result);
    if (result.count() == 2) output = result.at(1).at(0);

    return output;
}

void SKGObjectBase::copyFrom(const SKGObjectBase& iObject)
{
    d->id = iObject.d->id;
    d->table = iObject.d->table;
    d->document = iObject.d->document;
    d->attributes = iObject.d->attributes;
}

SKGObjectBase SKGObjectBase::cloneInto(SKGDocument* iDocument)
{
    SKGDocument* targetDocument = iDocument;
    if (targetDocument == NULL) targetDocument = d->document;

    SKGObjectBase output;
    output.copyFrom(*this);
    output.d->id = 0;
    output.d->document = targetDocument;

    return output;
}

SKGError SKGObjectBase::resetID()
{
    SKGError err;
    d->id = 0;
    return err;
}

QString SKGObjectBase::getTable() const
{
    return d->table;
}

QString SKGObjectBase::getRealTable() const
{
    return SKGServices::getRealTable(d->table);
}

SKGDocument* SKGObjectBase::getDocument() const
{
    return d->document;
}

SKGQStringQStringMap SKGObjectBase::getAttributes() const
{
    return d->attributes;
}

int SKGObjectBase::getNbAttributes() const
{
    return d->attributes.count();
}

SKGError SKGObjectBase::setAttributes(const QStringList& iNames, const QStringList& iValues)
{
    SKGError err;
    int nb = iNames.size();
    for (int i = 0; !err && i < nb; ++i) {
        QString att = iNames.at(i);
        QString val = iValues.at(i);

        if (att != "id") err = setAttribute(att, val);
        else  d->id = SKGServices::stringToInt(val);
    }
    return err;
}

SKGError SKGObjectBase::setAttribute(const QString& iName, const QString& iValue)
{
    SKGError err;
    if (iValue != NOUPDATE) {
        QString val = iValue;

        //Case modification on pointed document
        if (this->getRealTable() == this->getTable() && iName != iName.toLower()) {
            QString realAttribute = d->document->getRealAttribute(iName);
            if (!realAttribute.isEmpty()) {
                QStringList l = realAttribute.split('.');
                if (l.size() == 3) {
                    SKGObjectBase obj;
                    QString refId = getAttribute(l.at(1));
                    if (!refId.isEmpty()) {
                        err = getDocument()->getObject("v_" % l.at(0), "id=" % refId, obj);
                        if (!err) {
                            err = obj.setAttribute(l.at(2), iValue);
                            d->objects.push_back(obj);
                        }
                    }
                }
            }
        }
        //Case modificator
        else if (iValue.startsWith(QLatin1String("="))) {
            QString op = iValue.right(iValue.length() - 1).toLower();
            val = d->attributes[iName];
            if (op == i18nc("Key word to modify a string into a field", "lower")) val = val.toLower();
            else if (op == i18nc("Key word to modify a string into a field", "upper")) val = val.toUpper();
            else if (op == i18nc("Key word to modify a string into a field", "capwords")) val = KStringHandler::capwords(val);
            else if (op == i18nc("Key word to modify a string into a field", "capitalize")) val = val.left(1).toUpper() % val.right(val.length() - 1).toLower();
            else if (op == i18nc("Key word to modify a string into a field", "trim")) val = val.trimmed();
            else val = iValue;
        }
        d->attributes[iName] = val;
    }
    return err;
}

QString SKGObjectBase::getAttribute(const QString& iName) const
{
    QString output;
    if (d->attributes.contains(iName)) {
        output = d->attributes[iName];
    } else if (iName == "id") {
        output = SKGServices::intToString(getID());
    } else {
        //Is the iName a number ?
        bool ok;
        int pos = iName.toInt(&ok);
        if (ok) {
            //What is the key corresponding to this name ?
            QStringList keys = d->attributes.keys();
            if (pos >= 0 && pos < keys.count()) output = d->attributes[keys[pos]];
        }
    }

    return output;
}

bool SKGObjectBase::exist() const
{
    SKGTRACEIN(20, "SKGObjectBase::exist");

    SKGStringListList result;
    QString wc = getWhereclauseId();
    if (wc.isEmpty() && d->id) wc = "id=" % SKGServices::intToString(d->id);
    if (wc.isEmpty()) return false;

    QString sql = "SELECT count(1) FROM " % d->table % " WHERE " % wc;
    if (getDocument()) getDocument()->executeSelectSqliteOrder(sql, result);
    return (result.size() >= 2 && result.at(1).at(0) != "0");
}

SKGError SKGObjectBase::load()
{
    SKGError err;
    SKGTRACEINRC(20, "SKGObjectBase::load", err);

    if (getDocument() && !getTable().isEmpty()) {
        //Prepare where clause
        QString wc = getWhereclauseId();
        if (wc.isEmpty()) wc = "id=" % SKGServices::intToString(d->id);

        //Execute sql order
        SKGStringListList result;
        err = getDocument()->executeSelectSqliteOrder("SELECT * FROM " % d->table % " WHERE " % wc, result);
        if (!err) {
            int size = result.size();
            if (size == 1)  err = SKGError(ERR_INVALIDARG, i18nc("Error message: Could not load something because it is not in the database", "Load of '%1' with '%2' failed because it was not found in the database", d->table, wc));
            else if (size != 2)  err = SKGError(ERR_INVALIDARG, i18np("Load of '%2' with '%3' failed because of bad size of result (found one object)",
                                                    "Load of '%2' with '%3' failed because of bad size of result (found %1 objects)",
                                                    size - 1, d->table, wc));
            else {
                SKGStringListListIterator itrow = result.begin();
                QStringList columns = *(itrow);
                ++itrow;
                QStringList values = *(itrow);
                err = setAttributes(columns, values);
            }
        }
    }

    return err;
}

QString SKGObjectBase::getWhereclauseId() const
{
    int id = getID();
    if (id != 0) return "id=" % SKGServices::intToString(id);
    return "";
}

SKGError SKGObjectBase::save(bool iInsertOrUpdate, bool iReloadAfterSave)
{
    SKGError err;
    SKGTRACEINRC(20, "SKGObjectBase::save", err);

    if (!d->document) err = SKGError(ERR_POINTER, i18nc("Error message", "Operation impossible because the document is missing"));
    else {
        //Save linking objects
        int nb = d->objects.count();
        for (int i = 0; !err && i < nb; ++i) {
            SKGObjectBase ref = d->objects.at(i);
            err = ref.save(iInsertOrUpdate, iReloadAfterSave);
        }

        //Check if we are in a transaction
        if (!err) err = d->document->checkExistingTransaction();
        if (!err) {
            //Table to use
            QString tablename = getRealTable();

            //Build order
            QString part1Insert;
            QString part2Insert;
            QString partUpdate;

            SKGQStringQStringMapIterator it;
            for (it = d->attributes.begin() ; it != d->attributes.end(); ++it) {
                QString att = SKGServices::stringToSqlString(it.key());
                QString attlower = att.toLower();
                if (att.length() > 2 && att == attlower) { //We must ignore attributes coming from views
                    QString value = '\'' % SKGServices::stringToSqlString(it.value()) % '\'';

                    if (!part1Insert.isEmpty()) {
                        part1Insert.append(',');
                        part2Insert.append(',');
                        partUpdate.append(',');
                    }
                    //Attribute
                    part1Insert.append('\'' % att % '\'');

                    //Value
                    part2Insert.append(value);

                    //Attribute=Value for update
                    partUpdate.append(att % '=' % value);
                }
            }

            //We try an Insert
            if (d->id == 0) {
                //We have to try un insert
                err = getDocument()->executeSqliteOrder("INSERT INTO " % tablename % " (" % part1Insert % ") VALUES (" % part2Insert % ')', &(d->id));
            } else {
                //We must try an update
                err = SKGError(ERR_ABORT, ""); //Just to go in UPDATE code
            }

            if (err && iInsertOrUpdate) {
                //INSERT failed, could we try an update ?
                QString wc = this->getWhereclauseId();
                if (!wc.isEmpty()) {
                    //Yes ==> Update
                    err = getDocument()->executeSqliteOrder("UPDATE " % tablename % " SET " % partUpdate % " WHERE " % wc);
                }
            }
        }
    }

    //Reload object is updated
    if (!err && iReloadAfterSave) {
        //The object has been updated ==>load
        err = load();
    }

    return err;
}

SKGError SKGObjectBase::remove(bool iSendMessage, bool iForce) const
{
    SKGError err;
    SKGTRACEINRC(20, "SKGObjectBase::remove", err);
    if (!d->document) err = SKGError(ERR_POINTER, i18nc("Error message", "Operation impossible because the document is missing"));
    else {
        //Check if we are in a transaction
        err = d->document->checkExistingTransaction();

        //delete order
        QString viewForDelete = QString("v_") % getRealTable() % "_delete";

        //Check if the delete view exist
        SKGStringListList temporaryResult;
        d->document->executeSelectSqliteOrder("PRAGMA table_info( " % viewForDelete % " );", temporaryResult);
        if (!iForce && temporaryResult.count() > 1) { //At least one attribute
            //Delete view exists, check if the delete is authorized
            err = d->document->executeSelectSqliteOrder("SELECT t_delete_message FROM " % viewForDelete % " WHERE id=" % SKGServices::intToString(d->id), temporaryResult);
            if (!err) {
                QString msg;
                if (temporaryResult.count() > 1) msg = temporaryResult.at(1).at(0);
                // Should the string below be translated ??? It contains no word
                if (!msg.isEmpty()) err = SKGError(ERR_FORCEABLE, i18nc("Error message for an object", "'%1': %2", getDisplayName(), msg));
            }
        }

        QString displayname = getDisplayName(); //Must be done before the delete order
        if (!err) err = d->document->executeSqliteOrder("DELETE FROM " % getRealTable() % " WHERE id=" % SKGServices::intToString(d->id));
        if (iSendMessage && !err && !displayname.isEmpty())
            err = d->document->sendMessage(i18nc("An information to the user that something was deleted", "'%1' has been deleted", displayname), false);
    }

    return err;
}

SKGError SKGObjectBase::dump()
{
    SKGError err;

    //dump
    SKGTRACE << "=== START DUMP [" << getUniqueID() << "]===" << endl;
    SKGQStringQStringMapIterator it;
    for (it = d->attributes.begin() ; it != d->attributes.end(); ++it)
        SKGTRACE << it.key() << "=[" << it.value() << ']' << endl;
    SKGTRACE << "=== END DUMP [" << getUniqueID() << "]===" << endl;
    return err;
}

QStringList SKGObjectBase::getProperties() const
{
    return !getDocument() ? QStringList() : getDocument()->getParameters(getUniqueID());
}

QString SKGObjectBase::getProperty(const QString& iName) const
{
    return !getDocument() ? QString() : getDocument()->getParameter(iName, getUniqueID());
}

QVariant SKGObjectBase::getPropertyBlob(const QString& iName) const
{
    return !getDocument() ? QVariant() : getDocument()->getParameterBlob(iName, getUniqueID());
}

SKGError SKGObjectBase::setProperty(const QString& iName, const QString& iValue, const QString& iFileName, SKGPropertyObject* oObjectCreated) const
{
    return !getDocument() ? SKGError() : getDocument()->setParameter(iName, iValue, iFileName, getUniqueID(), oObjectCreated);
}

SKGError SKGObjectBase::setProperty(const QString& iName, const QString& iValue, const QVariant& iBlob, SKGPropertyObject* oObjectCreated) const
{
    return!getDocument() ? SKGError() :  getDocument()->setParameter(iName, iValue, iBlob, getUniqueID(), oObjectCreated);

}

#include "skgobjectbase.moc"

