/*
Copyright 2012 Aurélien Gâteau <agateau@kde.org>

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) version 3, or any
later version accepted by the membership of KDE e.V. (or its
successor approved by the membership of KDE e.V.), which shall
act as a proxy defined in Section 6 of version 3 of the license.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library.  If not, see <http://www.gnu.org/licenses/>.
*/
// Local
#include <changenotifier.h>
#include <pathmodel.h>
#include <installedappsmodel.h>
#include <installedappsconfigurationwidget.h>
#include <sourceregistry.h>

// Qt
#include <QIcon>
#include <QAction>
#include <QTimer>

// KDE
#include <KDebug>
#include <KLocale>
#include <kmacroexpander.h>
#include <KRun>
#include <KService>
#include <KServiceTypeTrader>
#include <KSycocaEntry>

#include <Plasma/RunnerManager>

namespace Homerun {

static const char *SOURCE_ID = "InstalledApps";

//- AbstractNode ---------------------------------------------------------------
AbstractNode::~AbstractNode()
{
}

bool AbstractNode::lessThan(AbstractNode *n1, AbstractNode *n2)
{
    Q_ASSERT(n1);
    Q_ASSERT(n2);
    return n1->m_sortKey < n2->m_sortKey;
}

//- GroupNode ------------------------------------------------------------------
GroupNode::GroupNode(KServiceGroup::Ptr group, InstalledAppsModel *model)
: m_model(model)
{
    m_icon = group->icon();
    m_name = group->caption();
    m_entryPath = group->entryPath();
    m_sortKey = m_name.toLower();
}

bool GroupNode::trigger()
{
    QVariantMap args;
    args.insert("entryPath", m_entryPath);
    m_model->openSourceRequested(SOURCE_ID, args);
    return false;
}

//- AppNode --------------------------------------------------------------------
AppNode::AppNode(KService::Ptr service)
: m_service(service)
{
    m_icon = service->icon();
    m_name = service->name();
    m_service = service;
    m_sortKey = m_name.toLower();
}

bool AppNode::trigger()
{
    return KRun::run(*m_service, KUrl::List(), 0);
}

QString AppNode::favoriteId() const
{
    return QString("app:") + m_service->storageId();
}

//- InstallerNode --------------------------------------------------------------
InstallerNode::InstallerNode(KServiceGroup::Ptr group, KService::Ptr installerService)
: m_group(group)
, m_service(installerService)
{
    m_icon = m_service->icon();
    m_name = m_service->name();
}

bool InstallerNode::trigger()
{
    QHash<QString, QString> map;
    QString category = m_group->entryPath();
    if (category.endsWith('/')) {
        category.truncate(category.length() - 1);
    }
    map.insert("category", category);

    QString command = KMacroExpander::expandMacros(m_service->exec(), map, '@');
    return KRun::run(command, KUrl::List(), 0, m_service->name(), m_service->icon());
}

//- InstalledAppsModel ------------------------------------------------------------
InstalledAppsModel::InstalledAppsModel(const QString &entryPath, const QString &installer, QObject *parent)
: QAbstractListModel(parent)
, m_entryPath(entryPath)
, m_pathModel(new PathModel(this))
, m_installer(installer)
{
    QHash<int, QByteArray> roles;
    roles.insert(Qt::DisplayRole, "display");
    roles.insert(Qt::DecorationRole, "decoration");
    roles.insert(FavoriteIdRole, "favoriteId");

    setRoleNames(roles);

    refresh();
}

InstalledAppsModel::~InstalledAppsModel()
{
    qDeleteAll(m_nodeList);
}

int InstalledAppsModel::rowCount(const QModelIndex& index) const
{
    return index.isValid() ? 0 : m_nodeList.count();
}

int InstalledAppsModel::count() const
{
    return m_nodeList.count();
}

QVariant InstalledAppsModel::data(const QModelIndex &index, int role) const
{
    if (!index.isValid() || index.row() >= m_nodeList.count()) {
        return QVariant();
    }

    const AbstractNode *node = m_nodeList.at(index.row());
    if (role == Qt::DisplayRole) {
        return node->name();
    } else if (role == Qt::DecorationRole) {
        // at least show the oxygen question-mark, otherwise it looks weird blank.
        return node->icon().isEmpty() ? QLatin1String("unknown") : node->icon();
    } else if (role == FavoriteIdRole) {
        return node->favoriteId();
    }

    return QVariant();
}

bool InstalledAppsModel::trigger(int row)
{
    return m_nodeList.at(row)->trigger();
}

void InstalledAppsModel::refresh()
{
    m_pathModel->clear();
    beginResetModel();
    qDeleteAll(m_nodeList);
    m_nodeList.clear();

    if (m_entryPath.isEmpty()) {
        loadRootEntries();
    } else {
        KServiceGroup::Ptr group = KServiceGroup::group(m_entryPath);
        loadServiceGroup(group);
        QVariantMap args;
        args.insert("entryPath", m_entryPath);
        m_pathModel->addPath(group->caption(), SOURCE_ID, args);
    }

    endResetModel();

    emit countChanged();
}

void InstalledAppsModel::loadRootEntries()
{
    KServiceGroup::Ptr group = KServiceGroup::root();
    KServiceGroup::List list = group->entries(false /* sorted: set to false as it does not seem to work */);

    for( KServiceGroup::List::ConstIterator it = list.constBegin(); it != list.constEnd(); it++) {
        const KSycocaEntry::Ptr p = (*it);

        if (p->isType(KST_KServiceGroup)) {
            KServiceGroup::Ptr subGroup = KServiceGroup::Ptr::staticCast(p);

            if (!subGroup->noDisplay() && subGroup->childCount() > 0) {
                m_nodeList << new GroupNode(subGroup, this);
            }
        }
    }
    qSort(m_nodeList.begin(), m_nodeList.end(), AbstractNode::lessThan);
}

void InstalledAppsModel::loadServiceGroup(KServiceGroup::Ptr group)
{
    doLoadServiceGroup(group);

    qSort(m_nodeList.begin(), m_nodeList.end(), AbstractNode::lessThan);

    if (!m_installer.isEmpty()) {
        KService::Ptr service = KService::serviceByDesktopName(m_installer);
        if (service) {
            m_nodeList << new InstallerNode(group, service);
        } else {
            kWarning() << "Could not find service for" << m_installer;
        }
    }
}

void InstalledAppsModel::doLoadServiceGroup(KServiceGroup::Ptr group)
{
    /* This method is separate from loadServiceGroup so that
     * - only one installer node is added at the end
     * - sorting is done only once
     */
    if (!group || !group->isValid()) {
        return;
    }

    KServiceGroup::List list = group->entries(false /* see above */);

    for( KServiceGroup::List::ConstIterator it = list.constBegin();
        it != list.constEnd(); it++) {
        const KSycocaEntry::Ptr p = (*it);

        if (p->isType(KST_KService)) {
            const KService::Ptr service = KService::Ptr::staticCast(p);

            if (!service->noDisplay()) {
                QString genericName = service->genericName();
                if (genericName.isNull()) {
                    genericName = service->comment();
                }
                m_nodeList << new AppNode(service);
            }

        } else if (p->isType(KST_KServiceGroup)) {
            const KServiceGroup::Ptr subGroup = KServiceGroup::Ptr::staticCast(p);

            if (!subGroup->noDisplay() && subGroup->childCount() > 0) {
                doLoadServiceGroup(subGroup);
            }
        }
    }
}

PathModel *InstalledAppsModel::pathModel() const
{
    return m_pathModel;
}

QString InstalledAppsModel::name() const
{
    if (m_pathModel->count() > 0) {
        QModelIndex index = m_pathModel->index(m_pathModel->count() - 1, 0);
        return index.data().toString();
    } else {
        return i18n("Applications");
    }
}

//- InstalledAppsSource ---------------------------------------------
InstalledAppsSource::InstalledAppsSource(QObject *parent)
: AbstractSource(parent)
{}

QAbstractItemModel *InstalledAppsSource::createModelFromConfigGroup(const KConfigGroup &group)
{
    QString entryPath = group.readEntry("entryPath");
    return createModel(entryPath);
}

QAbstractItemModel *InstalledAppsSource::createModelFromArguments(const QVariantMap &arguments)
{
    QString entryPath = arguments.value("entryPath").toString();
    return createModel(entryPath);
}

QAbstractItemModel *InstalledAppsSource::createModel(const QString &entryPath)
{
    KConfigGroup group(config(), "PackageManagement");
    QString installer = group.readEntry("categoryInstaller");

    InstalledAppsModel *model = new InstalledAppsModel(entryPath, installer);
    ChangeNotifier *notifier = new ChangeNotifier(model);
    connect(notifier, SIGNAL(changeDetected()), model, SLOT(refresh()));
    return model;
}

bool InstalledAppsSource::isConfigurable() const
{
    return true;
}

SourceConfigurationWidget *InstalledAppsSource::createConfigurationWidget(const KConfigGroup &group)
{
    return new InstalledAppsConfigurationWidget(group);
}

} // namespace Homerun

#include "installedappsmodel.moc"
