#include <qstringlist.h>
#include <qpixmap.h>

#include <kstandarddirs.h>

#include "gphoto_interface.h"
#include "gphoto_controller.h"
#include "gphoto_events.h"
#include "gphoto_messages.h"

#include "thumbnail_mt.h"
#include "list_mt.h"
#include "camera_fileinfo.h"

Gphoto_Controller :: Gphoto_Controller(QObject *_parent) : QThread() {

  parent = _parent;
  cmdQueue = new Command_Queue<Gphoto_Command>;
  gpInterface = new Gphoto_Interface();


  globalExit = false;

  connect(Gphoto_Messages::gpMessagesWrapper(), SIGNAL(statusChanged(const QString&)),
	  this, SLOT(slot_cameraStatusMsg(const QString&)) );

  connect(Gphoto_Messages::gpMessagesWrapper(), SIGNAL(progressChanged(int)),
     this, SLOT(slot_cameraProgressVal(int)) );

  connect(Gphoto_Messages::gpMessagesWrapper(), SIGNAL( errorMessage(const QString&) ),
     this, SLOT( slot_cameraErrorMsg(const QString&) ));

}

Gphoto_Controller :: ~Gphoto_Controller() {

  globalExit = true;
  wait();

  delete cmdQueue;

  delete gpInterface;
  gpInterface = NULL;
  parent = NULL;

}


void Gphoto_Controller :: initializeCamera() {

  cmdQueue->enqueue(new Gphoto_Command(Gphoto_Command::Init));

}

void Gphoto_Controller :: getSubFolders(const QString& folder) {

  cmdQueue->enqueue(new Gphoto_Command_GetSubFolders(folder));

}

void Gphoto_Controller :: makeFolder(const QString& folder,
				    const QString& newFolder) {

  cmdQueue->enqueue(new Gphoto_Command_MakeFolder(folder, newFolder));

}

void Gphoto_Controller :: deleteFolder(const QString& folder) {

  cmdQueue->enqueue(new Gphoto_Command_DeleteFolder(folder));

}

void Gphoto_Controller :: getThumbNail(const QString& folder,
				       const QString& imageName) {

  cmdQueue->enqueue(new Gphoto_Command_GetThumbNail(folder, imageName));

}

void Gphoto_Controller::getImagesInfo(const QString& folder)
{
  cmdQueue->enqueue(new Gphoto_Command_GetImagesInfo(folder));
}

void Gphoto_Controller :: downloadImage(const QString& folder,
					const QString& imageName,
					const QString& saveFileName) {

  cmdQueue->enqueue(new Gphoto_Command_DownloadImage(folder, imageName,
						    saveFileName));
}

void Gphoto_Controller :: deleteImage(const QString& folder,
				      const QString& imageName) {

  cmdQueue->enqueue(new Gphoto_Command_DeleteImage(folder, imageName));

}

void Gphoto_Controller::openImage(const QString& folder, const QString& imageName)
{

    cmdQueue->enqueue(new Gphoto_Command_OpenImage(folder, imageName));

}

void Gphoto_Controller :: stopOperations() {

  cmdQueue->flush();

}


void Gphoto_Controller :: run() {


  while(true) {

    if (globalExit) return;

    while (cmdQueue->isEmpty()) {
      if (globalExit) return;
      msleep(200);
      showStatus(false);
    }

    showStatus(true);

    Gphoto_Command *cmd = cmdQueue->dequeue();

    switch (cmd->type()) {

    case Gphoto_Command::Init:
      initializeCamera_Real();
      break;

    case Gphoto_Command::GetSubFolders: {

      Gphoto_Command_GetSubFolders*
	command(static_cast<Gphoto_Command_GetSubFolders *>(cmd));

      getSubFolders_Real(command->parentFolder());

      postEvent(parent, new Gphoto_Event_GetSubFolders(true, true, 0, 0, 0));

      delete command;
      cmd = NULL;
      break;
    }

    case Gphoto_Command::MakeFolder:
      qWarning("Make Folder");
      break;

    case Gphoto_Command::DeleteFolder:
      qWarning("Delete Folder");
      break;

    case Gphoto_Command::GetImagesInfo: {

      Gphoto_Command_GetImagesInfo*
	command(static_cast<Gphoto_Command_GetImagesInfo *>(cmd));

      getImagesInfo_Real(command->parentFolder());

      delete command;
      cmd = NULL;
      break;
    }


    case Gphoto_Command::GetThumbNail: {

      Gphoto_Command_GetThumbNail*
	command(static_cast<Gphoto_Command_GetThumbNail *>(cmd));

      getThumbNail_Real(command->parentFolder(),command->imageName());

      delete command;
      cmd = NULL;

      break;
    }

    case Gphoto_Command::DownloadImage: {

      Gphoto_Command_DownloadImage*
	command(static_cast<Gphoto_Command_DownloadImage *>(cmd));

      downloadImage_Real(command->parentFolder(),
			 command->imageName(),
			 command->saveFileName());
      break;
    }

    case Gphoto_Command::DeleteImage: {
      Gphoto_Command_DeleteImage*
	command(static_cast<Gphoto_Command_DeleteImage *>(cmd));

      deleteImage_Real(command->parentFolder(),
		       command->imageName());
      break;
    }

    case Gphoto_Command::OpenImage: {
      Gphoto_Command_OpenImage*
	command(static_cast<Gphoto_Command_OpenImage *>(cmd));

      openImage_Real(command->parentFolder(),
                     command->imageName());
      break;
    }

    default:
      qWarning("Unknown Command\n");
      break;

    }

    if (cmd) delete cmd;

  }


}

void Gphoto_Controller :: initializeCamera_Real() {

  int retVal = gpInterface->initializeCamera();

  if ( retVal == ErrorGphotoSuccess) {

    QString camModel, camPort;

    gpInterface->getCameraSettings(camModel, camPort);
    postEvent(parent, new Gphoto_Event_InitializeCamera(camModel, camPort));

  }
  else if ( retVal == ErrorGphotoSetup){

    QString msg("Camera Model or Port not specified correctly.\n"
		"Please run Setup");
    postEvent(parent, new Gphoto_Event_CameraErrorMsg(msg));

  }
  else if ( retVal == ErrorGphotoInit) {

    QString msg("Failed to initialize camera.\n"
		"Please ensure camera is connected properly and turned on");
    postEvent(parent, new Gphoto_Event_CameraErrorMsg(msg));

  }

}

void Gphoto_Controller :: getSubFolders_Real(const QString& folder) {


  QStringList subFolderList;
  QStringList subFolderNameList;
  int numSubFolders = 0;

  subFolderList.clear();
  subFolderNameList.clear();

  if (gpInterface->getSubFolders(folder, subFolderList, subFolderNameList,
				 numSubFolders) == ErrorGphotoSuccess) {

    if (numSubFolders > 0) {
      postEvent(parent, new Gphoto_Event_GetSubFolders(true, false, folder,
						       subFolderList,
						       subFolderNameList));
    }

    for (int i=0; i<numSubFolders; i++) {
      getSubFolders_Real(subFolderList[i]);
    }

    return;

  }
  else {
    subFolderList.clear();
    subFolderNameList.clear();
    postEvent(parent, new Gphoto_Event_GetSubFolders(false, false, folder,
						     subFolderList,
						     subFolderNameList));
    return;
  }

}

void Gphoto_Controller::getImagesInfo_Real(const QString& folder)
{

    List_MT<Camera_FileInfo> infoList;

    if (gpInterface->getFilesInformation(folder, infoList)
        == ErrorGphotoSuccess)
        postEvent(parent, new Gphoto_Event_GetImagesInfo(folder,
                                                            infoList));
    else {
        QString msg("Failed to get images information from ");
        msg += folder;
        postEvent(parent, new Gphoto_Event_CameraErrorMsg(msg));
    }

}



void Gphoto_Controller :: getThumbNail_Real(const QString& folder,
					     const QString& imageName) {

  Thumbnail_MT thumb;
  gpInterface->getThumbNail(folder, imageName, thumb);


  if (!thumb.isNull())
    postEvent(parent, new Gphoto_Event_GetThumbNail(folder, imageName, thumb));

}

void Gphoto_Controller :: downloadImage_Real(const QString& folder,
					     const QString& imageName,
					     const QString& saveFileName) {

  if (gpInterface->downloadImage(folder, imageName, saveFileName)
      != ErrorGphotoSuccess) {

    QString errorMsg("Failed to download ");
    errorMsg += imageName;
    errorMsg += " from ";
    errorMsg += folder;
    slot_cameraErrorMsg(errorMsg);

  }

}

void Gphoto_Controller :: deleteImage_Real(const QString& folder,
					   const QString& imageName) {

  if (gpInterface->deleteImage(folder, imageName)
      != ErrorGphotoSuccess) {

    QString errorMsg("Failed to delete ");
    errorMsg += imageName;
    errorMsg += " from ";
    errorMsg += folder;
    slot_cameraErrorMsg(errorMsg);

  }
  else {

    postEvent(parent, new Gphoto_Event_DeleteImage(folder, imageName));

  }

}

void Gphoto_Controller::openImage_Real(const QString& folder,
                                       const QString& imageName)
{

    QString saveFile;
    KStandardDirs stdDirs;

    saveFile = stdDirs.resourceDirs("tmp").first();
    saveFile += imageName;

    if ( gpInterface->downloadImage(folder, imageName, saveFile)
         == ErrorGphotoSuccess) {

        postEvent(parent, new Gphoto_Event_OpenImage(folder,
                                                     imageName));

        // Put this in the gui thread
        //        KService::Ptr ptr = KService::serviceByDesktopName(serviceDesktopPath);
        //        if (ptr) KRun::run(*ptr, KURL(saveFile));
    }

}


void Gphoto_Controller :: showStatus(bool _busy) {

  postEvent(parent, new Gphoto_Event_Busy(_busy));

}


bool Gphoto_Controller :: isCameraInitialised() {

  return (gpInterface->isCameraInitialised());

}

bool Gphoto_Controller :: cameraSupportsThumbNails() {

  return (gpInterface->cameraSupportsThumbNails());

}

bool Gphoto_Controller :: cameraSupportsDelete() {

  return (gpInterface->cameraSupportsDelete());

}

bool Gphoto_Controller :: cameraSupportsUpload() {

  return (gpInterface->cameraSupportsUpload());

}

bool Gphoto_Controller :: cameraSupportsMkDir(){

  return (gpInterface->cameraSupportsMkDir());

}

bool Gphoto_Controller :: cameraSupportsDelDir() {

  return (gpInterface->cameraSupportsDelDir());

}

QString Gphoto_Controller :: cameraModel() const {

  QString camModel, camPort;
  gpInterface->getCameraSettings(camModel,camPort);
  return camModel;

}


QString Gphoto_Controller :: cameraPort() const {

  QString camModel, camPort;
  gpInterface->getCameraSettings(camModel,camPort);
  return camPort;

}


void Gphoto_Controller :: slot_cameraStatusMsg(const QString& msg) {

  if (!msg.isEmpty())
    postEvent(parent, new Gphoto_Event_CameraStatusMsg(msg));

}

void Gphoto_Controller :: slot_cameraProgressVal(int val) {


  postEvent(parent, new Gphoto_Event_CameraProgress(val));

}

void Gphoto_Controller :: slot_cameraErrorMsg(const QString& msg) {

  if (!msg.isEmpty())
    postEvent(parent, new Gphoto_Event_CameraErrorMsg(msg));

}

