/***************************************************************************
                          kfmprojectlist.cpp  -  description
                             -------------------
    begin                : Sat Sep 1 2001
    copyright            : (C) 2001 by Jeffrey Yu
    email                : wehjin@yahoo.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kfmprojectlist.h"

#include "kfmproject.h"

KFMProjectList::KFMProjectList()
: QObject(), currentProject_(NULL)
{
	nextProjectId_ = 0;
	
	projectList_.setAutoDelete(true);
}

KFMProjectList::~KFMProjectList(){
	clear();
}


///////////////////////////////////////////////////////////////////
// Public methods

KFMProjectList::Iterator KFMProjectList::iterator() const{
	return Iterator(projectList_);
}

void KFMProjectList::addProject(KFMProject* project){
	if (project != NULL && (projectList_.findRef(project) < 0)) {
		// Assign an id to projects that don't have one
		if (project->id() < 0) {
			project->id(nextProjectId());
		}
		
		// Note the ids that are in use
		existingProjectId(project->id());
	
		projectList_.inSort(project);
		emit projectAdded(project);
		emit contentsChanged();
		
		currentProject(project);
	}
}
	
void KFMProjectList::removeProject(KFMProject* project){
	if (projectList_.findRef(project) >= 0) {
		if (currentProject() == project) {
			currentProject(NULL);
		}

		int pos = projectList_.findRef(project);		
		projectList_.removeRef(project);
		emit projectRemoved(project, pos);
		emit contentsChanged();
	}
}

void KFMProjectList::clear(){
	currentProject(NULL);
	
	projectList_.clear();
	nextProjectId_ = 0;
	emit projectRemoved(NULL, -1);
	emit contentsChanged();
}
	
//////////////////////////////////////////////

int KFMProjectList::projectPosition(KFMProject* project) const{
	if (project != NULL) {
		return projectList_.findRef(project);
	}
	else {
		return -1;
	}
}

KFMProject* KFMProjectList::projectAt(int pos) const{
	if (pos >= 0 && pos < (int)projectList_.count()) {
		return projectList_.at(pos);
	}
	else {
		return NULL;
	}
}

KFMProject* KFMProjectList::firstProject() const{
	return projectList_.first();
}
		
//////////////////////////////////////////////

KFMProject* KFMProjectList::currentProject() const{
	return currentProject_;
}


void KFMProjectList::currentProject(KFMProject* project){
	if (project != currentProject_) {
	
		if (currentProject_ != NULL) {
			disconnect(currentProject_, 0, this, 0);
		}
		
		currentProject_ = project;
		
		if (currentProject_ != NULL) {
			connect(currentProject_, SIGNAL(contentsChanged()), this,
				SLOT(onComponentContentsChanged()));
		}	
		
		emit currentProjectChanged(currentProject_);
	}
}

int KFMProjectList::currentPosition() const{
	return projectPosition(currentProject());
}

//////////////////////////////////////////////

QString KFMProjectList::xmlTag(){
	return QString("projectlist");
}

QDomElement KFMProjectList::domElement(QDomDocument* doc){
	QDomElement eProjectList = doc->createElement(xmlTag());
	
		// Write down the id of the current project
		if (currentProject() != NULL)
			eProjectList.setAttribute("currentid", currentProject()->id());	
	
		Iterator it = iterator();
		for (;it.current() != NULL; ++it) {
			eProjectList.appendChild(it.current()->domElement(doc));
		}
	
		return eProjectList;
}

void KFMProjectList::domElement(QDomElement eProjectList){
	// Read in the id of the current project
	int currentid = eProjectList.attribute("currentid", QString("-1")).toInt();
	KFMProject* currentProject = NULL;
		
	QDomNode n = eProjectList.firstChild();
	while (!n.isNull()) {
		QDomElement e = n.toElement();
		if (!e.isNull()) {
			if (e.tagName() == KFMProject::xmlTag()) {
			
				KFMProject* project = new KFMProject();
				project->domElement(e);
				
				addProject(project);
				if (project->id() == currentid)
					currentProject = project;				
			}
		}
		n = n.nextSibling();
	}
	
	if (currentProject != NULL) {
		this->currentProject(currentProject);
	}
}


///////////////////////////////////////////////////////
// Private Methods
	
/** Notify the projectlist of an existing project id.  This method may
		or may not alter the value of nextProjectId() */
void KFMProjectList::existingProjectId(const int existingProjectId){

	// Make sure we never assign an existing project id to
	// another project.
	if (nextProjectId_ <= existingProjectId)
		nextProjectId_ = existingProjectId + 1;		
}

void KFMProjectList::onComponentContentsChanged(){
	projectList_.sort();
	emit contentsChanged();
}
