/***************************************************************************
                          splineitem.cpp  -  description
                             -------------------
    begin                : Wed Feb 27 2002
    copyright            : (C) 2002 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include <ksimpleconfig.h>
#include <klocale.h>
#include "kplgraph.h"
#include "kplchecklistitem.h"
#include "fitpack.h"
#include "utils.h"
#include "kpldoc.h"
#include "splinedlg.h"
#include "splineitem.h"

SplineItem::SplineItem() : logxo(false), kdeg(3), nk(0), nderiv(0), kdego(3),
 nko(0), nderivo(0), xmin(0.0), xmax(0.0), dx(0.0), xlow(0.0), xmino(0.0),
 xmaxo(0.0), dxo(0.0), xlowo(0.0)
{
}

SplineItem::SplineItem(const SplineItem& f) :
 ScaledItem(f), logxo(f.logxo), kdeg(f.kdeg), nk(f.nk), nderiv(f.nderiv),
 kdego(f.kdego), nko(f.nko), nderivo(f.nderivo), xmin(f.xmin), xmax(f.xmax),
 dx(f.dx), xlow(f.xlow), xmino(f.xmino), xmaxo(f.xmaxo), dxo(f.dxo),
 xlowo(f.xlowo), xv(f.xv), yv(f.yv), t(f.t), c(f.c)
{
  xv.detach();
  yv.detach();
  c.detach();
  t.detach();
}

SplineItem::SplineItem(Kpl::AutoStruct* aut) : ScaledItem(aut), logxo(false),
 kdeg(3), nk(0), nderiv(0), kdego(3), nko(0), nderivo(0), xmin(0.0), xmax(0.0),
 dx(0.0), xlow(0.0), xmino(0.0), xmaxo(0.0), dxo(0.0), xlowo(0.0)
{
}

SplineItem::SplineItem(KSimpleConfig* plo, Kpl::AutoStruct* aut,
                       const KURL&) :
 ScaledItem(plo, aut), logxo(false), kdego(3), nko(0), nderivo(0), xmino(0.0),
 xmaxo(0.0), dxo(0.0), xlowo(0.0)
{
  kdeg = plo->readNumEntry("kdeg", 3);
  nk = plo->readNumEntry("nk");
  nderiv = plo->readNumEntry("nderiv");
  xmin = plo->readDoubleNumEntry("xmin");
  xmax = plo->readDoubleNumEntry("xmax", xmin + 1.0);
  dx = plo->readDoubleNumEntry("dx");
  xlow = plo->readDoubleNumEntry("xlow", xmin);
  t.resize(nk);
  QString s;
  for (int i = 0; i < nk; i++)
    t[i] = plo->readDoubleNumEntry(s.sprintf("t%i", i));
  int nc = QMAX(0, nk - kdeg - 1);
  c.resize(nc);
  for (int i = 0; i < nc; i++)
    c[i] = plo->readDoubleNumEntry(s.sprintf("c%i", i));
}

SplineItem::SplineItem(bool act, int sym, const QString& col, double xn,
                       double yn, double xmi, double xma, double d_x,
                       int deriv, double low, double relSize) :
 ScaledItem(act, sym, col, xn, yn, relSize), logxo(false), kdeg(3), nk(0),
 nderiv(deriv), kdego(3), nko(0), nderivo(0), xmin(xmi), xmax(xma), dx(d_x),
 xlow(low), xmino(0.0), xmaxo(0.0), dxo(0.0), xlowo(0.0)
{
}

SplineItem::~SplineItem()
{
}

SplineItem& SplineItem::operator=(const SplineItem& f)
{
  if (this != &f) {
    *(ScaledItem*)this = f;
    logxo = f.logxo;
    kdeg = f.kdeg;
    nk = f.nk;
    nderiv = f.nderiv;
    kdego = f.kdego;
    nko = f.nko;
    nderivo = f.nderivo;
    xmin = f.xmin;
    xmax = f.xmax;
    dx = f.dx;
    xlow = f.xlow;
    xmino = f.xmino;
    xmaxo = f.xmaxo;
    dxo = f.dxo;
    xlowo = f.xlowo;
    xv = f.xv;
    xv.detach();
    yv = f.yv;
    yv.detach();
    t = f.t;
    t.detach();
    c = f.c;
    c.detach();
  }
  return *this;
}

KplItem::ItemTypes SplineItem::iType() const
{
  return Spline;
}

void SplineItem::draw(KplGraph* g)
{
  if (nk && active) {
    setProperties(g);
    double sav = g->setRelSize(relsiz);
    g->setRelSize(relsiz * sav);
    int n = calcTable(g->logx);
    g->plArray(xv, yv, fx, fy, n);
    g->setRelSize(sav);
  }
}

void SplineItem::writePlo(KSimpleConfig* plo, const KURL& url, bool _abs,
                       KplDoc* m) const
{
  plo->writeEntry("Type", "SPLINEITEM");
  ScaledItem::writePlo(plo, url, _abs, m);
  plo->writeEntry("kdeg", kdeg);
  plo->writeEntry("nk", nk);
  plo->writeEntry("nderiv", nderiv);
  plo->writeEntry("xmin", xmin);
  plo->writeEntry("xmax", xmax);
  plo->writeEntry("dx", dx);
  plo->writeEntry("xlow", xlow);
  QString s;
  for (int i = 0; i < nk; i++)
    plo->writeEntry(s.sprintf("t%i", i), t[i]);
  for (int i = 0; i < (nk - kdeg - 1); i++)
    plo->writeEntry(s.sprintf("c%i", i), c[i]);
}

void SplineItem::setText(KplCheckListItem* it, bool*, bool* funcs) const
{
  *funcs = true;
  it->setText(1, i18n("Spline"));
  QString s = i18n("degree") + " " + QString::number(kdeg);
  if (nderiv) {
    s += ", ";
    switch (nderiv) {
      case -1:
        s += i18n("integral");
        break;
      case 1:
        s += i18n("1st derivative");
        break;
      case 2:
        s += i18n("2nd derivative");
        break;
      case 3:
        s += i18n("3rd derivative");
        break;
      case 4:
        s += i18n("4th derivative");
        break;
      case 5:
        s += i18n("5th derivative");
    }
  }
  it->setText(2, s);
}

int SplineItem::editItem(QWidget* parent, KplDoc* m, int)
{
  SplineDlg dlg(parent, m, this);
  return dlg.exec();
}

KplItem* SplineItem::copy() const
{
  return new SplineItem(*this);
}

void SplineItem::expoItem(int* iext, int* ieyt, double* fxt, double* fyt) const
{
  if (nk) {
    Utils::expo(QMAX(fabs(xmin), fabs(xmax)), iext, fxt);
    double xmi, xma, ymi, yma;
    minMax(&xmi, &xma, &ymi, &yma);
    Utils::expo(QMAX(fabs(ymi), fabs(yma)), ieyt, fyt);
  }
}

void SplineItem::minMax(double* xmi, double* xma, double* ymi, double* yma) const
{
  if (nk) {
    *xmi = xmin;
    *xma = xmax;
    int n = calcTable(logxo);
    Utils::minMaxFile(ymi, yma, yv, n);
  }
}

void SplineItem::exportTable(QTextStream& ts, KplDoc* m) const
{
  int n = calcTable(logxo);
  for (int i = 0; i < n; i++)
    ts << m->number(fx * xv[i]) << m->separator()
       << m->number(fy * yv[i]) << "\n";
}

int SplineItem::calcTable(bool logx) const
{
  bool newv = xv.isNull();
  double d;
  int n;
  if (dx) {
    n = int(fabs((xmax - xmin) / dx)) + 1;
    d = logx ? (log10(xmax / xmin) / (n - 1)) : dx;
  } else {
    n = 101;
    d = 0.01 * (logx ? log10(xmax / xmin) : (xmax - xmin));
  }
  int i;
  if (newv || (logx != logxo) || (kdeg != kdego) || (nk != nko) ||
      (nderiv != nderivo) || (xmin != xmino) || (xmax != xmaxo) ||
      (d != dxo) || (xlow != xlowo)) {
    xv.detach();
    xv.resize(n);
    yv.detach();
    yv.resize(n);
    for (i = 0; i < n; i++)
      xv[i] = logx ? pow(10.0, log10(xmin) + i * d) : (xmin + i * d);
    int ier;
    if (nderiv >= 0)
      FitPack::splder(t, nk, c, kdeg, nderiv, xv, yv.data(), n, &ier);
    else {
      QArray<double> wrk(nk);
      for (i = 0; i < n; i++)
        yv[i] = FitPack::splint(t, nk, c, kdeg, xlow, xv[i], wrk.data());
    }
  }
  logxo = logx;
  kdego = kdeg;
  nko = nk;
  nderivo = nderiv;
  xmino = xmin;
  xmaxo = xmax;
  dxo = d;
  xlowo = xlow;
  return n;
}
