/***************************************************************************
                          kplitem.h  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Wed Aug 25 1999
    copyright            : (C) 2002 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KPLITEM_H
#define KPLITEM_H

#include <qrect.h>
#include <qtextstream.h>
#include <kurl.h>
#include "kplnamespace.h"

class QWidget;
class KSimpleConfig;
class KplDoc;
class KplGraph;
class KplCheckListItem;

/**
  * Kpl item class. Base class for plot items.
  * @author Werner Stille
  */
class KplItem
{
public:
  /** Constructor */
  KplItem();
  /**
   * Constructor. Initializes corresponding to plot file.
   * @param plo KSimpleConfig object containing item data.
   */
  KplItem(KSimpleConfig* plo);
  /**
    * Constructor. Initializes corresponding to arguments.
    * @param active true for visible item.
    * @param x x position in cm.
    * @param y y position in cm.
    * @param relSize relative size of text and symbols.
    */
  KplItem(bool active, double x, double y, double relSize);
  /** Destructor */
  virtual ~KplItem();
  /** Types of items */
  enum ItemTypes{Frame = 1, Array, Function, ParFunction, Spline, Array3D,
                 Function3D, Legend, Text, Line, Arrow, Arc, Rectangle,
                 Ellipse};
  /**
   * Determines type of plot item.
   * @return one of the following:
   * Frame, Array, Function, ParFunction, Spline, Array3D, Function3D,
   * Legend, Text, Line, Arrow, Rectangle, Ellipse.
   */
  virtual ItemTypes iType() const = 0;
  /**
   * Plots item.
   * @param g pointer to KplGraph object.
   */
  virtual void draw(KplGraph* g) = 0;
  /**
   * Writes into plot file.
   * @param plo KSimpleConfig object.
   * @param url URL of plot file.
   * @param abs true for writing absolute file paths.
   * @param m pointer to KplDoc object.
   */
  virtual void writePlo(KSimpleConfig* plo, const KURL& url, bool abs,
                        KplDoc* m) const;
  /**
   * Sets text of corresponding KplCheckListItem in item list.
   * @param it pointer to KplCheckListItem.
   * @param arrays pointer to bool which is set true in case of an array item.
   * @param funcs pointer to bool which is set true in case of an
   * ordinary or parametric function item.
   */
  virtual void setText(KplCheckListItem* it, bool* arrays,
                       bool* funcs) const = 0;
  /**
   * Opens dialog for editing the item.
   * @param parent pointer parent widget.
   * @param m pointer to KplDoc object.
   * @param i item index.
   * @return return code of QDialog::exec().
   */
  virtual int editItem(QWidget* parent, KplDoc* m, int i) = 0;
  /**
   * Copies item.
   * @return pointer to item copy.
   */
  virtual KplItem* copy() const  = 0;
  /**
   * Sets normalization factors of item.
   * @param fx0 normalization factor for x axis.
   * @param fy0 normalization factor for y axis.
   */
  virtual void normalize(double fx, double fy);
  /**
   * Determines normalization factors and their logarithms.
   * @param iex pointer to int for logarithm of normalization factor for x axis.
   * @param iey pointer to int for logarithm of normalization factor for y axis.
   * @param fx pointer to double for normalization factor for x axis.
   * @param fy pointer to double for normalization factor for y axis.
   */
  virtual void expoItem(int* iex, int* iey, double* fx, double* fy) const;
  /**
   * Determines minimum and maximum values for x and y axis.
   * @param xmi pointer to double for minimum x value.
   * @param xma pointer to double for maximum x value.
   * @param ymi pointer to double for minimum y value.
   * @param yma pointer to double for maximum y value.
   */
  virtual void minMax(double* xmi, double* xma, double* ymi, double* yma) const;
  /**
   * Sets function parameter.
   * @param i parameter index.
   * @param value parameter value.
   * @param yFun true for y function parameter, false for x function parameter
   *  (only for parametric functions).
   */
  virtual void setPar(int i, double value, bool yFun);
  /**
   * Exports function value table to file.
   * @param ts stream for output file.
   * @param m KplDoc model object.
   */
  virtual void exportTable(QTextStream& ts, KplDoc* m) const;
  /**
   * Returns boundings rectangle of the item.
   * @return boundings rectangle.
   */
  QRect boundingRect() const;
  /**
   * Returns x position of the item in cm.
   * @return x position in cm.
   */
  double xPos() const;
  /**
   * Returns y position of the item in cm.
   * @return y position in cm.
   */
  double yPos() const;
  /**
   * Sets position of the item in cm.
   * @param x x position in cm.
   * @param y y position in cm.
   */
  void setPos(double x, double y);
  /**
   * Sets sctivation state of the item.
   * @param active true for visible.
   */
  void setActive(bool active);

protected:
  /** Active state of item. */
  bool active;
  double x0, y0, relsiz;
  QRect bRect;

  friend class ItemDlg;
  friend KplDoc;
};

/**
  * Scaled item class. Base class for array and function items.
  * @author Werner Stille
  */
class ScaledItem : public KplItem
{
public:
  /** Constructor */
  ScaledItem();
  /**
    * Constructor. Initializes corresponding to autoplot settings.
    * @param aut pointer to structure containing autoplot settings.
    */
  ScaledItem(Kpl::AutoStruct* aut);
  /**
    * Constructor. Initializes corresponding to plot file.
    * @param plo pointer to KSimpleConfig object containing item data.
    * @param aut pointer to structure containing autoplot settings.
    */
  ScaledItem(KSimpleConfig* plo, Kpl::AutoStruct* aut);
  /**
    * Constructor. Initializes corresponding to arguments.
    * @param active true for visible item.
    * @param symb 0 for circles, < 0 for different markers,
    * >0 for different line types.
    * @param color color for function plot.
    * @param fx normalization factor for x values.
    * @param fy normalization factor for y values.
    * @param relSize relative size.
    */
  ScaledItem(bool active, int symb, const QString& color,
             double fx, double fy, double relSize);
  /** Destructor */
  virtual ~ScaledItem();
  /**
    * Initializes corresponding to plot file.
    * @param plo pointer to KSimpleConfig object containing item data.
    * @param aut pointer to structure containing autoplot settings.
    */
  void readPlo(KSimpleConfig* plo, Kpl::AutoStruct* aut);
  virtual void writePlo(KSimpleConfig* plo, const KURL& url, bool abs,
                        KplDoc* m) const;
  virtual void normalize(double fx0, double fy0);

protected:
  /**
   * Sets color and symbol of item.
   * @param g pointer to KplGraph object for painting.
   */
  void setProperties(KplGraph* g) const;
  int symb;
  unsigned color;
  double fx, fy;
  friend class ArrayDlg;
  friend class FuncDlg;
  friend class ParFuncDlg;
  friend class ItemDlg;
  friend KplDoc;
};

/**
  * Decoration item class. Base class for legend and arrow items.
  * @author Werner Stille
  */
class DecoItem : public KplItem
{
public:
  /** Constructor */
  DecoItem();
  /**
    * Constructor. Initializes corresponding to autoplot settings.
    * @param aut pointer to structure containing autoplot settings.
    */
  DecoItem(Kpl::AutoStruct* aut);
  /**
    * Constructor. Initializes corresponding to plot file.
    * @param plo pointer to KSimpleConfig object containing item data.
    * @param aut pointer to structure containing autoplot settings.
    */
  DecoItem(KSimpleConfig* plo, Kpl::AutoStruct* aut);
  /**
    * Constructor. Initializes corresponding to arguments.
    * @param active true for visible item.
    * @param color color for function plot.
    * @param x x position in cm.
    * @param y y position in cm.
    * @param relSize relative size.
    */
  DecoItem(bool active, const QString& color, double x, double y,
           double relSize);
  /** Destructor */
  virtual ~DecoItem();
  virtual void writePlo(KSimpleConfig* plo, const KURL& url, bool abs,
                        KplDoc* m) const;

protected:
  unsigned color;

  friend class DecoDlg;
};

#endif
