/* 
 *  Copyright (C) 1999-2001 Bernd Gehrmann
 *                          bernd@physik.hu-berlin.de
 *
 * This program may be distributed under the terms of the Q Public
 * License as defined by Trolltech AS of Norway and appearing in the
 * file LICENSE.QPL included in the packaging of this file.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */


#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qwidgetlist.h>
#include <kapp.h>
#include <kbuttonbox.h>
#include <kconfig.h>
#include <kfontdialog.h>
#include <klocale.h>
#include <knuminput.h>

#include "misc.h"

#include "settingsdlg.h"
#include "settingsdlg.moc"


FontButton::FontButton( const QString &text, QWidget *parent, const char *name )
    : QPushButton(text, parent, name)
{
    connect( this, SIGNAL(clicked()), this, SLOT(chooseFont()) );
}


void FontButton::chooseFont()
{
    QFont newFont(font());

    if (KFontDialog::getFont(newFont, false, this) == QDialog::Rejected)
        return;

    setFont(newFont);
    repaint(false);
}


SettingsDialog::SettingsDialog(QWidget *parent, const char *name)
    : QDialog(parent, name, true)
{
    setCaption(i18n("Configure Cervisia"));
    
    QGridLayout *layout = new QGridLayout(this, 3, 2, 10);
    QBoxLayout *leftvbox = new QVBoxLayout();
    QBoxLayout *rightvbox = new QVBoxLayout();
    layout->addLayout(leftvbox, 0, 0);
    layout->addLayout(rightvbox, 0, 1);

    QGroupBox *generalgroup = new QGroupBox(i18n("General"), this);
    leftvbox->addWidget(generalgroup);
    QBoxLayout *generallayout = new QVBoxLayout(generalgroup, 10, 6);
    generallayout->addSpacing(5);
    
    QLabel *timeoutlabel = new QLabel(i18n("&Timeout after which a progress dialog appears (in ms):"), generalgroup);
    timeoutlabel->setMinimumSize(timeoutlabel->sizeHint());
    generallayout->addWidget(timeoutlabel);

    timeoutedit = new KIntNumInput(0, generalgroup);
    timeoutedit->setRange(0, 50000, 100, false);
    timeoutlabel->setBuddy(timeoutedit);
    timeoutedit->setFocus();
    timeoutedit->setMinimumSize(timeoutedit->sizeHint());
    generallayout->addWidget(timeoutedit);
    generallayout->addSpacing(5);

    QLabel *usernamelabel = new QLabel(i18n("&User name for the ChangeLog editor:"), generalgroup);
    usernamelabel->setMinimumSize(usernamelabel->sizeHint());
    generallayout->addWidget(usernamelabel);

    usernameedit = new QLineEdit(generalgroup);
    usernamelabel->setBuddy(usernameedit);
    usernameedit->setMinimumSize(usernameedit->sizeHint());
    generallayout->addWidget(usernameedit);
    generallayout->addSpacing(5);

    QLabel *cvspathlabel = new QLabel(i18n("&Path to cvs:"), generalgroup);
    cvspathlabel->setMinimumSize(cvspathlabel->sizeHint());
    generallayout->addWidget(cvspathlabel);

    cvspathedit = new QLineEdit(generalgroup);
    cvspathlabel->setBuddy(cvspathedit);
    cvspathedit->setMinimumSize(cvspathedit->sizeHint());
    generallayout->addWidget(cvspathedit);

    generallayout->activate();
    
    QGroupBox *diffgroup = new QGroupBox(i18n("Diff"), this);
    leftvbox->addWidget(diffgroup);
    QBoxLayout *difflayout = new QVBoxLayout(diffgroup, 10, 6);
    difflayout->addSpacing(5);

    QLabel *contextlabel = new QLabel(i18n("&Number of context lines in diff dialog:"), diffgroup);
    contextlabel->setMinimumSize(contextlabel->sizeHint());
    difflayout->addWidget(contextlabel);

    contextedit = new KIntNumInput(0, diffgroup);
    contextedit->setRange(0, 65535, 1, false);
    contextlabel->setBuddy(contextedit);
    contextedit->setMinimumSize(contextedit->sizeHint());
    difflayout->addWidget(contextedit);
    difflayout->addSpacing(5);

    QLabel *diffoptlabel = new QLabel(i18n("Additional options for cvs &diff:"), diffgroup);
    diffoptlabel->setMinimumSize(diffoptlabel->sizeHint());
    difflayout->addWidget(diffoptlabel);

    diffoptedit = new QLineEdit(diffgroup);
    diffoptlabel->setBuddy(diffoptedit);
    diffoptedit->setMinimumSize(diffoptedit->sizeHint());
    difflayout->addWidget(diffoptedit);

    difflayout->activate();
    
    QGroupBox *statusgroup = new QGroupBox(i18n("Status"), this);
    leftvbox->addWidget(statusgroup);
    QBoxLayout *statuslayout = new QVBoxLayout(statusgroup, 10, 6);
    statuslayout->addSpacing(5);

    remotestatusbox = new QCheckBox(i18n("When opening a sandbox from a &remote repository,\n"
                                         "start a File->Status command automatically"), statusgroup);
    remotestatusbox->setMinimumSize(remotestatusbox->sizeHint());
    statuslayout->addWidget(remotestatusbox);
    
    localstatusbox = new QCheckBox(i18n("When opening a sandbox from a &local repository,\n"
                                        "start a File->Status command automatically"), statusgroup);
    localstatusbox->setMinimumSize(localstatusbox->sizeHint());
    statuslayout->addWidget(localstatusbox);
    
    statuslayout->activate();
    
#if 0
    QGridLayout *editorlayout = new QGridLayout(editorgroup, 4, 2, 10, 6);

    editoredit = new QLineEdit(editorgroup);
    editoredit->setMinimumSize(editoredit->sizeHint());
    editorlayout->addWidget(editoredit, 0, 1);

    QLabel *editorlabel = new QLabel(editoredit, i18n("&Editor:"), editorgroup);
    editorlabel->setMinimumSize(editorlabel->sizeHint());
    editorlayout->addWidget(editorlabel, 0, 0);

    usedcopbox = new QCheckBox(i18n("Use &DCOP"), editorgroup);
    usedcopbox->setMinimumSize(usedcopbox->sizeHint());
    editorlayout->addMultiCellWidget(usedcopbox, 1, 1, 0, 1);

    clientedit = new QLineEdit(editorgroup);
    clientedit->setMinimumSize(clientedit->sizeHint());
    editorlayout->addWidget(clientedit, 2, 1);
    
    QLabel *clientlabel = new QLabel(clientedit, i18n("&Client:"), editorgroup);
    clientlabel->setMinimumSize(clientlabel->sizeHint());
    editorlayout->addWidget(clientlabel, 2, 0);

    objectedit = new QLineEdit(editorgroup);
    objectedit->setMinimumSize(objectedit->sizeHint());
    editorlayout->addWidget(objectedit, 3, 1);

    QLabel *objectlabel = new QLabel(objectedit, i18n("&Object:"), editorgroup);
    objectlabel->setMinimumSize(objectlabel->sizeHint());
    editorlayout->addWidget(objectlabel, 3, 0);

    connect(usedcopbox, SIGNAL(toggled(bool)),
            clientedit, SLOT(setEnabled(bool)));
    connect(usedcopbox, SIGNAL(toggled(bool)),
            objectedit, SLOT(setEnabled(bool)));
    editorlayout->activate();
#endif

    QGroupBox *editorgroup = new QGroupBox(i18n("Open File"), this);
    rightvbox->addWidget(editorgroup);

    QBoxLayout *editorlayout = new QVBoxLayout(editorgroup, 10, 6);
    editorlayout->addSpacing(5);
    
    QLabel *editorlabel = new QLabel(i18n("&Editor:"), editorgroup);
    editorlabel->setMinimumSize(editorlabel->sizeHint());
    editorlayout->addWidget(editorlabel);

    editoredit = new QLineEdit(editorgroup);
    editorlabel->setBuddy(editoredit);
    editoredit->setMinimumSize(editoredit->sizeHint());
    editorlayout->addWidget(editoredit);

    QGroupBox *lookgroup = new QGroupBox(i18n("Look & Feel"), this);
    rightvbox->addWidget(lookgroup);
    QBoxLayout *looklayout = new QVBoxLayout(lookgroup, 10, 6);
    looklayout->addSpacing(5);

    protocolfontbox = new FontButton(i18n("Font for protocol window"), lookgroup);
    protocolfontbox->setMinimumSize(protocolfontbox->sizeHint());
    looklayout->addWidget(protocolfontbox);
    
    annotatefontbox = new FontButton(i18n("Font for annotate view"), lookgroup);
    annotatefontbox->setMinimumSize(annotatefontbox->sizeHint());
    looklayout->addWidget(annotatefontbox);

    difffontbox = new FontButton(i18n("Font for diff view"), lookgroup);
    difffontbox->setMinimumSize(difffontbox->sizeHint());
    looklayout->addWidget(difffontbox);

    splitterbox = new QCheckBox(i18n("Split main window &horizontally"), lookgroup);
    splitterbox->setMinimumSize(splitterbox->sizeHint());
    looklayout->addSpacing(10);
    looklayout->addWidget(splitterbox);

    looklayout->activate();
    rightvbox->addStretch();
    
    readSettings();

    QFrame *frame = new QFrame(this);
    frame->setFrameStyle(QFrame::HLine | QFrame::Sunken);
    frame->setMinimumHeight(frame->sizeHint().height());
    layout->addMultiCellWidget(frame, 1, 1, 0, 1);

    KButtonBox *buttonbox = new KButtonBox(this);
    buttonbox->addStretch();
    QPushButton *ok = buttonbox->addButton(i18n("OK"));
    QPushButton *cancel = buttonbox->addButton(i18n("Cancel"));
    connect( ok, SIGNAL(clicked()), this, SLOT(accept()) );
    connect( cancel, SIGNAL(clicked()), this, SLOT(reject()) );
    ok->setDefault(true);
    buttonbox->layout();
    layout->addMultiCellWidget(buttonbox, 2, 2, 0, 1);

    layout->activate();
    resize(sizeHint());
}


void SettingsDialog::readSettings()
{
    KConfig *config = kapp->config();
    config->setGroup("General");
    timeoutedit->setValue((int)config->readUnsignedNumEntry("Timeout", 4000));
    usernameedit->setText(config->readEntry("Username", userName()));
    cvspathedit->setText(config->readEntry("CVSPath", "cvs"));
    contextedit->setValue((int)config->readUnsignedNumEntry("ContextLines", 65535));
    diffoptedit->setText(config->readEntry("DiffOptions", ""));
    remotestatusbox->setChecked(config->readBoolEntry("StatusForRemoteRepos", false));
    localstatusbox->setChecked(config->readBoolEntry("StatusForLocalRepos", false));
    config->setGroup("Communication");
    editoredit->setText(config->readEntry("Editor", "kwrite"));
#if 0
    bool usedcop = config->readBoolEntry("UseDCOP", false);
    usedcopbox->setChecked(usedcop);
    clientedit->setEnabled(usedcop);
    objectedit->setEnabled(usedcop);
    clientedit->setText(config->readEntry("DCOPClient"));
    objectedit->setText(config->readEntry("DCOPObject"));
#endif
    config->setGroup("LookAndFeel");
    protocolfontbox->setFont(config->readFontEntry("ProtocolFont"));
    annotatefontbox->setFont(config->readFontEntry("AnnotateFont"));
    difffontbox->setFont(config->readFontEntry("DiffFont"));
    splitterbox->setChecked(config->readBoolEntry("SplitHorizontally"));
}


void SettingsDialog::writeSettings()
{
    KConfig *config = kapp->config();
    config->setGroup("General");
    config->writeEntry("Timeout", (unsigned)timeoutedit->value());
    config->writeEntry("Username", usernameedit->text());
    config->writeEntry("CVSPath", cvspathedit->text());
    config->writeEntry("ContextLines", (unsigned)contextedit->value());
    config->writeEntry("DiffOptions", diffoptedit->text());
    config->writeEntry("StatusForRemoteRepos", remotestatusbox->isChecked());
    config->writeEntry("StatusForLocalRepos", localstatusbox->isChecked());
    config->setGroup("Communication");
    config->writeEntry("Editor", editoredit->text());
#if 0
    config->writeEntry("UseDCOP", usedcopbox->isChecked());
    config->writeEntry("DCOPClient", clientedit->text());
    config->writeEntry("DCOPObject", objectedit->text());
#endif
    config->setGroup("LookAndFeel");
    config->writeEntry("ProtocolFont", protocolfontbox->font());
    config->writeEntry("AnnotateFont", annotatefontbox->font());
    config->writeEntry("DiffFont", difffontbox->font());
    config->writeEntry("SplitHorizontally", splitterbox->isChecked());

    // I'm not yet sure whether this is a hack or not :-)
    QWidgetListIt it(*QApplication::allWidgets());
    for (; it.current(); ++it)
        {
            QWidget *w = it.current();
            if (w->inherits("ProtocolView"))
                w->setFont(protocolfontbox->font());
            if (w->inherits("AnnotateView"))
                w->setFont(annotatefontbox->font());
            if (w->inherits("DiffView"))
                w->setFont(difffontbox->font());
        }
}

void SettingsDialog::done(int res)
{
    if (res == Accepted)
        writeSettings();
    QDialog::done(res);
    delete this;
}


// Local Variables:
// c-basic-offset: 4
// End:
