/***************************************************************************
                          psgraph.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Wed Sep 08 1999
    copyright            : (C) 2001 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qglobal.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include "psgraph.h"
#include "utils.h"

const double PSGraph::pts = 28.34645669;

PSGraph::PSGraph()
{
}

PSGraph::~PSGraph()
{
}

void PSGraph::psInit(FILE* f, double xf, double yf)
{
  fps = f;
  logx = logy = false;
  kSymb = 0;
  relSize = 1.0;
  xmin = ymin = x0 = y0 = 0.0;
  xmax = xl = 21.0;
  ymax = yl = 29.7;
  time_t t = time(0);
  fprintf(fps, "%%!PS-Adobe-3.0 EPSF-3.0\n"
    "%%%%BoundingBox: 0 0 %i %i              \n"
    "%%%%Creator: Kpl\n"
    "%%%%CreationDate: %s"
    "%%%%Pages: 1\n\n",
    qRound(xf * pts), qRound(yf * pts),
    ctime(&t));
  fprintf(fps, "%%User code starts at line 407\n\n"
    "/enc [\n 16#A0 /space\n 16#A4 /currency\n"
    " 16#A6 /brokenbar\n 16#A8 /dieresis\n"
    " 16#A9 /copyright\n 16#AA /ordfeminine\n"
    " 16#AC /logicalnot\n 16#AD /endash\n"
    " 16#AE /registered\n 16#AF /macron\n"
    " 16#B0 /ring\n 16#B1 /plusminus\n"
    " 16#B2 /twosuperior\n 16#B3 /threesuperior\n"
    " 16#B4 /acute\n 16#B5 /mu\n"
    " 16#B7 /periodcentered\n 16#B8 /cedilla\n"
    " 16#B9 /onesuperior\n 16#BA /ordmasculine\n"
    " 16#BC /onequarter\n 16#BD /onehalf\n"
    " 16#BE /threequarters\n 16#C0 /Agrave\n"
    " 16#C1 /Aacute\n 16#C2 /Acircumflex\n"
    " 16#C3 /Atilde\n 16#C4 /Adieresis\n"
    " 16#C5 /Aring\n 16#C6 /AE\n"
    " 16#C7 /Ccedilla\n 16#C8 /Egrave\n"
    " 16#C9 /Eacute\n 16#CA /Ecircumflex\n"
    " 16#CB /Edieresis\n 16#CC /Igrave\n"
    " 16#CD /Iacute\n 16#CE /Icircumflex\n"
    " 16#CF /Idieresis\n 16#D0 /Eth\n"
    " 16#D1 /Ntilde\n 16#D2 /Ograve\n"
    " 16#D3 /Oacute\n 16#D4 /Ocircumflex\n"
    " 16#D5 /Otilde\n 16#D6 /Odieresis\n"
    " 16#D7 /multiply\n 16#D8 /Oslash\n"
    " 16#D9 /Ugrave\n 16#DA /Uacute\n"
    " 16#DB /Ucircumflex\n 16#DC /Udieresis\n"
    " 16#DD /Yacute\n 16#DE /Thorn\n"
    " 16#DF /germandbls\n 16#E0 /agrave\n"
    " 16#E1 /aacute\n 16#E2 /acircumflex\n"
    " 16#E3 /atilde\n 16#E4 /adieresis\n"
    " 16#E5 /aring\n 16#E6 /ae\n"
    " 16#E7 /ccedilla\n 16#E8 /egrave\n"
    " 16#E9 /eacute\n 16#EA /ecircumflex\n"
    " 16#EB /edieresis\n 16#EC /igrave\n"
    " 16#ED /iacute\n 16#EE /icircumflex\n"
    " 16#EF /idieresis\n 16#F0 /eth\n"
    " 16#F1 /ntilde\n 16#F2 /ograve\n"
    " 16#F3 /oacute\n 16#F4 /ocircumflex\n"
    " 16#F5 /otilde\n 16#F6 /odieresis\n"
    " 16#F7 /divide\n 16#F8 /oslash\n"
    " 16#F9 /ugrave\n 16#FA /uacute\n"
    " 16#FB /ucircumflex\n 16#FC /udieresis\n"
    " 16#FD /yacute\n 16#FE /thorn\n"
    " 16#FF /ydieresis\n] def\n");
  fprintf(fps, "/ws 12 dict def\n"
    "/xmin 0 def\n"
    "/ymin 0 def\n"
    "/xmax 21 def\n"
    "/ymax 29.7 def\n"
    "/scalx 28.346456 def\n"
    "/xscal scalx def\n"
    "/yscal xscal def\n"
    "/minx 0 def\n"
    "/miny 0 def\n"
    "/maxx xscal xmax mul def\n"
    "/maxy yscal ymax mul def\n"
    "/xcur 0 def\n"
    "/ycur 0 def\n"
    "/colframe 0 def\n"
    "/colgrid 0 def\n"
    "/coldata 0 def\n"
    "/dir 0 def\n"
    "/symbol 0 def\n"
    "/dpos 0 def\n"
    "/relSize 1 def\n"
    "/logx false def\n"
    "/logy false def\n\n"
    "/ed {exch def} bind def\n\n"
    "/encfont {\n ws begin\n /newname ed /orgname ed\n"
    " /orgdict orgname findfont def\n"
    " /newfont orgdict maxlength dict def orgdict {\n"
    "  exch dup /FontID ne {\n   dup /Encoding eq {\n"
    "    exch dup length array copy newfont 3 1 roll put\n   }"
    " {exch newfont 3 1 roll put} ifelse\n  } {pop pop} ifelse\n"
    " } forall\n  newfont /FontName newname put\n"
    " enc aload pop\n  enc length 2 idiv\n"
    " {newfont /Encoding get 3 1 roll put} repeat\n"
    " newname newfont definefont end\n"
    "} bind def\n\n"
    "/strcat {\n"
    " /s2 ed /s1 ed s1 length dup s2 length add string /s3 ed s3 exch "
    "s2 putinterval\n" " s3 0 s1 putinterval s3\n"
    "} bind def\n\n"
    "/r2r {ymin sub yscal mul miny add exch"
    " xmin sub xscal mul minx add exch} bind def\n\n");
  fprintf(fps, "/mv {newpath r2r moveto} bind def\n\n"
    "/rect {\n /y2 ed /x2 ed /y1 ed /x1 ed\n"
    " newpath x1 y1 moveto x2 y1 lineto x2 y2 lineto x1 y2 lineto closepath\n"
    "} bind def\n\n"
    "/showc {\n dup stringwidth exch 2 div neg exch rmoveto currentpoint"
    " 3 2 roll show moveto\n"
    "} bind def\n\n"
    "/showr {\n dup stringwidth exch neg exch rmoveto currentpoint\n"
    " 3 2 roll show moveto\n"
    "} bind def\n\n"
    "/rel {maxx minx sub mul relSize mul} bind def\n\n"
    "/slw {rel setlinewidth} bind def\n\n"
    "/scf {rel scalefont setfont} bind def\n\n"
    "/mrk {\n /sh 0.01 rel def sh mul exch sh mul exch rmoveto {\n"
    "  sh mul exch sh mul exch rlineto\n } repeat\n} bind def\n\n"
    "/m0 {newpath r2r 0.01 rel 0 360 arc} bind def\n\n"
    "/m1 {0.004 slw mv 0 0 rlineto stroke 0.00295 slw} bind def\n\n"
    "/m2 {mv -1 1 1 1 1 -1 -1 -1 4 0 1 mrk} bind def\n\n"
    "/m3 {mv 0 2 2 0 0 -2 -2 0 4 1 1 mrk} bind def\n\n"
    "/m4 {mv -1 1.732 2 0 -1 -1.732 3 0 0.866 mrk} bind def\n\n"
    "/m5 {mv 2 2 -2 0 2 -2 -2 0 4 1 1 mrk} bind def\n\n"
    "/m6 {mv 0 1 0.5 0.5 1 0 0.5 -0.5 0 -1 -0.5 -0.5 -1 0 -0.5"
    " 0.5 8 1 0.5 mrk} bind def\n\n"
    "/m7 {mv 2 -2 -1 1 1 1 -2 -2 4 1 1 mrk} bind def\n\n"
    "/m8 {mv 2 0 -1 0 0 1 0 -2 4 0 1 mrk} bind def\n\n"
    "/m9 {mv -1 1 -1 -1 1 1 0 0.866 4 0 -0.866 mrk} bind def\n\n"
    "/m10 {mv 1 1 -2 0 1 -1 0 2 4 0 -1 mrk} bind def\n\n"
    "/lintyp {\n 1 sub dup 0 lt {pop 0} if 9 mod dup\n"
    " [8 7 2 1 9 8 7 4 1] exch get /p0 ed\n"
    " [8 0 0 0 0 1 2 4 1] exch get /p1 ed\n"
    " /f 0.003125 rel def 4 array dup 0 p0 f mul put dup 1 16 p0 sub p1 sub f mul\n"
    " p1 0 eq {put dup dup 0 get 2 exch} {0.5 mul put dup 2 p1 f mul} ifelse put\n"
    " dup dup 1 get 3 exch put dpos setdash\n"
    "} bind def\n\n"
    "/err {\n fy mul yscal mul /dy ed fy mul /y ed logy\n"
    " {y 0 gt {/dy dy y div 0.4343 mul def} {/dy 0 def} ifelse} if\n"
    " dy sh lt {pop} {\n"
    "  fx mul logx {log} if y logy {log} if mv 0 sh rmoveto 0 dy sh sub rlineto\n"
    "  0.5 sh mul 0 rmoveto sh neg 0 rlineto 0 -2 dy mul rmoveto sh 0 rlineto\n"
    "  -0.5 sh mul 0 rmoveto 0 dy sh sub rlineto stroke\n"
    " } ifelse\n"
    "} bind def\n\n");
  fprintf(fps, "/format {\n /ymax ed /xmax ed /xmin 0 def /ymin 0 def\n"
    " /xscal scalx def /yscal xscal def /minx 0 def /miny 0 def\n"
    " /maxx xscal xmax mul def /maxy yscal ymax mul def /xcur 0 def /ycur 0 def\n"
    "} bind def\n\n"
    "/window {\n /ymax ed /ymin ed /xmax ed /xmin ed\n"
    " /xmax xmax xmin add def /ymax ymax ymin add def\n"
    " /minx xscal xmin mul def /miny yscal ymin mul def\n"
    " /maxx xscal xmax mul def /maxy yscal ymax mul def\n"
    " /xcur 0 def /ycur 0 def 0.00295 slw\n"
    " /latin1 findfont 0.033333 scf\n"
    "} bind def\n\n"
    "/psscale {\n /logy ed /logx ed /ymax ed /ymin ed /xmax ed /xmin ed\n"
    " logx {/xmin xmin log def /xmax xmax log def} if\n"
    " logy {/ymin ymin log def /ymax ymax log def} if\n"
    " /xscal maxx minx sub xmax xmin sub div def\n"
    " /yscal maxy miny sub ymax ymin sub div def\n"
    " /xcur xmin def /ycur ymin def\n"
    "} bind def\n\n"
    "/scol {\n"
    " /col ed col 16#ff and 16#ff div col -8 bitshift 16#ff and 16#ff div\n"
    " col -16 bitshift 16#ff div setrgbcolor\n"
    "} bind def\n\n"
    "/frame {minx miny maxx maxy rect stroke} bind def\n\n"
    "/psclip {\n"
    " /d 0.00208 rel def minx d add miny d add maxx d sub maxy d sub rect clip\n"
    "} bind def\n\n"
    "/prenum {\n"
    " /ndig ed ndig 1 lt\n"
    " {round ndig 0 eq\n"
    "  {cvr 256 string cvs dup 0 exch length 1 sub getinterval}\n"
    "  {cvi 256 string cvs} ifelse}\n"
    " {/f 10 ndig exp def f mul round f div cvr 256 string cvs dup (.) search\n"
    " {pop pop length ndig exch sub /n ed n 0 ne\n"
    " {dup length n add string /t ed t 0 3 -1 roll putinterval\n"
    " 1 1 n {t length exch sub t exch 48 put} for t}\n"
    " if} {pop} ifelse} ifelse\n"
    "} bind def\n\n"
    "/incLog {\n"
    " log /lgf ed /xlg ed /mant xlg dup dup abs 1.0e-6 mul add cvi sub def\n"
    " mant abs 1.0e-9 lt {/mant 0 def} if mant 0 lt {/mant mant 1 add def} if\n"
    " xlg mant 0.1 lt {lgf add} {mant 0.4 lt {2.5 log add}\n"
    "  {mant 0.6 lt {10 3 div log add} {2 log add} ifelse} ifelse} ifelse\n"
    "} bind def\n\n");
  fprintf(fps, "/raster {\n"
    " /mode ed /inty ed /intx ed /ytic ed /xtic ed\n"
    " mode 0 ne {\n"
    "  logx {xtic 2.5 lt {/xtic 2 def} {xtic 5.5 lt {/xtic 3 def}\n"
    "   {xtic 99 lt {/xtic 10 def} if}} ifelse} if\n"
    "  colgrid scol 0.00208 slw /dx maxx minx sub def /dy maxy miny sub def\n"
    "  mode 2 le {/dx dx 0.02 mul def /dy dx def} if /x xmin def {logx\n"
    "   {/xa x def /x x xtic incLog def\n"
    "    x xmax lt {x ymin mv 0 dy rlineto stroke} if intx 1 gt {\n"
    "     /xd 10 xa mant sub exp def {/xa 10 xa exp xd add log def\n"
    "      xa x ge xa xmax ge or {exit} if xa ymin mv 0 dy\n"
    "      mode 2 le {0.5 mul} if rlineto stroke} loop\n"
    "    } if} {mode 2 le intx 1 gt and {\n"
    "     /xm x def intx 1 sub {\n"
    "      /xm xtic intx div xm add def xm xmax ge {exit} if\n"
    "      xm ymin mv 0 dy 0.5 mul rlineto stroke\n"
    "     } repeat\n"
    "    } if /x x xtic add def x xmax ge {exit} if x ymin mv 0 dy rlineto stroke\n"
    "   } ifelse x xmax ge {exit} if} loop /y ymin def {logy\n"
    "   {/ya y def /y y ytic incLog def\n"
    "    y ymax lt {xmin y mv dx 0 rlineto stroke} if inty 1 gt {\n"
    "     /yd 10 ya mant sub exp def {/ya 10 ya exp yd add log def\n"
    "      ya y ge ya ymax ge or {exit} if xmin ya mv dx\n"
    "      mode 2 le {0.5 mul} if 0 rlineto stroke} loop\n"
    "    } if} {mode 2 le inty 1 gt and {\n"
    "     /ym y def inty 1 sub {\n"
    "      /ym ytic inty div ym add def ym ymax ge {exit} if\n"
    "      xmin ym mv dx 0.5 mul 0 rlineto stroke\n"
    "     } repeat\n"
    "    } if /y y ytic add def y ymax ge {exit} if xmin y mv dx 0 rlineto stroke\n"
    "   } ifelse y ymax ge {exit} if} loop\n"
    "  mode 2 eq mode 4 eq or {\n"
    "   colframe scol /y ymin xmax xmin sub 0.05 mul relSize mul xscal mul yscal div\n"
    "   sub def /e xmax 2.99 gt xmin -2.99 lt or xtic 9.9 gt and logx and def\n"
    "   /nd ndigx def /x xmin def {\n"
    "    x xmax gt {/x xmax def} if\n"
    "    e {(10#nu) x round 4 string cvs strcat x y 2 text}\n"
    "     {logx {x 0 le {/nd 1 x 0.01 add cvi sub def} {/nd -1 def} ifelse} if\n"
    "     x y mv x logx {10 exch exp} if nd prenum showc} ifelse\n"
    "    logx {/x x xtic incLog def} {/x x xtic add def} ifelse\n"
    "    x xtic 1e-3 mul xmax add gt {exit} if\n"
    "   } loop /x xmin xmax xmin sub 0.025 mul relSize mul sub def\n"
    "   /dy xmax xmin sub 0.0125 mul relSize mul xscal mul yscal div def\n"
    "   /e ymax 2.99 gt ymin -2.99 lt or ytic 9.9 gt and logy and def\n"
    "   /nd ndigy def /y ymin def {\n"
    "    y ymax gt {/y ymax def}\n"
    "    e {(10#nu) y round 4 string cvs strcat x y dy sub 3 text}\n"
    "     {logy {y 0 le {/nd 1 y 0.01 add cvi sub def} {/nd -1 def} ifelse} if\n"
    "     x y dy sub mv y logy {10 exch exp} if nd prenum showr} ifelse\n"
    "    logy {/y y ytic incLog def} {/y y ytic add def} ifelse\n"
    "    y ytic 1e-3 mul ymax add gt {exit} if\n"
    "   } loop\n"
    "  } if\n"
    " } if colframe scol 0.00417 slw frame coldata scol 0.00295 slw\n"
    "} bind def\n\n");
  fprintf(fps, "/text {\n"
    " /iorg ed gsave translate dir rotate 0 0 moveto /str ed\n"
    " /latin1 findfont 0.033333 scf colframe scol str (#) search {\n"
    "  pop pop pop /dx 0 def iorg 1 eq {1} {0} ifelse 1 1 {\n"
    "   /j ed /ilev 0 def /s str def\n"
    "   j 1 eq {dx iorg 2 eq {0.5 mul} if neg 0 rmoveto} if {\n"
    "    s (#) search exch /s1 ed {pop /c ed} {/c 0 string def} ifelse\n"
    "    j 0 eq {s1 stringwidth pop dx add /dx ed} {\n"
    "     ilev 0 eq {s1 show} {\n"
    "      ilev 1 eq {0.015} {-0.01} ifelse\n"
    "      rel dup 0 exch rmoveto s1 show neg 0 exch rmoveto\n"
    "     } ifelse\n"
    "    } ifelse c length 2 lt {exit} {\n"
    "     c 0 get dup 96 gt {32 sub} if 1 string dup 0 4 -1 roll put\n"
    "     (NHBOSZI) exch search {\n"
    "      length [\n"
    "       /latin1 /Helvetica /Helvetica-Bold /Helvetica-Oblique\n"
    "       /Symbol /ZapfDingbats /Helvetica-BoldOblique\n"
    "      ] exch get findfont c 1 get dup 96 gt {32 sub} if 1 string dup\n"
    "      0 4 -1 roll put (NUD) exch search {length /ilev ed pop} if\n"
    "      pop ilev 0 eq {0.033333} {0.022222} ifelse scf pop\n"
    "     } if pop /s c 2 c length 2 sub getinterval def\n"
    "    } ifelse s length 0 eq {exit} if\n"
    "   } loop /latin1 findfont 0.033333 scf\n"
    "  } for\n"
    " } {iorg 1 eq {show} {iorg 2 eq {showc} {showr} ifelse} ifelse} ifelse\n"
    " grestore coldata scol\n"
    "} bind def\n\n"
    "/letH {\n"
    " minx maxx add 0.5 mul 0.025 rel maxy add 2 text\n"
    "} bind def\n\n"
    "/letXy {\n"
    " /iax ed /iex ed /str ed str length iex or 0 ne {\n"
    "  iex 0 eq {str} {(10#nu) iex 4 string cvs strcat ( #nn) strcat str strcat}\n"
    "  ifelse iax 2 eq {\n"
    "   minx ymin ndigy prenum length ymax ndigy prenum length 2 copy lt {exch} if pop\n"
    "   3 add 0.01875 rel mul sub miny maxy add 0.5 mul /dir 90 def\n"
    "  } {minx maxx add 0.5 mul miny 0.1125 rel sub} ifelse 2 text\n"
    "  /dir 0 def\n"
    " } if\n"
    "} bind def\n\n");
  fprintf(fps, "/letX {1 letXy} bind def\n\n"
    "/letY {2 letXy} bind def\n\n"
    "/plarry {\n"
    " gsave psclip 3 1 roll /fy ed /fx ed symbol 0 gt {\n"
    "  symbol lintyp 3 1 roll fy mul logy {log} if exch fx mul logx {log} if\n"
    "  exch newpath r2r /y ed /x ed x y moveto 1 sub {\n"
    "   fy mul logy {log} if exch fx mul logx {log} if exch r2r dup y sub dup\n"
    "   mul 2 index x sub dup mul add sqrt\n"
    "   dpos add /dpos ed dup /y ed /x 2 index def lineto\n"
    "  } repeat stroke /dpos dpos dup 0.05 rel div floor 0.05 rel mul sub def\n"
    "  [] dpos setdash\n"
    " } {\n"
    "  /m [/m0 /m1 /m2 /m3 /m4 /m5 /m6 /m7 /m8 /m9 /m10 /m0 /m2 /m3 /m4 /m5 /m6]\n"
    "  symbol neg 17 mod get load def {\n"
    "   symbol -17 eq {pop pop} {fy mul logy {log} if exch fx mul logx {log} if\n"
    "   exch m symbol neg\n"
    "   17 mod 10 le {stroke} {fill} ifelse} ifelse\n"
    "  } repeat\n"
    " } ifelse grestore\n"
    "} bind def\n\n"
    "/plerry {\n"
    " gsave psclip 3 1 roll /fy ed /fx ed /sh 0.01 rel def {err} repeat"
    " grestore\n} bind def\n\n"
    "%%%%Page: 1 1\n\n"
    "gsave\n"
    "/Helvetica /latin1 encfont findfont 0.033333 scf\n"
    "1 setlinecap 1 setlinejoin\n\n"
    "0.00295 slw\n\n%%User code starts here\n\n");
}

void PSGraph::format(double x, double y)
{
  fprintf(fps, "%.2f %.2f format\n", x, y);
}

double PSGraph::setRelSize(double r)
{
  double old = relSize;
  relSize = r;
  fprintf(fps, "/relSize %.2f def\n", r);
  return old;
}

void PSGraph::Window(double x0a, double xla, double y0a, double yla)
{
  x0 = x0a;
  xl = xla;
  y0 = y0a;
  yl = yla;
  xmin = xmax = 0;
  xmax = xl;
  ymax = yl;
  fprintf(fps, "%.2f %.2f %.2f %.2f window\n", x0, xl, y0, yl);
}

void PSGraph::scale(double xl, double xr, double yb, double yt,
                    bool lgx, bool lgy)
{
  logx = lgx;
  logy = lgy;
  if (logx) {
    xmin = log10(xl);
    xmax = log10(xr);
  } else {
    xmin = xl;
    xmax = xr;
  }
  if (logy) {
    ymin = log10(yb);
    ymax = log10(yt);
  } else {
    ymin = yb;
    ymax = yt;
  }
  fprintf(fps, "%g %g %g %g %s %s psscale\n", xl, xr, yb, yt,
          logx ? "true" : "false", logy ? "true" : "false");
}

void PSGraph::setDig(int nx, int ny)
{
  fprintf(fps, "/ndigx %i def\n", nx);
  fprintf(fps, "/ndigy %i def\n", ny);
}

void PSGraph::setColFrame(unsigned icol)
{
  fprintf(fps, "/colframe 16#%x def\n", Utils::rgbQt1(icol));
}

void PSGraph::setColGrid(unsigned icol)
{
  fprintf(fps, "/colgrid 16#%x def\n", Utils::rgbQt1(icol));
}

void PSGraph::setColData(unsigned icol)
{
  fprintf(fps, "/coldata 16#%x def\n",  Utils::rgbQt1(icol));
  fprintf(fps, "coldata scol\n");
}

void PSGraph::setCol(unsigned icol1, unsigned icol2, unsigned icol3)
{
  setColFrame(icol1);
  setColGrid(icol2);
  setColData(icol3);
}

void PSGraph::frame(void)
{
  fprintf(fps, "frame\n");
}

void PSGraph::raster(double xtic, double ytic, int intx, int inty, int mode)
{
  fprintf(fps, "%g %g %i %i %i raster\n", xtic, ytic, intx, inty, mode);
}

void PSGraph::text(const char* str, double x, double y, int iorg)
{
  psStr(str);
  fprintf(fps, " %g %g %i text\n", x, y, iorg);
}

void PSGraph::textcm(const char* str, double x, double y, int iorg)
{
  text(str, pts * x, pts * y, iorg);
}

void PSGraph::letH(const char* str)
{
  psStr(str);
  fprintf(fps, " letH\n");
}

void PSGraph::letX(const char* str, int iex)
{
  psStr(str);
  fprintf(fps, " %i letX\n", iex);
}

void PSGraph::letY(const char* str, int iey)
{
  psStr(str);
  fprintf(fps, " %i letY\n", iey);
}

void PSGraph::setSymbol(int is)
{
  kSymb = is;
  fprintf(fps, "/symbol %i def\n", is);
}

#define NMAX 127
void PSGraph::plArray(const double* x, const double* y,
                      double fx, double fy, int n)
{
  if (n > 0) {
    int ie = 0;
    int ia;
    do {
      ia = QMIN(n - 1, ie + NMAX);
      fprintf(fps, "\n");
      for (int i = ia; i >= ie; i--)
        fprintf(fps, "%g %g\n", x[i], y[i]);
      fprintf(fps, "%g %g %i plarry\n", fx, fy, ia - ie + 1);
      if (kSymb > 0)
        ie = ia;
      else
        ie = ia + 1;
    } while (ia != (n - 1));
  }
}

void PSGraph::plError(const double* x, const double* y, const double* e,
                      double fx, double fy, int n)
{
  if (n > 0) {
    int ie = 0;
    int ia;
    do {
      ia = QMIN(n - 1, ie + NMAX);
      fprintf(fps, "\n");
      for (int i = ia; i >= ie; i--)
        fprintf(fps, "%g %g %g\n", x[i], y[i], e[i]);
      fprintf(fps, "%g %g %i plerry\n", fx, fy, ia - ie + 1);
      ie = ia + 1;
    } while (ia != (n - 1));
  }
}
#undef NMAX

void PSGraph::psExit(void)
{
  fprintf(fps, "\nshowpage grestore\n\n%%%%Trailer\n%%%%EOF\n");
  fclose(fps);
}

void PSGraph::psStr(const char* str)
{
  fprintf(fps, "(");
  const unsigned char* s = (const unsigned char*) str;
  while (*s) {
    if (*s > 127)
      fprintf(fps, "\\%03o", *s);
    else
      switch (*s) {
        case 40:
        case 41:
        case 92:
          fprintf(fps, "\\%03o", *s);
          break;
        default:
          fprintf(fps, "%c", *s);
      }
    s++;
    }
  fprintf(fps, ")");
}

void PSGraph::cm2r(double xcm, double ycm, double fxn, double fyn,
                   double* xr, double* yr) const
{
  *xr = xmin + (xcm - x0) * fxn * (xmax - xmin) / xl;
  if (logx)
    *xr = pow(10, *xr);
  *yr = ymin + (ycm - y0) * fyn * (ymax - ymin) / yl;
  if (logy)
    *yr = pow(10, *yr);
}
