// kviewer.cpp
// Author: Markus Wuebben <markus.wuebben@kde.org>
// This code is published under the GPL.
// March 1998

#include "krfbproto.h"
#include "kvncauthwdg.h"
#include "kviewer.h"
#include <assert.h>
#include <qimage.h>

KApplication *app;

KVNCCanvas::KVNCCanvas(QWidget *parent, const char *name)
  : QWidget(parent,name) {

 
  rfb = 0L;
  mask = 0L; // We will keep the button mask in case of a drag
  setHost("");
  setDisplay(-1);

  timer = new QTimer(this);
  connect(timer,SIGNAL(timeout()),this,SLOT(handleTimeout()));
  refreshing = false;

  resize(640,480);
  setBackgroundColor(white);
  setFocusPolicy(QWidget::StrongFocus);
  setMouseTracking(true);
  setConnected(false);
  setUpdatesEnabled(true);

}

void KVNCCanvas::handleTimeout() {

  if(!connected())
    return;
  
  if(!rfb)
    return;

  if(!refreshing) {
    if(!rfb->writeFramebufferUpdateRequest(0,0,width(),
					   height(), true))
      fatal();
  }

}

void KVNCCanvas::setRefresh(bool _re) {

  refreshing = _re;
}


KVNCCanvas::~KVNCCanvas() {

}

bool KVNCCanvas::openConnection() {
  
  QString temp;
  QDialog *widget = new QDialog(parentWidget(),0L,false);
  QPixmap startXpm;
  if(connected()) {
    KMsgBox::message(0L,0L,"Already conected!\nPlease disconnect first!\n");
    return false;
  }
  
  KVNCConnectDlg *dlg = new KVNCConnectDlg(0L,0L,this);
  emit updateStatusBar("Connecting to host...",0);
  if(!dlg->exec()) {
    emit updateStatusBar("",0);
    delete rfb;
    rfb = 0L;
    return false;
  }
  else {
    //temp.sprintf("Connected to %s. Initializing connection...",host());
    //emit updateStatusBar(temp,0);
    if(display() < 0) {
      KMsgBox::message(0L,"kvncviewer","You need to specify a display!");
      return false;
    }
    temp.sprintf("%s",host());
    if(temp.isEmpty()) {
      KMsgBox::message(0L,"kvncviewer","You need to specify a host!");
      return false;
    }

    if(!initConnection()) {
      KMsgBox::message(0L,"kvncviewer","Failed to connect to host");
      emit updateStatusBar("",0);
      delete rfb;
      rfb = 0L;
      widget->hide();
      delete widget;
      widget = 0L;
      return false;
    }    
  }

  widget->hide();
  delete widget;
  widget = 0L;

  setConnected(true);
  emit updateStatusBar("",0);
  return true;
}


bool KVNCCanvas::initConnection() {

  // Initialize connection to server
  rfb = new kRfbProto(host(),display() + 5900);
  if(rfb->connect() < 0 ) 
     return false;

  // Read server version
  rfb->readVersionMsg();

  // Write Client Version
  rfb->writeVersionMsg();

  // Authentification 
  if(rfb->readAuthScheme() ==  kRfbProto::VncAuth) {
    cout << "VNC challenges authentification\n";
    emit updateStatusBar("VNC challenges authentification",0);
    switch(authenticate()) {

    case kRfbProto::VncAuthOK:
      break;

    case kRfbProto::VncAuthFailed:
      KMsgBox::message(0L,"Server login...","Authentification failed!");
      closeConnection();
      return false;

    case kRfbProto::VncAuthTooMany:
      KMsgBox::message(0L,"Server login...","Too many connection attempts");
      closeConnection();
      return false;

    case kRfbProto::VncAuthUnknown:
      KMsgBox::message(0L,"Server login...",
			 "Unknown authentification scheme!");
      closeConnection();
      return false;

    case -1 :
      KMsgBox::message(0L,"Warning!",
		       "A serious error occured.\nPlease reconnect!");
      closeConnection();
      return false;

    case -2 : // Cancel pressed
      cout << "cancel pressed\n";
      closeConnection();
      return false;
      
    default:
      KMsgBox::message(0L,"Server login...","Unknown error!");
      closeConnection();
      return false;

    }
      
  }

  emit updateStatusBar("Initializing client and server...",0);

  if(!rfb->writeClientInit())
    return FALSE;

  if(!rfb->readServerInit())
    return FALSE;
  
  // Hi Markus!
  // Some notes so you can see what I've done:
  // Rather than allowing the server to choose the pixel format
  // I've made it request a standard 24 bit colour encoding - this
  // is dead easy to deal with, but of course we should really add
  // support for the other encodings later. For now however we know
  // that the protocol allows the client to set the encoding and pixel
  // formats so we're safe doing this.
  // 
  // Rich.

  // The argument is now ignored
  if(!rfb->writePixelFormat(rfb->getServerInformation()))
    return FALSE;
  bpp = rfb->bpp();

  if(bpp < 8) {
    KMsgBox::message(0,"KVNCViewer","Invalid bits per pixel");
    return FALSE;
  }

  if(!rfb->writeEncodings())
    return FALSE;

  initCanvas();

  rfbServerInitMsg *s = rfb->getServerInformation();
  
  screenPixmap.resize(s->framebufferWidth, s->framebufferHeight);
  //  if (!screenImage.create(s->framebufferWidth, s->framebufferHeight, 32))
  //fatal();
  //screenImage.fill(0);
  resize(s->framebufferWidth, s->framebufferHeight);
  emit sizeChanged();

  assert(s);

  // Connect some SIGNALs
  connect(rfb,SIGNAL(sendData(CARD8*,int,int,int,int)),this,
	      SLOT(drawRawRect(CARD8*,int,int,int,int)));

  connect(rfb,SIGNAL(bell()),this,SLOT(bell()));

  connect(rfb,SIGNAL(finished()),this,SLOT(displayData()));
	
  connect(rfb,SIGNAL(fatal()),this,SLOT(fatal()));

  connect(rfb,SIGNAL(refresh(bool)),this,SLOT(setRefresh(bool))); 

  connect(rfb,SIGNAL(copyRect(int,int,int,int,int,int)),this,
	  SLOT(copyRect(int,int,int,int,int,int)));
  
  connect(rfb,SIGNAL(fillRect(int,int,int,int,unsigned long)),this,
	  SLOT(fillRect(int,int,int,int,unsigned long)));

  // First frameBuffer update
  if(!rfb->writeFramebufferUpdateRequest(0,0,s->framebufferWidth,
				    s->framebufferHeight, true))
    return false;

  // we check for data every 1 milisec
  timer->start(1);
  return TRUE;

}

void KVNCCanvas::setConnected(bool c) {

  _connected = c;
  

}

void KVNCCanvas::refreshDisplay() {

  if(!rfb)
    return;

  if(!rfb->writeFramebufferUpdateRequest(0,0,width(),
				    height(), false))
    fatal();
}

int KVNCCanvas::authenticate() {

  KVNCAuthWdg *authWdg = new KVNCAuthWdg(0L,0L,this);

  if(!authWdg->exec()) 
    return (-2);

  else {
    switch(rfb->transmitPassword((char*)password())) {

    case kRfbProto::VncAuthOK:
      return  kRfbProto::VncAuthOK;

    case  kRfbProto::VncAuthFailed:
       return kRfbProto::VncAuthFailed;

    case  kRfbProto::VncAuthTooMany:
       return kRfbProto::VncAuthTooMany;

    case kRfbProto::VncAuthUnknown:
      return kRfbProto::VncAuthUnknown;

    case -1:
      return (-1);

    default:
      return (-1);
    }
  }

}

void KVNCCanvas::closeConnection() {

  if(!rfb)
    return;
  cout << "Closing down connection...\n";
  
  timer->stop();

  KConfig *config = app->getKApplication()->getConfig();
  config->setGroup("General");
  config->writeEntry("host",host());
  config->writeEntry("display",display());
  config->sync();

  disconnect(rfb,SIGNAL(sendData(CARD8*,int,int,int,int)),this,
	      SLOT(drawRawRect(CARD8*,int,int,int,int)));

  disconnect(rfb,SIGNAL(bell()),this,SLOT(bell()));

  disconnect(rfb,SIGNAL(finished()),this,SLOT(displayData()));
	      
  disconnect(rfb,SIGNAL(fatal()),this,SLOT(fatal())); 

  rfb->close();
  setConnected(false);
  
  delete rfb;
  rfb = 0L;

  screenPixmap.resize(0,0);
  tempPixmap.resize(0,0);
  //  screenImage.reset();

  setCursor(arrowCursor);
  mask = 0L;

  setBackgroundColor(white);
}

void KVNCCanvas::fatal() {

  closeConnection();
  emit sigFatal();
}

// Draw Raw data
void KVNCCanvas::drawRawRect(CARD8 *buf, int x, int y, int h, int len) {

  QImage img;
  uint *start;

  if (!img.create((len/h/4),h, 32)) 
    return;
  offW = (len/h/4);
    
  for (int i= 0; i < h; i++) {
    start= (uint *)img.scanLine(i);
    memcpy(start, buf+len/h*i, len/h);
  }
  offX = x;
  offY = y;
  offH = h;

  if(tempPixmap.convertFromImage(img) == FALSE) {
    cout << "covertFromImage failed..I have no clue why!\n";
    return;
  }
  
  bitBlt(&screenPixmap,offX,offY,&tempPixmap); 
  bitBlt(this, offX, offY, &tempPixmap,0,0,offW,offH);
  img.reset();
  
  //cout << "KVNCanvas::drawRawRect done...\n";
}


void KVNCCanvas::paintEvent(QPaintEvent *) {
  if (connected())  {
    bitBlt(this, 0, 0, &screenPixmap);
  }
}

// copy rect to new position
void KVNCCanvas::copyRect(int srcX, int srcY, int destX, int destY, int w, int h) {

  //printf("srcx: %d srcy: %d destx: %d desty: %d w: %d h: %d\n",
  // srcX,srcY,destX,destY,w,h);

  // Resize temp pixmap and only update this rectangle on canvas
  tempPixmap.resize(w,h);
  bitBlt(&tempPixmap,0,0,&screenPixmap,srcX,srcY,w,h);

  offX = destX;
  offY = destY;
  offW = w;
  offH = h;

  // Always keep a copy of the latest screen contents in a buffer
  bitBlt(&screenPixmap,destX,destY,&screenPixmap,srcX,srcY,w,h);
  bitBlt(this, offX, offY, &tempPixmap,0,0,offW,offH);
  tempPixmap.resize(0,0);

}

// used by RRE CoREE and hextile encoding

void KVNCCanvas::fillRect(int x, int y, int w, int h, unsigned long value) {

  QImage img;

  if(!img.create(w,h,32)) {
    cout << "Creating 32 bit image failed\n";
    return;
  }
  img.fill(value);

  tempPixmap.resize(w,h);
  tempPixmap.convertFromImage(img);
  
  bitBlt(&screenPixmap,x,y,&tempPixmap);
  bitBlt(this,x,y,&tempPixmap,0,0,w,h);

  tempPixmap.resize(0,0);
  img.reset();

}


void KVNCCanvas::displayData() {


}



rfbServerInitMsg* KVNCCanvas::getServerInformation() {

  return rfb->getServerInformation();

}



void KVNCCanvas::initCanvas() {

  rfbServerInitMsg *initMsg = getServerInformation();

  assert(initMsg);
  
  assert(rfb);

  depth = initMsg->format.bitsPerPixel;
  frameWidth = initMsg->framebufferWidth;
  frameHeight = initMsg->framebufferHeight;
  setMaximumSize(frameWidth,frameHeight); // dont resize larger than size of virtual screen

  createCursor();

}

void KVNCCanvas::createCursor() {
  
  const uchar srcBits[] = { 0,0,14,14,14,0,0 };
  const uchar mskBits[] = { 31,31,31,31,31,31,31 };

  const QBitmap src(7,7,srcBits);
  const QBitmap msk(7,7,mskBits);

  QCursor curs(src,msk);
  
  setCursor(curs);
  
}




void KVNCCanvas::keyPressEvent(QKeyEvent *k) {

  KeySym ks;

  if(!rfb)
    return;

  if(!(ks  = determineKey(k))) {
    cout << "Unknown key!\n";
    return;
  }
    
  /*  printf("keyPressed(QT): 0x%02x \n", k->key());
  printf("keyPressed(X): 0x%02x \n",(uint)ks);
  printf("keyPressedAscii: %c\n", k->ascii());
  */

  if(!rfb->writeKeyEvent(ks,true))
    fatal();
}

void KVNCCanvas::keyReleaseEvent(QKeyEvent *k) {

  KeySym ks;

  if(!rfb)
    return;

  //  printf("keyReleased(QT): 0x%02x \n", k->key());
  // printf("keyReleasedAscii: %c\n", k->ascii());
  
 
  if(!(ks  = determineKey(k))) {
    cout << "Unknown key!\n";
    return;
  }
  
  //  printf("keyReleased(X): 0x%02x \n", (uint)ks);
  

  if(!rfb->writeKeyEvent(ks,false))
    fatal();

}

KeySym KVNCCanvas::determineKey(QKeyEvent *k) {


  int ke = 0;

  ke = k->ascii();
  // Markus: Crappy hack. I dont know why lower case letters are not defined in qkeydefs.h. 
  // The key() for e.g. 'l' == 'L'. 
  // This sucks. :-(

  if((ke == 'a') ||(ke == 'b') ||(ke == 'c') ||(ke == 'd') ||(ke == 'e') ||(ke == 'f') ||(ke == 'g') ||
     (ke == 'h') ||(ke == 'i') ||(ke == 'j') ||(ke == 'k') ||(ke == 'l') ||(ke == 'm') ||(ke == 'n') ||
     (ke == 'o') ||(ke == 'p') ||(ke == 'q') ||(ke == 'r') ||(ke == 's') ||(ke == 't') ||(ke == 'u') ||
     (ke == 'v') ||(ke == 'w') ||(ke == 'x') ||(ke == 'y') ||(ke == 'z')) {
    //cout << "Converting key!\n";
    ke = k->key();
    ke = ke + 0x20;
    return ke;
  }

 // qkeydefs = xkeydefs! :-)
  if((k->key() >= 0x0a0) && k->key() <= 0x0ff)
    return k->key();

  if((k->key() >= 0x20) && (k->key() <= 0x7e))
    return k->key();
  
  
  // qkeydefs != xkeydefs! :-(
  // This is gonna suck :-(

  switch(k->key()) {
  case SHIFT:
    return XK_Shift_L;
  case CTRL:
    return XK_Control_L;
  case ALT:
    return XK_Alt_L;

  case Key_Escape:
    return  XK_Escape;
  case Key_Tab:
    return XK_Tab;
  case Key_Backspace:
    return XK_BackSpace;
  case Key_Return:
    return XK_Return;
  case Key_Enter:
    return XK_Return;
  case Key_Insert:
    return XK_Insert;
  case Key_Delete:
    return XK_Delete;
  case Key_Pause:
    return XK_Pause;
  case Key_Print:
    return XK_Print;
  case Key_SysReq:
    return XK_Sys_Req;
  case Key_Home:
    return XK_Home;
  case Key_End:
    return XK_End;
  case Key_Left:
     return XK_Left;
  case Key_Up:
     return XK_Up;
  case Key_Right:
     return XK_Right;
  case Key_Down:
     return XK_Down;
  case Key_Prior:
     return XK_Prior;
  case Key_Next:
     return XK_Next;

  case Key_Shift:
    return XK_Shift_L;
  case Key_Control:
    return XK_Control_L;
  case Key_Meta:
    return XK_Meta_L;
  case Key_Alt:
    return XK_Alt_L;
  case Key_CapsLock:
    return XK_Caps_Lock;
  case Key_NumLock:
    return XK_Num_Lock;
  case Key_ScrollLock:
    return XK_Scroll_Lock;

  case Key_F1:
    return XK_F1;
  case Key_F2:
    return XK_F2;
  case Key_F3:
    return XK_F3;
  case Key_F4:
    return XK_F4;
  case Key_F5:
    return XK_F5;
  case Key_F6:
    return XK_F6;
  case Key_F7:
    return XK_F7;
  case Key_F8:
    return XK_F8;
  case Key_F9:
    return XK_F9;
  case Key_F10:
    return XK_F10;
  case Key_F11:
    return XK_F11;
  case Key_F12:
    return XK_F12;
  case Key_F13:
    return XK_F13;
  case Key_F14:
    return XK_F14;
  case Key_F15:
    return XK_F15;
  case Key_F16:
    return XK_F16;
  case Key_F17:
    return XK_F17;
  case Key_F18:
    return XK_F18;
  case Key_F19:
    return XK_F19;
  case Key_F20:
    return XK_F20;
  case Key_F21:
    return XK_F21;
  case Key_F22:
    return XK_F22;
  case Key_F23:
    return XK_F23;
  case Key_F24:
    return XK_F24;

  case Key_unknown:
    return 0;
  default:
    return 0;
  }

  // Puhhhhh done. :-)

    
  return 0;
  
}


void KVNCCanvas::mousePressEvent(QMouseEvent *mouse) {

  
  //  cout << "mousePressEvent with button  " << mouse->button() 
  //   << " at (" << mouse->x() << "," << mouse->y() << ")" << endl;

  if(!rfb)
    return;
  
  switch (mouse->button()) {
  case LeftButton:
    leftButtonDown = true;
    mask |= LeftButtonBits;
    break;
  case RightButton:
    rightButtonDown = true;
    mask |= RightButtonBits;
    break;
  case MidButton:
    midButtonDown = true;
    mask |= MidButtonBits;
    break;
  default:
    cout << "Unknown button pressed: " << mouse->button() << endl;
    return;
  }

  if(!rfb->writePointerEvent(mask,mouse->x(),mouse->y()))
    fatal();
  
}


void KVNCCanvas::mouseReleaseEvent(QMouseEvent *mouse) {

  //cout << "mouseReleaseEvent with button: " << mouse->button() << endl;

  if(!rfb)
    return;
  
  switch (mouse->button()) {
  case 1: {
    leftButtonDown = false;
    break;
  }
  case 2:
    rightButtonDown = false;
    break;
  case 3:
    midButtonDown = false;
    break;
  default:
    cout << "Unknown button released: " << mouse->button() << endl;
    return;
  }

  mask = 0; // zero mask cause no button will be pressed anymore
    
  if(!rfb->writePointerEvent(mask,mouse->x(),mouse->y()))
    fatal();

}

void KVNCCanvas::mouseMoveEvent(QMouseEvent *mouse) {

  int x,y;

  // Return if not connected
  if(!rfb)
    return;

  x = mouse->x();
  y = mouse->y();
  
  // We dont want drags outside the display
  if(x < 0)
    x = 0;
  if(x > frameWidth)
    x = frameWidth;
  if(y < 0)
    y = 0;
  if(y > frameHeight)
    y = frameHeight;

  if(!rfb->writePointerEvent(mask,x,y))
    fatal();


}

void KVNCCanvas::bell() {
  
  XBell(this->x11Display(),100);
  if(!rfb) 
    return;

  if(rfb->deiconifyOnBell())
    ((KVNCViewer*)parentWidget())->show();
  

}



KVNCViewer::KVNCViewer(QWidget *, const char *name)
  :KTopLevelWidget(name) {

  viewport= new QwViewport(this);
  canvas = new KVNCCanvas(viewport->portHole(),name);

  viewport->resize(canvas->size());
  connect(canvas, SIGNAL(sizeChanged()),
	  viewport, SLOT(resizeScrollBars()));    
  setView(viewport);
      
  setupMenuBar();
  setupToolBar();
  setupStatusBar();
  setConnected(false);
  initConfig();
  connect(canvas,SIGNAL(updateStatusBar(const char*,int)),this,
	  SLOT(updateStatusBar(const char* , int)));
  connect(canvas,SIGNAL(sigFatal()),this,SLOT(fatal()));

}

KVNCViewer::~KVNCViewer() {
  
  delete menuBar;
  delete statusBar;
}

void KVNCViewer::initConfig() {

  QString host;
  int display;
  KConfig *config = app->getKApplication()->getConfig();
  config->setGroup("General");
  host = config->readEntry("host","localhost");
  display = config->readNumEntry("display",-1);
  canvas->setHost(host);
  canvas->setDisplay(display);

}

void KVNCViewer::setupMenuBar() {

  KMenuBar *mB = new KMenuBar(this);
  QPopupMenu *pop = new QPopupMenu();

  pop->insertItem("&Connect...",this,SLOT(openConnection()));
  pop->insertItem("&Disconnect...",this,SLOT(closeConnection()));
  pop->insertItem("&Quit",this,SLOT(close()));
  mB->insertItem("&Connection",pop);

  pop = new QPopupMenu();
  pop->insertItem("&Information",this,SLOT(serverInformation()));
  pop->insertItem("&Refresh",canvas,SLOT(refreshDisplay()));
  mB->insertItem("&Options",pop);

  QString aboutTxt = "K VNC Viewer 0.0.3\n\n";
  aboutTxt += "A VNC Viewer for KDE\nWritten by Markus Wuebben\n";
  aboutTxt += "<markus.wuebben@kde.org>\n\n";
  aboutTxt += "Special thanks to Rich Moore\n <rich@kde.org>\n\n (c)August 1998";

  pop = app->getHelpMenu(0L,aboutTxt);
  mB->insertItem("&Help",pop);
  
  menuBar = mB;
  setMenu(menuBar);
  
}

void KVNCViewer::setupToolBar() {

}

void KVNCViewer::updateStatusBar(const char *text,int id) {
  
  statusBar->changeItem(text,id);
}

void KVNCViewer::setupStatusBar() {

  KStatusBar *bar = new KStatusBar(this);
  bar->setInsertOrder(KStatusBar::RightToLeft);
  bar->insertItem(" Disconnected  ",1);
  bar->insertItem("",0);
  statusBar = bar;
  setStatusBar(statusBar);

}


bool KVNCViewer::openConnection() {

  if(connected()) {
    KMsgBox::message(0L,0L,"Already conected!\nPlease disconnect first!\n");
    return false;
  }

  if(!canvas->openConnection()) {
    setConnected(false);
    return false;
  }
  
  setConnected(true);
  return true;
}


void KVNCViewer::setConnected(bool c) { 

  QString cap;
  _connected = c;
  if(c) {
    updateStatusBar(" Connected     ",1);
    cap.sprintf("K VNC Viewer: %s",canvas->desktop());
    setCaption(cap);
  }
  else {
    updateStatusBar(" Disconected  ",1);
    cap = "K VNC Viewer: --No connection--";
    setCaption(cap);
  }

} 

void KVNCViewer::fatal() {

  KMsgBox::message(0,"Network Error!","A network error occured\nClosing down connection\n");
  closeConnection();

}


void KVNCViewer::closeConnection() {
  
  if(!connected())
    return;
  canvas->closeConnection();
  setConnected(false);
 
}


void KVNCViewer::close() {

  closeConnection();
  app->quit();
} 


void KVNCViewer::serverInformation() {

  QString text;
  if(!connected())
    return;
  rfbServerInitMsg *info;
  info = canvas->getServerInformation();
  text.sprintf("Host: %s\nPort: %i\nDesktop Name: %s\nDepth: %i bits\n\
Width: %i\nHeight: %i\nRed Max %i\nGreen Max: %i\nBlue Max :%i\n", 
	       canvas->host(),canvas->port(),canvas->desktop(),
	       info->format.bitsPerPixel,
	       info->framebufferWidth, info->framebufferHeight,
	       info->format.redMax,info->format.greenMax,
	       info->format.blueMax);
  KMsgBox::message(0L,0L,text);
  
}

#include "kviewer.moc"

main(int argc, char **argv) {
  QPixmap::setDefaultOptimization(QPixmap::NoOptim);
  app = new KApplication(argc,argv,"kvncviewer");
  KVNCViewer * view = new KVNCViewer;
  app->setMainWidget(view);
  view->show();
  return app->exec();
}
  













