    /*

    tspgda.C  for  ktsp-0.1.0

    Copyright (C) 1999 Uwe Thiem
                       uwe@kde.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    */



#include <iostream.h>
#include <stdlib.h>
#include <values.h>
#include <math.h>
#include <kapp.h>

#include "tspgda.h"
#include "random.h"
#include "ktsp.h"



TspGda::TspGda() :
     Gda()
  {
  io = new TspIO();
  CHECK_PTR( io );
  rand = new Random();
  CHECK_PTR( rand );
  points = 0;
  actualPoints = 0;
  bestPoints = 0;
  _access = 0;
  _indeces = 0;
  _distances = 0;
  _numberOfPoints = _confNumberOfPoints = 300;
  _numberOfNeighbours = _confNumberOfNeighbours = 10;
  resumedBestName = 0;
  }


TspGda::~TspGda()
  {
  }


void TspGda::startTsp( int iterations )
  {
  QString s;
  startTsp( s, iterations, GENERATE_TSP );
  }


void TspGda::resumeTsp( QString orig, QString best,
                        int iterations, int performedIterations,
                        double bestQuality )
  {
  resumedBestName = best;
  resumedBestQuality = bestQuality;
  itStart = performedIterations;
  startTsp( orig, iterations, RESUME_TSP );
  }


void TspGda::startTsp( QString name, int iterations, int mode )
  {
  stopped = false;
  switch( mode )
    {
    case GENERATE_TSP:
      checkOldTsp();
      emit statusChanged( KTsp::STATUSBAR_GENERATING );
      points = io->generateTsp( _confNumberOfPoints, _confNumberOfNeighbours );
      itStart = 0;
      break;
    case LOAD_TSP:
      {
      emit statusChanged( KTsp::STATUSBAR_LOADING );
      kapp->processEvents();
      Point *pTemp = io->loadTsp( name, _confNumberOfNeighbours );
      if ( pTemp == 0 )
        {
        emit statusChanged( KTsp::STATUSBAR_LOAD_ERROR );
        return;
        }
      checkOldTsp();
      points = pTemp;
      }
      itStart = 0;
      break;
    case RESUME_TSP:
      {
      emit statusChanged( KTsp::STATUSBAR_RESUMING );
      kapp->processEvents();
      Point *pTemp = io->loadTsp( name, _numberOfNeighbours, true );
      if ( pTemp == 0 )
        {
        emit statusChanged( KTsp::STATUSBAR_RESUME_ERROR );
        return;
        }
      checkOldTsp();
      points = pTemp;
      }
      break;
    }
  _numberOfPoints = io->numberOfPoints();
  emit pointsNumber( _numberOfPoints );
  _numberOfNeighbours = io->numberOfNeighbours();
  emit neighboursNumber( _numberOfNeighbours );
  if ( ( _numberOfPoints < 10 ) || ( _numberOfNeighbours < 3 ) )
    {
    cerr << "Not a valid TSP.\n";
    cerr << "Points = " << _numberOfPoints;
    cerr << " Neighbours = " << _numberOfNeighbours << "\n";
    exit (1);
    }
  emit statusChanged( KTsp::STATUSBAR_NEIGHBOURS );
  calcNeighbours();
  if ( stopped ) return;
  emit statusChanged( KTsp::STATUSBAR_OPTIMIZING );
  actualPoints = new Point[_numberOfPoints];
  CHECK_PTR( actualPoints );
  copyPoints( actualPoints, points );
  if( !resumedBestName.isEmpty() )
    {
    bestPoints = io->loadTsp( resumedBestName,
                              _numberOfNeighbours, true );
    }
  if( bestPoints == 0 )
    {
    bestPoints = new Point[_numberOfPoints];
    CHECK_PTR( bestPoints );
    copyPoints( bestPoints, points );
    }
  emit coursesReady();
  _access = new int[_numberOfPoints];
  CHECK_PTR( _access );
  if ( resumedBestName.isEmpty() )
    {
    _bestQuality = calcQuality();
    }
  else
    {
    _bestQuality = resumedBestQuality;
    }
  _bestRun = 0;
  emit currentRun( itStart );
  emit bestRun( _bestRun );
  emit bestQuality( _bestQuality );
  calcScale();
  for ( int it = itStart; it < iterations; it++ )
    {
    actPoint = 0;
    actNeighbour = 0;
    copyPoints( actualPoints, points );
    for ( int i = 0; i < _numberOfPoints; i ++ )
      {
      _access[i] = i;
      }
    for ( int i = 0; i < 10 * _numberOfPoints; i++ )
      {
      int temp;
      long i1, i2;
      i1 = rand->lrand( 0, _numberOfPoints - 1);
      i2 = rand->lrand( 0, _numberOfPoints - 1);
      temp = _access[i1];
      _access[i1] = _access[i2];
      _access[i2] = temp;
      }
    emit newRun();
    startGda( scaleFactor, calcQuality(), _numberOfPoints * _numberOfNeighbours );
    if ( stopped ) return;
    DBL tempQuality = calcQuality();
    emit currentQuality( tempQuality );
    if ( tempQuality < _bestQuality )
      {
      _bestQuality = tempQuality;
      copyPoints( bestPoints, actualPoints );
      _bestRun = it + 1;
      emit bestRun( _bestRun );
      emit bestQuality( _bestQuality );
      }
    emit currentRun( it + 1 );
    }
  emit statusChanged( KTsp::STATUSBAR_READY );
  emit tspReady( _bestRun );
  }


bool TspGda::saveTsp( QString name, int which )
  {
  Point *p = 0;
  switch ( which )
    {
    case KTsp::ORIG_TSP:
      p = points;
      break;
    case KTsp::CURRENT_TSP:
      p = actualPoints;
      break;
    case KTsp::BEST_TSP:
      p = bestPoints;
      break;
    }
  emit statusChanged( KTsp::STATUSBAR_SAVING );
  if ( !io->saveTsp( name, p, _numberOfPoints ) )
    {
    emit statusChanged( KTsp::STATUSBAR_SAVE_ERROR );
    return false;
    }
  else
    {
    emit statusChanged( KTsp::STATUSBAR_READY );
    return true;
    }
  }


void TspGda::calcNeighbours()
  {
  _indeces = new int[_numberOfPoints];
  _distances = new DBL[_numberOfPoints];
  if ( ( _indeces == 0 ) || ( _distances == 0 ) )
    {
    cerr << "Out of memory in TspGda::calcNeighbours.\n";
    exit ( 1 );
    }
  emit progressNeighbours( 0 );
  for ( int i = 0; i < _numberOfPoints; i++ )
    {
    kapp->processEvents();
    if ( stopped ) return;
    sortNeighbours( i );
    for ( int t = 0; t < _numberOfNeighbours; t++ )
      {
      points[i].setNeighbour( t, _indeces[_numberOfNeighbours - t - 1] );
      }
    emit progressNeighbours( i + 1 );
    }
  delete[] _indeces;
  _indeces = 0;
  delete[] _distances;
  _distances = 0;
  }


void TspGda::sortNeighbours( int index )
  {
  DBL td;
  int ti;

  for ( int i = 0; i < _numberOfPoints; i++ )
    {
    if ( stopped ) return;
    _indeces[i] = i;
    if ( i == index )
      {
      _distances[i] = MAXDBL;
      }
    else
      {
      _distances[i] = sqrt( ( points[i].x() - points[index].x() ) *
                            ( points[i].x() - points[index].x() ) +
                            ( points[i].y() - points[index].y() ) *
                            ( points[i].y() - points[index].y() ) );
      }
    }

  for ( int i = 0; i < _numberOfPoints - 1; i++ )
    {
    for ( int t = i; t < _numberOfPoints; t++ )
      {
      if ( stopped ) return;
      if ( _distances[t] < _distances[i] )
        {
        if ( stopped ) return;
        td = _distances[t];
        _distances[t] = _distances[i];
        _distances[i] = td;
        ti = _indeces[t];
        _indeces[t] = _indeces[i];
        _indeces[i] = ti;
        }
      }
    }
  }



DBL TspGda::calcQuality()
  {
  DBL result = 0.0;
  DBL tempX, tempY;
  Point *p = actualPoints;
  for ( int i = 0; i < _numberOfPoints; i++ )
    {
    tempX = p->x() - p->nextPointer()->x();
    tempY = p->y() - p->nextPointer()->y();
    result += sqrt( tempX * tempX + tempY * tempY );
    p = p->nextPointer();
    }
  return result;
  }


DBL TspGda::nextAction()
  {
  int actP = _access[actPoint];
  Point *cpPointer = &(actualPoints[actP]);
  Point *cpnPointer = cpPointer->nextPointer();
  Point *cnPointer = &(actualPoints[cpPointer->neighbour( actNeighbour )]);
  Point *cnnPointer = cnPointer->nextPointer();
  DBL tempX = cpPointer->x() - cpnPointer->x();
  DBL tempY = cpPointer->y() - cpnPointer->y();
  DBL oldPart = sqrt( tempX * tempX + tempY * tempY );
  tempX = cnPointer->x() - cnnPointer->x();
  tempY = cnPointer->y() - cnnPointer->y();
  oldPart += sqrt( tempX * tempX + tempY * tempY );
  tempX = cpPointer->x() - cnPointer->x();
  tempY = cpPointer->y() - cnPointer->y();
  DBL newPart = sqrt( tempX * tempX + tempY * tempY );
  tempX = cpnPointer->x() - cnnPointer->x();
  tempY = cpnPointer->y() - cnnPointer->y();
  newPart += sqrt( tempX * tempX + tempY * tempY );
  oldPoint = actPoint++;
  oldNeighbour = actNeighbour;
  if ( actPoint >= _numberOfPoints )
    {
    actPoint = 0;
    oldNeighbour = actNeighbour++;
    if ( actNeighbour >= _numberOfNeighbours ) actNeighbour = 0;
    }
  return oldPart - newPart;
  }


void TspGda::acceptAction()
  {
  int cpIndex, cnIndex;
  int cpnIndex, cnnIndex;
  Point *cpPointer, *cnPointer;
  Point *cpnPointer, *cnnPointer;

  // First we we initialize some indeces and pointers with:
  // cp[Index|Pointer]       - the current point
  // cpn[Index|Pointer]      - the next point to the current point
  // cn[Index|Pointer]       - the current neighbour
  // cnn[Index|Pointer]      - the next point to the current neighbour
  cpIndex = _access[oldPoint];
  cpPointer = &(actualPoints[cpIndex]);
  cpnIndex = cpPointer->nextIndex();
  cpnPointer = cpPointer->nextPointer();
  cnIndex = cpPointer->neighbour( oldNeighbour );
  cnPointer = &(actualPoints[cnIndex]);
  cnnIndex = cnPointer->nextIndex();
  cnnPointer = cnPointer->nextPointer();

  // If actual point and actual neighbour are directly after (or before)
  // each other in the course the new course would dublicate the old one,
  // but our data structure would become corrupted. So we stop here in
  // this case.
  if ( ( cpnIndex == cnIndex ) || ( cnnIndex == cpIndex ) )
      return;

  // Now we cut apart the old connections and build the new ones.
  // First new connection: cpIndex -> cnIndex. This one must be done
  // with the direction.
  cpPointer->setNextPointer( cnPointer );
  cpPointer->setNextIndex( cnIndex );

  // Next connection: cpnIndex -> cnnIndex. This one must be done
  // against the direction.
  cpnPointer->setLastPointer( cnnPointer );
  cpnPointer->setLastIndex( cnnIndex );

  // Next connection: cnnIndex -> cpnIndex. This one must also be done
  // against the direction.
  cnnPointer->setLastPointer( cpnPointer );
  cnnPointer->setLastIndex( cpnIndex );

  // Last connection: cnIndex -> cpIndex. This one must be done with
  // the direction again.
  cnPointer->setNextPointer( cpPointer );
  cnPointer->setNextIndex( cpIndex );

  // All new connections are done. Now we have to invert the direction
  // of one of the two part of our course. It would be good if we could
  // know (or even make a good guess) which is the shorter one.
  // Unfortunately we don't know any efficient algorithm to find that out.
  // If someone knows one drop us a note. For the time being we always
  // choose the same one: cpnIndex up to cnIndex (including them).
  Point *tmp;
  do {
    tmp = cpnPointer->nextPointer();
    cpnPointer->flipDirection();
    cpnPointer = tmp;
    } while ( cpnPointer != cnPointer );
  // That leaves just the last direction to invert.
  cpnPointer->flipDirection();

  // So ENIAC agrees, everything is done!
  }



void TspGda::copyPoints( Point *target, Point *source )
  {
  Point *t = target;
  Point *s = source;
  int tmp;
  for ( int i = 0; i < _numberOfPoints; i++ )
    {
    t->setX( s->x() );
    t->setY( s->y() );
    tmp = s->forwardI();
    t->setForwardI( tmp );
    t->setForwardP( &(target[tmp]) );
    tmp = s->backwardI();
    t->setBackwardI( tmp );
    t->setBackwardP( &(target[tmp]) );
    t->setDirectionForward( s->directionForward() );
    t->setNumberOfNeighbours( _numberOfNeighbours );
    for ( int n = 0; n < _numberOfNeighbours; n++ )
      {
      t->setNeighbour( n, s->neighbour( n ) );
      }
    t++;
    s++;
    }
  }



void TspGda::checkOldTsp()
  {
  if ( points != 0 )
    {
    delete[] points;
    points = 0;
    }
  if ( actualPoints != 0 )
    {
    delete[] actualPoints;
    actualPoints = 0;
    }
  if ( bestPoints != 0 )
    {
    delete[] bestPoints;
    bestPoints = 0;
    }
  if ( _access != 0 )
    {
    delete[] _access;
    _access = 0;
    }
  emit noCoursesReady();
  }


void TspGda::calcScale()
  {
  DBL xmin = MAXDBL;
  DBL xmax = MINDBL;
  DBL ymin = MAXDBL;
  DBL ymax = MINDBL;
  Point *p = points;
  for ( int i = 0; i < _numberOfPoints; i++ )
    {
    if ( p->x() < xmin ) xmin = p->x();
    if ( p->x() > xmax ) xmax = p->x();
    if ( p->y() < ymin ) ymin = p->y();
    if ( p->y() > ymax ) ymax = p->y();
    p++;
    }
  scaleFactor = sqrt( ( xmax - xmin ) * ( xmax - xmin ) +
                      ( ymax - ymin ) * ( ymax - ymin ) );
  }


Point *TspGda::getPoints( int id )
  {
  Point *tmp;
  switch( id )
    {
    case KTsp::ORIG_TSP:
      tmp = points;
      break;
    case KTsp::CURRENT_TSP:
      tmp = actualPoints;
      break;
    case KTsp::BEST_TSP:
      tmp = bestPoints;
      break;
    default:
      // Oops! Shouldn't happen.
      tmp = 0;
      break;
    }
  return tmp;
  }


int TspGda::numberOfPoints()
  {
  return _confNumberOfPoints;
  }


int TspGda::numberOfNeighbours()
  {
  return _confNumberOfNeighbours;
  }


void TspGda::setNumberOfPoints( int number )
  {
  _confNumberOfPoints = number;
  }


void TspGda::setNumberOfNeighbours( int number )
  {
  _confNumberOfNeighbours = number;
  }


