// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_WORKAREA_H
#define KROT_WORKAREA_H



#include "krotData.h"

#include <qwidget.h>

#include <sys/types.h>


class AssignedFrequency;
class DrawArea;
class KRotData;
class KRotScrollBar;
class QPopupMenu;



/**
 * @short Work area canvas.
 *
 * This is a working area canvas, consisting of a horizontal scrollbar and
 * a drawing area.
 *
 * @author Jochen Kpper
 * @version 1999/01/18
 */
class WorkArea : public QWidget
{
    Q_OBJECT

public:

    /**
     * Default Constructor.
     *
     * @param parent Passed to QWidget constructor.
     * @param name Passed to QWidget constructor.
     */
    WorkArea( KRotData *krotdata, QWidget *parent=0, const char *name=0 );

    ~WorkArea();
    

protected slots:

    void newAssignment( const AssignedFrequency& ass );

    /**
     * The global start/stop limits are changed.
     *
     * @param start Lowest frequency of all sets.
     * @param stop  Highest frequency of all sets.
     * @param min   Lowest intensity of all sets.
     * @param max   Highest intensity of all sets.
     */
    void limits( const DataRanges& r );

    /**
     * Show range selection popup
     */ 
    void selectRange();
	
    void setAveraging( const unsigned int avg );

    /**
     * @param freq Actual position of the mouse pointer in frequency units.
     */
    void setFreq( const int64_t freq );

    /**
     * Set display and scrollbar to the given position.
     *
     * @param pos Scrollbar value to go to.
     */
    void setPosition( const int64_t pos );

    /**
     * Show selected range
     */ 
    void showRange( int id );
	

protected:
    
    /**
     * Adjust scrollbars when we are resized.
     */
    void resizeEvent( QResizeEvent * );

    /**
     * Find range containing the given frequency.
     * If it is found, this->current is set to the range containing pos.
     *
     * @param pos Frequency to look for
     * @return true if range was found, false otherwise.
     */
    bool findRange( int64_t pos );

    /**
     * Update all @ref KRotScrollbar and @ref DrawArea to display the current range.
     */
    void showCurrentRange();
    
signals:

    /**
     * Sends every assignment made to the world.
     * These assignment need to be stored and should be editable and saveable.
     */
    void assignment( const AssignedFrequency& ass );

    /**
     * @param freq Actual position of the mouse pointer in frequency units.
     */
    void freq( const int64_t freq );
    
    /**
     * Message to user.
     *
     * @param msg Message to show. This shall go to a statusbar or similar.
     */
    void message( const QString& msg );
    

protected:

    /**
     * Actual averaging value - this is used after the user changed the combobox
     * value to really follow the change (drawarea + scrollbar).
     */
    unsigned int average;

    DrawArea *drawarea;

    KRotScrollBar *hscroll, *vscroll;

    KRotData *krotData;

    DataRange current;
    
    QPopupMenu *popup;

    DataRanges ranges;

    int64_t shifts[ 2 ];
};



#include "workarea_inline.h"


#endif



//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
