// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#include "transformations.h"
#include "lined.h"
#include "krotData.h"
#include "simulation.h"

#include <kapp.h>
#include <kbuttonbox.h>
#include <kspinbox.h>

#include <qbuttongroup.h>
#include <qlistbox.h>
#include <qlayout.h>
#include <qradiobutton.h>



DialogGeometryTransform::DialogGeometryTransform( KRotData *krotdata, QWidget* parent, const char *name )
    : QDialog( parent, name ),
      current( 0 ), krotData( krotdata )
{
    KROT_LAUNCH( "Launching DialogGeometryTransform constructor." );
    setCaption( i18n( "krot [Geometric Tranformation]" ) );
    QGridLayout *layout = new QGridLayout( this, 4, 3, 10, 5, "DialogGeometryTransformLayout" );
    // list
    list = new QListBox( this, "GeometryTransformList" );
    list->setMultiSelection( false );
    layout->addMultiCellWidget( list, 0, 2, 0, 1 );
    connect( list, SIGNAL( highlighted( const char* ) ),
	     SLOT( newSet( const char* ) ) );
    refresh();
    getCurrentSet();
    // scaling
    QButtonGroup *scaleGroup = new QButtonGroup( i18n( "Scaling" ), this, "scaleGroup" );
    QGridLayout *scaleLayout = new QGridLayout( scaleGroup, 2, 1, 15, 5, "scaleLayout" );
    QLabel *yscaleLBL = new QLabel( i18n( "intensity:" ), scaleGroup, "yScaleLabel" );
    scaleLayout->addWidget( yscaleLBL, 0, 0 );
    yscale = new Floated( 6, 0.0, scaleGroup, "yScale" );
    scaleLayout->addWidget( yscale, 1, 0 );
    scaleLayout->activate();
    layout->addWidget( scaleGroup, 0, 2 );
    // shift
    QButtonGroup *shiftGroup = new QButtonGroup( i18n( "Shifting" ), this, "shiftGroup" );
    QGridLayout *shiftLayout = new QGridLayout( shiftGroup, 4, 1, 15, 5, "shiftLayout" );
    QLabel *xshiftLBL = new QLabel( i18n( "frequency [MHz]:" ), shiftGroup, "xShiftLabel" );
    shiftLayout->addWidget( xshiftLBL, 0, 0 );
    xshift = new KNumericSpinBox( shiftGroup, "xShift" );
    xshift->setEditable( true );
    xshift->setFixedHeight( yscale->size().height() );
    xshift->setRange( INT_MIN, INT_MAX );
    xshift->setValue( 0 );
    shiftLayout->addWidget( xshift, 1, 0 );
    QLabel *yshiftLBL = new QLabel( i18n( "intensity:" ), shiftGroup, "yShiftLabel" );
    shiftLayout->addWidget( yshiftLBL, 2, 0 );
    yshift = new KNumericSpinBox( shiftGroup, "yShift" );
    yshift->setEditable( true );
    yshift->setFixedHeight( yscale->size().height() );
    yshift->setRange( INT_MIN, INT_MAX );
    yshift->setValue( 0 );
    shiftLayout->addWidget( yshift, 3, 0 );
    shiftLayout->activate();
    layout->addMultiCellWidget( shiftGroup, 1, 2, 2, 2 );
    // buttons
    KButtonBox *bbox = new KButtonBox( this );
    bbox->addStretch( 5 );
    QPushButton *doIt = bbox->addButton( i18n( "Do It" ) );
    bbox->addStretch( 10 );
    QPushButton *close = bbox->addButton( i18n( "Close" ) );
    bbox->addStretch( 10 );
    QPushButton *help   = bbox->addButton( i18n( "Help" ) );
    bbox->addStretch( 5 );
    bbox->layout();
    bbox->setFixedHeight( bbox->sizeHint().height() + 10 );
    connect( doIt, SIGNAL( clicked() ),
	     this, SLOT( doit() ) );
    connect( close, SIGNAL( clicked() ),
	     this, SLOT( reject() ) );
    connect( help, SIGNAL( clicked() ),
	     this, SLOT( help() ) );
    layout->addMultiCellWidget( bbox, 3, 3, 0, 2 );
    layout->activate();

    KROT_ASSERT( 0 < list->count(), "DialogGeometryTransform called without any set" );
    return;
}



void DialogGeometryTransform::doit()
{
    KROT_LAUNCH( "Launching DialogGeometryTransform::doit" );
    current->setScaling( yscale->value() );
    current->setShifts( make_pair< int64_t, int64_t >( xshift->getValue(), yshift->getValue() ) );
    krotData->checkLimits();
    return;
}



int DialogGeometryTransform::exec()
{
    refresh();
    if( 0 != list->count() )
	list->setCurrentItem( list->topItem() );
    return QDialog::exec();
}



void DialogGeometryTransform::getCurrentSet()
{
    current = krotData->getDataSet( list->text( list->currentItem() ) );
    return;
}



void DialogGeometryTransform::help()
{
    KROT_LAUNCH( "Launching DialogGeometryTransform::help." );
    kapp->invokeHTMLHelp( "krot/geomtransform.html", "" );
    return;
}



void DialogGeometryTransform::newSet( const char *name )
{
    KROT_LAUNCH( "Launching DialogGeometryTransform::newSet." );
    getCurrentSet();
    xshift->setValue( current->shifts().first );
    yshift->setValue( current->shifts().second );
    yscale->setValue( current->scaling() );
}



void DialogGeometryTransform::refresh()
{
    list->clear();
    list->setAutoUpdate( false );
    mapSimulation::const_iterator sims = krotData->simulations().begin();
    while( sims != krotData->simulations().end() ) {
	list->inSort( sims->first.data() );
	sims++;
    }
    mapSpectrum::const_iterator spec = krotData->spectra().begin();
    while( spec != krotData->spectra().end() ) {
	list->inSort( spec->first.data() );
	spec++;
    }
    list->setAutoUpdate( true );
    return;
}



//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
