// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#include "constants.h"
#include "importRawData.h"
#include "krot.h"
#include "rawData.h"

#include <qfile.h>
#include <qtextstream.h>

#include <kapp.h>

#include <algo.h>
#include <climits>
#include <values.h>
#include <set>
#include <vector>



RawData::RawData( const QString& name )
    : QObject( 0, "RawData" ),
      avg( 1 ), cutoff( 2048 ), dataZero( 0 ), filename( name ), numCoeff( 3 )
{
    KROT_LAUNCH( "Launching RawData constructor" );
    return;
}



void RawData::determineMarkerPositions()
{
    KROT_LAUNCH( "Launching RawData::determineMarkerPosition" );
    cutoff = static_cast< int >( mn.etalon + ( mx.etalon - mn.etalon ) / 10 );
    mks.erase( mks.begin(), mks.end() );
    if( data.size() > 3 ) {
	// count all cuttoff line crossings in upward direction
	unsigned int crossings = 0;
	for( unsigned int i=0; i < data.size() - 1; i++ )
	    if( ( data[i].etalon < cutoff ) && ( data[i+1].etalon > cutoff ) )
		crossings++;
	// calculate average distance between crossings. Distance between two markers
	// needs to be at least one third of that.
	if( ( crossings > 0 ) && ( crossings < data.size() / 4 ) ) {
	    int dist = data.size() / crossings / 3;
	    // determine positions of markers
	    for( int i=0; i < static_cast< int >( data.size() - 1 ); i++ ) {
		if( ( data[i].etalon > cutoff ) && ( data[i+1].etalon > cutoff ) ) {
		    int start = i;
		    while( ( data[i].etalon > cutoff ) && ( data[i+1].etalon > cutoff ) )
			i++;
		    int stop = i;
		    int pos= stop;
		    for( int x=start; x<stop; x++ )
			if( data[ x ].etalon > data[ pos ].etalon )
			    pos = x;
		    mks.insert( pos );
		    i += dist;
		}
	    }
	}
    }
    return;
}



double *RawData::linearize( const unsigned int delta, int64_t& start, u_int64_t& size ) const
{
    KROT_LAUNCH( "Launching RawData::linearize" );
    KConfig *config = kapp->getConfig();
    config->setGroup( KROT_CONFIG_IMPORT );
    int absIndex = config->readNumEntry( DialogIodineFrequency::config_AbsIndex, 0 );
    double *inten;
    double absFreq = config->readDoubleNumEntry( DialogIodineFrequency::config_AbsFreq, 0 ),
	fsr =  config->readDoubleNumEntry( DialogFSR::config_FSR, 150 );
    if( UNIT_CM == config->readNumEntry( DialogIodineFrequency::config_AbsFreqUnit, UNIT_CM ) ) {
	KROT_DEBUG( "Converting AbsFreq from cm-1 to MHz" );
	absFreq *= Constants::cm2MHz();
    }
    if( config->readBoolEntry( DialogIodineFrequency::config_Double, false ) ) {
	KROT_DEBUG( "Doubling AbsFreq" );
	absFreq *= 2;
    }
    switch( config->readNumEntry( DialogLinModel::config_LinModel, LINEAR ) ) {
    case LINEAR:
	{ // size
	    size = static_cast<unsigned int>( fsr * ( numMarker() - 1 ) / delta );
	    KROT_DEBUG1( "size %d", size );
	} { // start
	    set< int >::iterator iter = mks.begin();
	    int i = -1;
	    while( ( *iter < absIndex ) && ( mks.end() != iter ) ) {
		iter++;
		i++;
	    }
	    set< int >::iterator iter2 = iter--;
	    start = static_cast< int64_t >( absFreq - fsr * i
					    - fsr * ( absIndex - *iter ) / ( *iter2 - *iter ) );
	    KROT_DEBUG3( "start: %lld\tabsIndex: %d\tabsFreq: %f", start, absIndex, absFreq );
	} { // intensity array
	    inten = new double[ size ];
	    int64_t freq = start;
	    int i = 0, m = 0;
	    set< int >::iterator marker1 = mks.begin(),
		marker2 = mks.begin();
	    marker2++;
	    inten[ 0 ] = data[ *marker1 ].signal;
	    while( mks.end() != marker2 ) {
		int j2 = *marker1 + 1;
		double step = ( *marker2 - *marker1 ) / fsr,
		    p2 = m * fsr + start + step;
		m++; marker1++; marker2++;
		while( freq < m * fsr + start ) {
		    while ( p2 <= freq ) {
			p2 += step;
			j2++;
		    }
		    double p1 = p2 - step;
		    int j1 = j2 - 1;
		    inten[ ++i ] = ( ( p2 - freq ) * data[ j1 ].signal +  ( freq - p1 ) * data[ j2 ].signal ) / step;
// 		    KROT_DEBUG7("freq = %d,\t j1 = %d,\t p1 = %f,\t p2 = %f,\t"
// 				"data[j1] = %d,\t data[j2] = %d,\tinten = %d",
// 				freq, j1, p1, p2, data[ j1 ].signal, data[ j2 ].signal, inten[ i-1 ] );
		    freq += delta;
		}
	    }
	}
	break;
    case QUADRATIC:
	KROT_ERROR( KDEBUG_KROT_GENERAL, "RawData::spectrum - QUADRATIC not implemented yet" );
	break;
    case CUBICSPLINE:
	KROT_ERROR( KDEBUG_KROT_GENERAL, "RawData::spectrum - CUBICSPLINE not implemented yet" );
	break;
    }
    return inten;
}



KHiDaqData::KHiDaqData( const QString& filename )
    : RawData( filename )
{
    KROT_LAUNCH( "Launching KHiDaqData constructor" );
    read();
    determineMarkerPositions();
    return;
}



void KHiDaqData::read()
{
    KROT_LAUNCH( "Launching KHiDaqData::read" );
    QFile file( filename );
    if(! file.open( IO_ReadOnly ) ) {
	KROT_ERROR( KDEBUG_KROT_FILE, "KHiDaqData::read, file does not exist\n"
		    "We need to handle that situation correctly !" );
    } else {
	QTextStream stream( &file );
	mx.signal = mx.etalon = mx.iodine = mx.power = INT_MIN;
	mn.signal = mn.etalon = mn.iodine = mn.power = INT_MAX;
	QString line;
	do {
	    line = stream.readLine();
	    // parsing needs to be done
	} while( ! ( line.contains( "[Data]" ) || stream.eof() ) );
	while( ! stream.eof() ) {
	    RawDataPoint point;
	    stream >> point.etalon >> point.iodine >> point.power >> point.signal;
	    data.push_back( point );
	    if( point.etalon < mn.etalon )
		mn.etalon = point.etalon;
	    else if( point.etalon > mx.etalon )
		mx.etalon = point.etalon;
	    if( point.iodine < mn.iodine )
		mn.iodine = point.iodine;
	    else if( point.iodine > mx.iodine )
		mx.iodine = point.iodine;
	    if( point.power < mn.power )
		mn.power = point.power;
	    else if( point.power > mx.power )
		mx.power = point.power;
	    if( point.signal < mn.signal )
		mn.signal = point.signal;
	    else if( point.signal > mx.signal )
		mx.signal = point.signal;
	}
    }
    return;
}



JbaData::JbaData( const QString& file )
    : RawData( file )
{
#ifdef LAUNCH
    cerr << "Launching JbaData constructor" << endl;
#endif
    read();
    determineMarkerPositions();
    return;
}



void JbaData::read()
{
#ifdef LAUNCH
    cerr << "Launching KHiDaqData::read" << endl;
#endif
    KDEBUG( KDEBUG_WARN, KDEBUG_KROT_GENERAL, "Not implemented yet" );
    return;
}

