// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#include "drawarea.h"
#include "krot.h"
#include "krotData.h"
#include "lineSelection.h"

#include <qcursor.h>
#include <qevent.h>
#include <qpainter.h>
#include <qwidget.h>

#include <sys/types.h>



DrawArea::DrawArea( KRotData *krotdata, QWidget *parent, const char *name )
    : QWidget( parent, name ),
      krotData( krotdata ), min( 0 ), start( 0 )
{
    KROT_LAUNCH( "Launching DrawArea constructor" );
    setBackgroundMode ( NoBackground );
    setCursor( crossCursor );
    setMouseTracking( true );
    lineSelection = new DialogLineSelection( krotData );
    connect( this, SIGNAL( lineSelectionRange( const int64_t, const int64_t ) ),
	     lineSelection, SLOT( setRange( const int64_t, const int64_t ) ) );
    return;
}



void DrawArea::mouseMoveEvent( QMouseEvent *event )
{
    KROT_LAUNCH_VERBOSE( "Launching DrawArea::mouseMoveEvent" );
    switch( event->state() ) {
    case 0:
	emit freq( event->x() * krotData->averaging() + start );
	break;
    case ( LeftButton | ShiftButton ):
	// shift dataset
	break;
    }
    return;
}



void DrawArea::mousePressEvent( QMouseEvent *event )
{
     KROT_LAUNCH_VERBOSE( "Launching DrawArea::mousePressEvent" );
     switch( event->state() ) {
     case 0:
	 // no modifier key pressed
	 // -> left button starts selection of sim-lines
	 // -> middle button makes assignment
	 if( event->button() & LeftButton )
	     posPressed[0] = event->pos();
	 if( event->button() & MidButton ) {
	     AssignedFrequency ass( lineSelection->selection() );
	     ass.frequency() = start + event->x() * krotData->averaging();
	     emit assignment( ass );
	 }
	 break;
     case AltButton:
	 // -> left button scales nearest dataset
	 break;
     case ControlButton:
	 break;
     case ShiftButton:
	 // -> left button shifts nearest dataset
	 // find Nearest();
	 // save Position();
	 break;
     }
     return;
}



void DrawArea::mouseReleaseEvent( QMouseEvent *event )
{
    KROT_LAUNCH_VERBOSE( "Launching DrawArea::mouseReleaseEvent" );
    if( event->button() & LeftButton ) {
        // left button stops selection of sim-lines
	int64_t one = start + posPressed[0].x() * krotData->averaging();
	int64_t two = start + event->x() * krotData->averaging();
	emit lineSelectionRange( std::min( one, two ), std::max( one, two ) );
    }
    // other buttons don't have a meaning here
    return;
}



void DrawArea::paintEvent( QPaintEvent *event )
{
    KROT_LAUNCH_VERBOSE( "Launching DrawArea::paintEvent" );
    KROT_DEBUG2_VERBOSE( "window height: %d, width: %d", height(), width() );
    QPainter paint;
    paint.begin( this );
    if( krotData->empty() ) {
	paint.fillRect( event->rect(), paint.backgroundColor() );
    } else {
	paint.fillRect( event->rect(), kapp->windowColor );
	mapSpectrum::const_iterator spec = krotData->spectra().begin();
	while( krotData->spectra().end() != spec ) {
	    if( spec->second.start() < start + width() * krotData->averaging() && spec->second.stop() > start ) {
		paint.setPen( spec->second.color() );
		paint.moveTo( event->rect().left(),
			      computeIntensity( spec->second,
						start + static_cast< int64_t >( event->rect().left() * krotData->averaging() ) ) );
		for( int x = event->rect().left() + 1; x <= event->rect().right(); x++ ) {
		    paint.lineTo( x, computeIntensity( spec->second,
						       start + static_cast< int64_t >( x * krotData->averaging() ) ) );
		}
	    }
	    spec++;
	}
	mapSimulation::const_iterator sims = krotData->simulations().begin();
	while( krotData->simulations().end() != sims ) {
	    if( sims->second.start() < start + width() * krotData->averaging() && sims->second.stop() > start ) {
		paint.setPen( sims->second.color() );
		paint.moveTo( event->rect().left(),
			      computeIntensity( sims->second,
						start + event->rect().left() * krotData->averaging() ) );
		for( int x = event->rect().left() + 1; x <=  event->rect().right(); x++ )
		    paint.lineTo( x, computeIntensity( sims->second, start + x * krotData->averaging() ) );
	    }
	    sims++;
	}
    }
    paint.end();
    return;
}



void DrawArea::resizeEvent( QResizeEvent * )
{
    KROT_LAUNCH_VERBOSE( "Launching DrawArea::resizeEvent." );
    checkStartFreq();
    return;
}



void DrawArea::newPosition( int freq )
{
    KROT_LAUNCH_VERBOSE( "Launching DrawArea::setStart" );
#if SIZEOF_LONG >= 8
    KROT_DEBUG2_VERBOSE( "Old DrawArea start frequency = %ld, new pos = %ld", start, freq );
#else
    KROT_DEBUG2_VERBOSE( "Old DrawArea start frequency = %lld, new pos = %ld", start, freq );
#endif
    // set start
    start = range.start + ( freq  - width() / 2 ) * krotData->averaging();
#if SIZEOF_LONG >= 8
    KROT_DEBUG1_VERBOSE( "New DrawArea start frequency = %ld", start );
#else
    KROT_DEBUG1_VERBOSE( "New DrawArea start frequency = %lld", start );
#endif
    // a plain repaint is faster than scrolling the widget. ( Sep. 98, Qt-1.40, Jochen )
    repaint( true );
}



void DrawArea::newIntensity( int offset )
{
    min = range.min + offset - height() / 2;
    repaint( true );
}




//* Local Variabes:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
