// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with this package.



#ifndef KROT_DATASET_H
#define KROT_DATASET_H



#include "exception.h"
#include "krot.h"
#include "transition.h"

#include <kapp.h>

#include <qcolor.h>
#include <qstring.h>

#include <sys/types.h>
#include <algorithm>
#include <utility>



class KRotData;
class Spectrum;



/**
 * @short Storage object for data sets.
 *
 * This class provides the basic functionality common to all kinds of data sets
 * (like spectra / simulations).
 *
 * Possible exceptions: @ref NoDataSetOverlap, @ref ShortDataSet, @ref UnknownFileType
 *
 * @author Jochen Kpper
 * @version 1999/02/07
 */
class DataSet
{
public:

    enum { LINES_DAT_DATA, UNKNOWN_DATA, XY_DATA };

    /**
     * Constructor.
     */
    // DataSet();

    /**
     * Constructor.
     */
    DataSet( KRotData *krotdata );

    /**
     * Copy constructor.
     */
    DataSet( const DataSet& set );

    /**
     * Destructor; free all dynamically allocated memory.
     */
    virtual ~DataSet();

    /**
     * Compares the names of the two DataSets.
     *
     * @param set DataSet to compare to this.
     * @return True if this->name < set.name, false otherwise.
     */
    virtual bool operator<( const DataSet& set ) const;

    /**
     * Equality is defined as address identity (objetc identity) yet,
     * that might be changed when needed in some other context.
     *
     * @param set DataSet to compare to this.
     * @return True if the addresses of the two sets are equal, false otherwise.
     */
    virtual bool operator==( const DataSet& set ) const;

    /**
     * Read-only access to data points. This functions returns the 
     * intensity of this set at the frequency freq.
     *
     * @param freq Frequency to get the intensity at.
     * @return Averaged intensity.
     */
    double operator[]( int64_t freq ) const;

    /**
     * On object identity nothing is done, otherwise all data from the given set
     * is copied to this.
     *
     * @param set DataSet to clone.
     * @return this.
     */
    virtual DataSet& operator=( const DataSet& set );

    /**
     * Autocorrelate this @ref Spectrum.
     *
     * @return Autocorrelation
     */
    Spectrum& autocorrelate() const;
    
    /**
     * @return Color this dataset shall be painted in.
     */
    const QColor& color() const;

    /**
     * Calculate chi^2 deviation of this and the other DataSet.
     * This value will be normalized by the available data points.
     *
     * @param other Spectrum to calculate deviation to this for.
     * @return      chi^2 value
     */
    double chi2( const DataSet& other ) const;
    
    /**
     * Crosscorrelate this and the other DataSet.
     *
     * @param other Spectrum to correlate ourself with.
     * @return      Crosscorrelation
     */
    Spectrum& crosscorrelate( const DataSet& other ) const;
    
    /**
     * The distance of two data points is fixed at 1 MHz at this time.
     *
     * @return Frequency distance of any two neighbouring points in this set.
     */
    unsigned int delta() const;

    /**
     * Calculate the difference spectrum of two DataSet in the overlap-area.
     *
     * @param other DataSet to subtract from this.
     * @return Difference \ref Spectrum.
     */
    virtual Spectrum& difference( const DataSet& other ) const;

    /**
     * Calculate the integral of the DataSet.
     * This does calculate over the current set (scaled, shifted), not the
     * original one.
     *
     * Integration is done by trapezoidal rule.
     */
    double integral() const;
    
    /**
     * Load dataset from file
     */
    void load( const QString& filename, int type );

    /**
     * @return Maximum intensity within this set.
     */
    double max() const;

    /**
     * @return Minimum intensity within this set.
     */
    double min() const;

    /**
     * @return Name of this dataset.
     *
     * @see setName
     * @see shortName
     */
    const QString& name() const;

    virtual void save( const QString&, int ) const;

    /**
     * @return Intensity scaling factor.
     */
    virtual const double scaling() const;

    /**
     * @param color Set the color this set should be drawn in.
     */
    void setColor( const QColor& color );

    /**
     * The name of any dataset should be the fully qualified filename, if this
     * set was imported from a file; If this set was generated by krot, this
     * is the name given to it by the user within krot.
     * 
     * @param name Name of this dataset.
     */
    void setName( const QString& name );

    /**
     * Change the intensity scaling of this set.
     * 
     * @param scaling New intensity scaling factor.
     */
    void setScaling( const double scaling );

    /**
     * Geomtrically shift this set.
     *
     * @param pair<>( xshift, yshift ) New frequency and intensity offsets.
     *        These values are in pixels !
     */
    void setShifts( pair< int64_t, int64_t > val = make_pair( 0, 0 ) );
    
    /**
     * @return Pair of frequency and intensity offsets.
     */
    pair< int64_t, int64_t > shifts() const;
    
    /**
     * @return Short name of this dataset. This is the pure filename
     * without any path.
     *
     * @see setName
     * @see name
     */
    const QString shortName() const;

    /**
     * @return Number of data points in this set.
     */
    unsigned int size() const;

    /**
     * @return Start frequency of this set.
     */
    int64_t start() const;

    /**
     * @return Stop frequency of this set.
     */
    int64_t stop() const;

    
protected:

    void saveXY( const QString& ) const;

    void recalculateData();

    /**
     * Resize and set the data.
     *
     * Give the memory to the DataSet, you need not to use any pointer to that memory
     * outside this class any more !
     *
     * @param start New start frequency
     * @param stop  New stop frequency
     * @param data  New data, must have the correct size as defined by start/stop !
     */    
    void setData( const int64_t start,  const int64_t stop, double *data );
    
private:

    /**
     *
     */
    virtual bool loadLinesDat( const QString& );

    /**
     *
     */
    virtual void loadXY( const QString& filename );


protected:
    
    /**
     * Color we shall be displayed in.
     */
    QColor clr;

    /**
     *
     */
    KRotData *krotData;

    /**
     * name
     */
    QString nam;

    
private:

    /**
     * intensities
     */
    double *data, *realdata;

    /**
     * Is data valid
     */
    bool data_valid;
    
    /**
     * frequency step width [MHz]
     */
    const u_int64_t dlta;

    /**
     * Are mx / mn still valid
     */
    bool maxmin_valid;
    
    
    /**
     * maximum / minimum intensity
     */
    double *mx, *mn;

    /**
     * User-adjustable x- and y-shift of this dataset.
     *
     * x <-> first
     * y <-> second
     */
    pair< int64_t, int64_t > xy_shifts;

    /**
     * user-adjustable intensity-scaling
     */
    double scalg;

    /**
     * number of data points
     */
    u_int64_t siz;

    /**
     * first/last frequency
     */
    int64_t strt, stp, realstrt, realstp;
};



#include "dataset_inline.h"


/**
 * @short Exception ShortDataSet
 *
 * This exception is thrown whenever a operation on a single @ref DataSet needs
 * a larger @ref DataSet than the actual one to use.
 *
 * @author Jochen Kpper
 * @version 1999/02/07
 */
struct ShortDataSet : KRotException
{
    ShortDataSet( const DataSet& a )
	: KRotException(), set( a ) {};
    
    const DataSet& set;
};
    


/**
 * @short Exception NoDataSetOverlap
 *
 * This exception is thrown whenever a operation on two overlaping @ref DataSet
 * shall be performed but the given @ref DataSet do not overlap.
 *
 * @author Jochen Kpper
 * @version 1999/02/07
 */
struct NoDataSetOverlap : KRotException
{
    NoDataSetOverlap( const DataSet& one, const DataSet& two )
	: KRotException(), set1( one ), set2( two ) {};
    
    const DataSet& set1, set2;
};
    


#endif



//* Local Variables:
//* mode: C++
//* c-file-style: "Stroustrup"
//* End:
