c     This file is part of krot,
c     a program for the simulation, assignment and fit of HRLIF spectra.
c
c     Copyright (C) 1994-1998 Arnim Westphal
c     Copyright (C) 1997-1999 Jochen Kpper
c
c     If you use this program for your scientific work, please cite it according to
c     the file CITATION included with this package.
c
c     krot-arnirot
c     a program to calculate rotational resolved vibrational/vibronic bands


#include "arni.h"


c     utility routines and functions used by arnirot


c------------------------------------------------------------------------------
      subroutine chevec( evec, eval, dmham, iact )
c     check for incorrectly sorted eigenVECtors of numerically degenerate eigenstates

      implicit none

      integer        dmham
      integer        iact, ir, ir2, ic, J
      real*8         evec(dmham,dmham), eval(dmham)
      real*8         degdif
      parameter      (degdif = 1.d-5)

      ARNIROT_LAUNCH ( "Launching chevec." )

c     calculate J quantum number
      J = (iact - 1)/2

c     loop through the eigenvector matrix columns
c     only pairs of eigenvectors corresponding to the same parity of submatrices (E or O) may be numerically degenerate
c     i.e. in the (near) prolate case only the 2nd/3rd, 4th/5th, ... levels from the bottom
      do ic = 2, iact-1, 2
c        check the (ic) and (ic+1) energy levels for degeneracy; go to next column if they are not numerically degenerate
         if (dabs(eval(ic+1) - eval(ic)).gt.degdif) goto 1
c        loop through rows to find first element of significant size
         do ir = 1, iact
c           test this neg. K element if it is large enough
            if (dabs(evec(ir,ic)).gt.1.d-3) then
c              neg. K quantum number of the element that has passed the test    : k   = ir-(J+1)
c              row index of the corresponding element with pos. K quantum number: ir2 = (-k)+(J+1) = (-ir)+2(J+1)
               ir2 = iact + 1 - ir
c              examine whether this pos. K element changes sign or not;
c              exchange column if symmetric, it should be ANTISYMMETRIC
#warning Does this produces the same results ?
               if( evec( ir, ic ) * evec( ir2, ic ) .gt. 0.d0 ) then
                  call dswap(iact, evec( 1, ic ), 1, evec( 1, ic + 1 ), 1 )
c                 exit to next set if exchange has been made
                  goto 1
               end if
            end if
         end do
    1 end do

      return
      end


c------------------------------------------------------------------------------
      subroutine delabs( rotcog, rotcoe, npar, flag )
c     convert ABSolute constants to DELta or vice versa
c     - flag == 1: absolute to delta
c     - flag != 1: delta to absolute

      implicit none

      integer          flag, npar
      double precision rotcog( npar ), rotcoe( npar )

      ARNIROT_LAUNCH ( "Launching delabs." )

      if( flag .eq. 1 ) then
c        absolute values to deltas: rotcoe = -1 * rotcog + rotcoe
         call daxpy( npar, -1.d0, rotcog, 1, rotcoe, 1 )
      else
c        deltas to absolute values: rotcoe = +1 * rotcog + rotcoe
         call daxpy( npar, 1.d0, rotcog, 1, rotcoe, 1 )
      end if

      return
      end


c------------------------------------------------------------------------------
      subroutine delder( funcs, npar, maxnli, ndata )
c     convert array funcs to DELta DERivatives if fitting delta constants

      implicit none

      integer        maxnli, npar
      integer        i, l, ndata
      real*8         funcs(maxnli,2*npar+2)

      ARNIROT_LAUNCH ( "Launching delder." )

c     (A''+dA)De - A''Dg = A'De - A''Dg
      do l = 1, npar
         do i = 1, ndata
            funcs( i, l ) = funcs( i, l + npar ) + funcs( i, l )
         end do
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine derout( J, deriv, dmeval, npar, icqn, ldash )
c     OUTput the DERivatives for each state

      implicit none

      integer        dmeval, npar
      integer        i, iept, J, k, l
      integer        icqn(dmeval,3)
      real*8         deriv(dmeval,npar)
      character*81   ldash

      ARNIROT_LAUNCH ( "Launching derout." )

      write(*,'(a)') ldash
      write(*,'(a,/)') 'energy derivatives:'
c     calculate correct offset
      iept = J*J

c     A, B and C derivatives
      do i = 1, 2*J+1, 1
         write(*,100) (icqn(i+iept,k), k = 1,3,1), (deriv(i+iept,l), l = 1,3,1)
  100    format('|',3(i2),'>    A: ',d15.7,2x,'  B: ',d15.7,2x,'  C: ',d15.7)
      end do
      write(*,*) ' '

c     Dx, Dy, Dz derivatives
      do i = 1, 2*J+1, 1
         write(*,110) (icqn(i+iept,k), k = 1,3,1), (deriv(i+iept,l), l = 4,6,1)
  110    format('|',3(i2),'>   Dx: ',d15.7,2x,' Dy: ',d15.7,2x,' Dz: ',d15.7)
      end do
      write(*,*) ' '

c     Watson hamiltonian constants derivatives
      do i = 1, 2*J+1, 1
         write(*,120) (icqn(i+iept,k), k = 1,3,1), (deriv(i+iept,l), l = 7,9,1)
  120    format('|',3(i2),'>   DJ: ',d15.7,2x,'DJK: ',d15.7,2x,' DK: ',d15.7)
      end do
      write(*,*) ' '
      do i = 1, 2*J+1, 1
         write(*,130) (icqn(i+iept,k), k = 1,3,1), (deriv(i+iept,l), l = 10,11,1)
  130    format('|',3(i2),'>   dJ: ',d15.7,2x,' dK: ',d15.7)
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine eigval( eval, dmham, dmeval, J, icqn, ldash )
c     output EIGenVALues

      implicit none

      integer        dmham, dmeval
      integer        i, iept, J, l
      integer        icqn(dmeval,3)
      real*8         eval(dmham)
      character*81   ldash

      ARNIROT_LAUNCH ( "Launching eigval." )

      iept = J*J
      write(*,'(a)') ldash
      write(*,'(a,/)') 'eigenvalues:'

      do i = 1, 2*J+1, 1
         write(*,200) (icqn(i+iept,l), l = 1,3,1), eval(i)
  200    format('|',3(i2),'>',f21.6,' MHz')
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine funcsl( funcs, maxnli, npar, ndata, ipttem, derlg, derle, fsrdiv )
c     Load the ndata*2*npar derivatives into array FUNCS for nonzero assigned
c     lines as referenced by index ipttem; npar: number of parameters per state

      implicit none

      integer        maxnli, npar
      integer        i, l, ndata
      integer        ipttem(maxnli)

      real*8         derlg(maxnli,npar), derle(maxnli,npar)
      real*8         fsrdiv
      real*8         funcs(maxnli,2*npar+2)

      ARNIROT_LAUNCH ( "Launching funcsl." )

      do l = 1, npar
         do i = 1, ndata
c           excited state data loaded high in array funcs, ground state data low
            funcs(i,l+npar) =  fsrdiv * derle( ipttem(i), l )
            funcs(i,l)      = -fsrdiv * derlg( ipttem(i), l )
         end do
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine idef( MHz2uA, rotco, npar, defect )
c     calculate the Inertial DEFect

      implicit none

      integer        npar
      real*8         MHz2uA, rotco(npar), defect

      ARNIROT_LAUNCH ( "Launching idef." )

      defect = MHz2uA * ( 1.d0/rotco(3) - 1.d0/rotco(2) - 1.d0/rotco(1) )

      return
      end


c------------------------------------------------------------------------------
      subroutine ideferr( rotcog, rotcoe, npar, err, idelta )
c     calculate Inertial DEFects and their respective ERRors
c     - idelta = 0: absolute constants,
c     - idelta = 1: delta constants

      implicit none

      integer        npar
      integer        idelta

c     MHz to amu*Angstrom**2 conversion factor; dx assumed to be 0.0001 MHz
      real*8         MHz2uA, dx
      parameter      ( MHz2uA = 505379.07d0, dx = 1.d-4 )
      real*8         kappag, kappae, idefg, idefe
      real*8         sigidg, sigide, sigkpg, sigkpe
      real*8         rotcog(npar), rotcoe(npar), err(2*npar+2)
      real*8         errmat(3,3), sigvecl(3), sigvecr(3)

      ARNIROT_LAUNCH ( "Launching ideferr." )

      write(*,'(a)') 'inertial defect calculations'

c     convert excited state delta constants into absolute values if idelta==1
      if ( idelta .eq. 1 ) call delabs( rotcog, rotcoe, npar, 2 )

c     calculate inertial defects for both states with absolute constants
      call idef( MHz2uA, rotcog, npar, idefg )
      call idef( MHz2uA, rotcoe, npar, idefe )
c     calculate kappa values for both states with absolute constants
      call kap( rotcog, npar, kappag )
      call kap( rotcoe, npar, kappae )

c     propagate errors for ground state inertial defect
      call properr( rotcog, err, errmat, sigvecl, sigvecr, npar, 1, 1, MHz2uA, sigidg, dx )
c     propagate errors for excited state inertial defect
      call properr( rotcoe, err, errmat, sigvecl, sigvecr, npar, 1, 2, MHz2uA, sigide, dx )
c     propagate errors for ground state kappa
      call properr( rotcog, err, errmat, sigvecl, sigvecr, npar, 2, 1, MHz2uA, sigkpg, dx )
c     propagate errors for excited state kappa
      call properr( rotcoe, err, errmat, sigvecl, sigvecr, npar, 2, 2, MHz2uA, sigkpe, dx )

c     output kappa and inertial defects
c     ground state
      write(*,'(/,'' ground state kappa'',11x,'':'',f14.8,'' +- '',f12.8)') kappag, sigkpg
      write(*,'('' ground state inertial defect :'',f14.8,'' +- '',f12.8,'' amu angstrom**2'')') idefg, sigidg
c     excited state
      if ( idelta .eq. 0 ) then
         write(*,'(/,''excited state kappa'',11x,'':'',f14.8,'' +- '',f12.8)') kappae, sigkpe
         write(*,'(''excited state inertial defect :'',f14.8,'' +- '',f12.8,'' amu angstrom**2'')') idefe, sigide
      else
         write(*,'(/,8x,''delta kappa'',11x,'':'',f14.8,'' +- '',f12.8)') kappae - kappag, sigkpe
         write(*,'(8x,''delta inertial defect :'',f14.8,'' +- '',f12.8,'' amu angstrom**2'')') idefe - idefg, sigide
      end if

c     convert excited state constants back to deltas if idelta==1
      if ( idelta .eq. 1 ) call delabs( rotcog, rotcoe, npar, 1 )

      return
      end


c------------------------------------------------------------------------------
      subroutine ivecpt( ivpt, Jmax, Jmxcal )
c     setup a vector of 'pointers' (indices) to the eigenvectors of a given J block

      implicit none

      integer        i
      integer        J, Jmax, Jmxcal
      integer        ivpt(0:Jmax)

      ARNIROT_LAUNCH ( "Launching ivecpt." )

      ivpt(0) = 0
      do J = 1, Jmxcal, 1
         i = J - 1
         ivpt(J) = ivpt(i) + (2*i + 1)**2
      end do

      return
      end


c------------------------------------------------------------------------------
c     calculate KAPpa
      subroutine kap( rotco, npar, kappa )

      implicit none

      integer        npar
      real*8         kappa, rotco(npar)

      ARNIROT_LAUNCH ( "Launching kap." )

      kappa = ( 2.d0 * rotco(2) - rotco(1) - rotco(3) ) / ( rotco(1) - rotco(3) )

      return
      end


c------------------------------------------------------------------------------
      subroutine ket( icqn, dmeval, J )
c     determine the sets of quantum numbers for all rotational eigenstates
c     only run once since labels are the same for both ground and excited state rotational levels
c     icqn: matrix to store the numbers J Ka Kc for every eigenstate

      implicit none

      integer        dmeval
      integer        i, ii, iept, itaupt
      integer        J, Ka
      integer        icqn(dmeval,3)

      ARNIROT_LAUNCH ( "Launching ket." )

c     pointer on first element of J eigenvalue block
      iept = J*J

c     first Ka quantum number is zero
      itaupt = iept + 1
      icqn(itaupt,2) = 0
c     count through the rest of Ka numbers
      do Ka = 1, J, 1
         do ii = 1, 2, 1
            itaupt = itaupt + 1
            icqn(itaupt,2) = Ka
         end do
      end do

c     fill in Kc quantum numbers by counting down; fill in J
      itaupt = iept + 2*J+1
      do i = 1, 2*J+1, 1
         icqn(itaupt,3) = icqn(iept+i,2)
         itaupt = itaupt - 1
c        first column of icqn contains quantum number J
         icqn(i+iept,1) = J
      end do

      return
      end


c------------------------------------------------------------------------------
      function matsum( mat, dim, act )
c     calculate the sum of all matrix elements

      implicit none

      integer          dim, act
      double precision matsum, mat( dim, dim )

      integer          row, col
      double precision err, sum, tmp

      ARNIROT_LAUNCH ( "Launching matsum." )
c     use Kahan's trick to get some accuracy into the sum
      err = 0.d0
      sum = 0.d0
      do col = 1, act
         do row = 1, act
            err = err + mat( row, col )
            tmp = sum + err
            err = ( sum - tmp ) + err
            sum = tmp
         end do
      end do

      matsum = sum
      return
      end


c------------------------------------------------------------------------------
      subroutine smat0( mat, dim, n )
c     fill the n x n workspace of a square matrix with zeroes

      implicit none

      integer           n, dim
      double precision  mat( dim, dim )

      integer           col, row

      ARNIROT_LAUNCH( "Launching smat0." )

      do col = 1, n
         do row = 1, n
            mat( row, col ) = 0.d0
         end do
      end do
      
      return
      end


c------------------------------------------------------------------------------
      subroutine murot( tran, pol, n, shorti )
c     ROTate transition dipole vector MU

      implicit none

      integer        row, col, n
      integer        shorti
      real*8         tran(n,n), pol(n), prod(3)

      ARNIROT_LAUNCH ( "Launching murot." )

      if( shorti .eq. 0 ) then
         write(*,'(/,a)') 'original mu components:'
         write(*,300) pol(1), pol(2), pol(3)
  300    format('mu x:',f8.3,5x,'mu y:',f8.3,5x,'mu z:',f8.3)
      end if
      do row = 1, 3
         prod( row ) = 0.d0
         do col = 1, 3
            prod( row ) = prod( row ) + tran( row, col ) * pol( col )
         end do
      end do
      do row = 1, 3
         pol( row ) = prod( row )
      end do
      if( shorti .eq. 0 ) then
         write(*,'(/,a)') 'final mu components:'
         write(*,300) pol(1), pol(2), pol(3)
      end if

      return
      end


c------------------------------------------------------------------------------
      subroutine properr( rotco, err, errmat, sigvecl, sigvecr, npar, routine, state, MHz2uA, xerr, dx )
c     PROPagate ERRors
c     - routine = 1: inertial defect
c               = 2: kappa
c     - state   = 1: ground state
c               = 2: excited state

      implicit none

      double precision matsum

      integer          npar, routine, state
      double precision rotco(npar), err(2*npar+2), errmat(3,3)
      double precision sigvecl(3), sigvecr(3)
      double precision MHz2uA, xerr, dx

      integer          i, offset
      double precision y1, y2

      ARNIROT_LAUNCH ( "Launching properr." )

c     calculate the initial value of the unmodified quantity
      if ( routine .eq. 1 ) then
         call idef( MHz2uA, rotco, npar, y1 )
      else
         call kap( rotco, npar, y1 )
      end if

c     if excited state offset for error vector
      if ( state .eq. 2 ) then
         offset = npar
      else
         offset = 0
      end if

c     calculate the error vectors (df/dx1)*err x1
      do i = 1, 3, 1
c        increment constant by dx
         rotco(i) = rotco(i) + dx
         if ( routine .eq. 1 ) call idef( MHz2uA, rotco, npar, y2 )
         if ( routine .eq. 2 ) call kap( rotco, npar, y2 )
c        decrement constant by dx
         rotco(i) = rotco(i) - dx
         sigvecr(i) = ( y2 - y1 )/dx * err( i+offset )
      end do

c     equate error vectors
      call dcopy( 3, sigvecr, 1, sigvecl, 1 )
c     define error matrix by vector multiplication
      call vecvec( sigvecl, sigvecr, errmat, 3, 3 )
c     sum over matrix for total uncertainty squared
      xerr = matsum( errmat, 3, 3 )

      if ( xerr .ge. 0. ) then
         xerr = dsqrt( xerr )
      else
         write(*,*) 'negative value for xerr: ', xerr
         xerr = dsqrt( dabs( xerr ) )
      end if

      return
      end


c------------------------------------------------------------------------------
      subroutine rotder( deriv, dmeval, npar, tran, tranin, iact, ioff )
c     rotate axis switched rr derivatives back to unrotated frame

      implicit none

      integer        dmeval, npar
      integer        i, iact, ioff
      real*8         deriv(dmeval,npar), tran(3,3), tranin(3,3)
      real*8         prod(3,3), deraxs(3,3)

      ARNIROT_LAUNCH ( "Launching rotder." )

      do i = 1, iact, 1
c        load axis switched derivatives into a matrix to be rotated back to unrotated x,y,z coordinate system
         deraxs(1,1) = deriv(i+ioff,2)
         deraxs(2,2) = deriv(i+ioff,3)
         deraxs(3,3) = deriv(i+ioff,1)
         deraxs(1,2) = deriv(i+ioff,4)/2.d0
         deraxs(1,3) = deriv(i+ioff,5)/2.d0
         deraxs(2,3) = deriv(i+ioff,6)/2.d0
         deraxs(3,1) = deraxs(1,3)
         deraxs(3,2) = deraxs(2,3)
         deraxs(2,1) = deraxs(1,2)

c        apply similarity transformation to derivative matrix
         call thrmat( tranin, deraxs, tran, prod, 3, 3 )

c        load corrected derivatives back
c        Jz**2 term
         deriv(i+ioff,1) = prod(3,3)
c        Jx**2 term
         deriv(i+ioff,2) = prod(1,1)
c        Jy**2 term
         deriv(i+ioff,3) = prod(2,2)

c        zero derivative elements 4..6 since utilized in later calculations
         deriv(i+ioff,4) = 0.d0
         deriv(i+ioff,5) = 0.d0
         deriv(i+ioff,6) = 0.d0
      end do

      return
      end


c------------------------------------------------------------------------------
      integer function strlen( string )
c     determine the actual LENgth of a STRing

      implicit none

      integer        iend, length
      character*(*)  string

      length = len(string)
      do iend = length, 1, -1
         if (string(iend:iend).ne.' ')
     *        goto 1
      end do
    1 strlen = iend

      return
      end


c------------------------------------------------------------------------------
      subroutine thrmat( left, center, right, prod, dim, act )
c     evaluate the product of three general matrices: prod = left * center * right

      implicit none

      integer dim, act
      real*8  left( dim, dim ), center( dim, dim ), right( dim, dim ), prod( dim, dim )

      integer i, col, row
      real*8  help( 2 * ARNIROT_JMAX + 1, 2 * ARNIROT_JMAX + 1 )

      common /scratch/ help

      ARNIROT_LAUNCH ( "Launching thrmat." )

c     initialize product matrices to zero
      do col = 1, act
         do row = 1, act
            prod( row, col ) = 0.d0
            help( row, col ) = 0.d0
         end do
      end do
c     first matrix multiplication ( help = center * right )
      do col = 1, act
         do i = 1, act
            do row = 1, act
               help( row, col ) = help( row, col ) + center( row, i ) * right( i, col )
            end do
         end do
      end do
c     second matrix multiplication ( prod2 = matl * prod1 )
      do col = 1, act
         do i = 1, act
            do row = 1, act
               prod( row, col ) = prod( row, col ) + left( row, i ) * help( i, col )
            end do
         end do
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine vecvec( vec1, vec2, mat, dim, act )
c     outer product of vec1 and vec2
c     (multiply column VECtor vec1 with row VECtor vec2 to calculate a matrix)

      implicit none

      integer          dim, act
      double precision vec1( dim ), vec2( dim ), mat( dim, dim )

      integer          row, col

      ARNIROT_LAUNCH ( "Launching vecvec." )

c     initialize matrix
      call smat0( mat, dim, act )
c     multiply a vector times a vector
      do col = 1, act
         do row = 1, act
            mat( row, col ) = vec1( row ) * vec2( col )
         end do
      end do

      return
      end


c------------------------------------------------------------------------------
      subroutine vecsto ( evec, dmevec, vecmat, dmham, ivpt, Jmax, J )
c     STOre eigenvector coefficients from diagonalization routine into VECtor

      implicit none

      integer          Jmax, dmham, dmevec, ivpt(0:Jmax)
      double precision evec(dmevec), vecmat(dmham,dmham)

      integer          i, ic, J

      ARNIROT_LAUNCH ( "Launching vecsto." )
      
      i = ivpt(J) + 1
      do ic = 1, 2*J+1
         call dcopy( 2*J+1, vecmat( 1, ic ), 1, evec( i + (ic - 1)*(2*J+1) ), 1 )
      end do
      
      return
      end


c------------------------------------------------------------------------------
      subroutine wang( mat, dim, act )
c     generate the block diagonal Wang transformation matrix

      implicit none

      integer   act, dim, center, k, l
      real*8    val, mat( dim, dim )
      
c     val = 1 / sqrt( 2.d0 )
      parameter ( val = 0.7071067811865475244008444d0 )
      
      ARNIROT_LAUNCH ( "Launching wang." )
      
c     initialize transformation matrices
      call smat0( mat, dim, act )

c     fill Wang matrix elements
      center = ( act + 1 ) / 2
      l = act
      do k = 1, center - 1
         mat( k, k ) = -val
         mat( k, l ) =  val
         mat( l, k ) =  val
         mat( l, l ) =  val
         l = l - 1
      end do
      mat( center, center ) = 1.d0
      
      return
      end


c------------------------------------------------------------------------------
      subroutine trwang( trans, mat, prod, dim, act )
c     unitary Wang transformation of matrix mat: prod = trans^-1 * mat * trans

c     Very large savings are achieved ( compared to thrmat ) by using the
c     following matrix properties:
c     - trans^-1 == trans
c     - trans has elements on the diagonal and counter-diagonal only
c     - mat is pentadiagonal

      implicit none

      integer dim, act
      real*8  trans( dim, dim ), mat( dim, dim ), prod( dim, dim )

      integer col, row, i
      real*8  help( 2 * ARNIROT_JMAX + 1, 2 * ARNIROT_JMAX + 1 )

      common /scratch/ help

      ARNIROT_LAUNCH ( "Launching trwang." )

      if( act .lt. 5 ) then
c        Use thrmat for this small matrices, since we don't want to check all
c        the special conditions that would occur otherwise.
c        It's no problem with this small matrices anyhow.
         call thrmat( trans, mat, trans, prod, dim, act )
      else
c        initialize product matrices to zero
         do col = 1, act
            do row = 1, act
               prod( row, col ) = 0.d0
               help( row, col ) = 0.d0
            end do
         end do

c        first matrix multiplication ( help = mat * trans ), the special elements first
c        col = 1
         help( 1, 1 ) = help( 1, 1 ) + mat( 1, 1 ) * trans( 1, 1 )
         help( 2, 1 ) = help( 2, 1 ) + mat( 2, 1 ) * trans( 1, 1 )
         help( 3, 1 ) = help( 3, 1 ) + mat( 3, 1 ) * trans( 1, 1 )
         help( act - 2, 1 ) = help( act - 2, 1 ) + mat( act - 2, act ) * trans( act, 1 )
         help( act - 1, 1 ) = help( act - 1, 1 ) + mat( act - 1, act ) * trans( act, 1 )
         help( act, 1 )     = help( act, 1 )     + mat( act, act )     * trans( act, 1 )
c        col = 2
         help( 1, 2 ) = help( 1, 2 ) + mat( 1, 2 ) * trans( 2, 2 )
         help( 2, 2 ) = help( 2, 2 ) + mat( 2, 2 ) * trans( 2, 2 )
         help( 3, 2 ) = help( 3, 2 ) + mat( 3, 2 ) * trans( 2, 2 )
         help( 4, 2 ) = help( 4, 2 ) + mat( 4, 2 ) * trans( 2, 2 )
         help( act - 3, 2 ) = help( act - 3, 2 ) + mat( act - 3, act - 1 ) * trans( act - 1, 2 )
         help( act - 2, 2 ) = help( act - 2, 2 ) + mat( act - 2, act - 1 ) * trans( act - 1, 2 )
         help( act - 1, 2 ) = help( act - 1, 2 ) + mat( act - 1, act  - 1) * trans( act - 1, 2 )
         help( act, 2 )     = help( act, 2 )     + mat( act, act - 1 )     * trans( act - 1, 2 )
c        col = act - 1
         help( 1, act - 1 ) = help( 1, act - 1 ) + mat( 1, 2 ) * trans( 2, act - 1 )
         help( 2, act - 1 ) = help( 2, act - 1 ) + mat( 2, 2 ) * trans( 2, act - 1 )
         help( 3, act - 1 ) = help( 3, act - 1 ) + mat( 3, 2 ) * trans( 2, act - 1 )
         help( 4, act - 1 ) = help( 4, act - 1 ) + mat( 4, 2 ) * trans( 2, act - 1 )
         help( act - 3, act - 1 ) = help( act - 3, act - 1 ) + mat( act - 3, act - 1 ) * trans( act - 1, act - 1 )
         help( act - 2, act - 1 ) = help( act - 2, act - 1 ) + mat( act - 2, act - 1 ) * trans( act - 1, act - 1 )
         help( act - 1, act - 1 ) = help( act - 1, act - 1 ) + mat( act - 1, act - 1 ) * trans( act - 1, act - 1 )
         help( act, act - 1 )     = help( act, act - 1 )     + mat( act, act - 1 )     * trans( act - 1, act - 1 )
c        col = act
         help( 1, act ) = help( 1, act ) + mat( 1, 1 ) * trans( 1, act )
         help( 2, act ) = help( 2, act ) + mat( 2, 1 ) * trans( 1, act )
         help( 3, act ) = help( 3, act ) + mat( 3, 1 ) * trans( 1, act )
         help( act - 2, act ) = help( act - 2, act ) + mat( act - 2, act ) * trans( act, act )
         help( act - 1, act ) = help( act - 1, act ) + mat( act - 1, act ) * trans( act, act )
         help( act, act )     = help( act, act )     + mat( act, act )     * trans( act, act )
c        loop over all the other cols
         do col = 3, act - 2
            i = col
            do row = i - 2, i + 2
               help( row, col ) = help( row, col ) + mat( row, i ) * trans( i, col )
            end do
            i = act + 1 - col
            do row = i - 2, i + 2
               help( row, col ) = help( row, col ) + mat( row, i ) * trans( i, col )
            end do
         end do        
c        These were counted twice yet, need to be substracted
         i = ( act + 1 ) / 2
c        i >= 3, since act >= 5 if we get here
         do col = 1, act
            do row = i - 2, i + 2
               help( row, col ) = help( row, col ) - mat( row, i ) * trans( i, col )
            end do
         end do
         
c        second matrix multiplication ( prod = trans * help )
         do col = 1, act
            do i = 1, act
               row = i
               prod( row, col ) = prod( row, col ) + trans( row, i ) * help( i, col )
               row = act + 1 - i
               prod( row, col ) = prod( row, col ) + trans( row, i ) * help( i, col )
            end do
         end do
c        These were counted twice yet, need to be substracted
         row = ( act + 1 ) / 2
         do col = 1, act
            do i = 1, act
               prod( row, col ) = prod( row, col ) - trans( row, i ) * help( i, col )
            end do
         end do
      end if
      
      return
      end


c------------------------------------------------------------------------------
      subroutine bkwang( trans, mat, dim, act )
c     do Wang backtransformation of mat: mat = trans * mat

c     Very large savings ( compared to matmat ) are achieved by using the
c     following matrix properties:
c     - trans has elements on the diagonal and counter-diagonal only
c     - mat is  general square matrix

c     input:  trans, dim, act
c     inout:  mat

      implicit none

      integer dim, act
      real*8  trans( dim, dim ), mat( dim, dim )

      integer col, row
      real*8  help( 2 * ARNIROT_JMAX + 1, 2 * ARNIROT_JMAX + 1 )

      common /scratch/ help

      ARNIROT_LAUNCH ( "Launching bkwang." )

c     initialize intermediate matrix to zero
      do col = 1, act
         do row = 1, act
            help( row, col ) = 0.d0
         end do
      end do
c     calculate product
      do col = 1, act
         do row = 1, act
            help( row, col ) = help( row, col ) + trans( row, row )           * mat( row, col )
            help( row, col ) = help( row, col ) + trans( row, act + 1 - row ) * mat( act + 1 - row, col )
         end do
      end do
c     These were counted twice yet, need to be substracted
      row = ( act + 1 ) / 2
      do col = 1, act
         help( row, col ) = help( row, col ) - trans( row, row ) * mat( row, col )
      end do
c     copy result to mat
      do col = 1, act
         do row = 1, act
            mat( row , col ) = help( row, col )
         end do
      end do
      
      return
      end


cc Local Variables:
cc mode: FORTRAN
cc End:
