// -*- mode: C++; c-file-style: "Stroustrup" -*-
//
// This file is part of krot,
// a program for the simulation, assignment and fit of HRLIF spectra.
//
// Copyright (C) 1998,1999 Jochen Kpper
//
// This program is free software; you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation; either version 2 of the License, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but WITHOUT
// ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
// FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// this program; see the file License. if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA
//
// If you use this program for your scientific work, please cite it according to
// the file CITATION included with the package.



#include "arnirot.h"
#include "calculationParameter.h"
#include "krot.h"
#include "transition.h"

#include <kapp.h>
#include <kmsgbox.h>
#include <kprocess.h>

#include <qmessagebox.h>

#include <map>
#include <string>
#include <strstream>
#include <vector>



Arnirot::Arnirot( const CalculationParameter& param, const mapAssignment& asn )
    : QObject(),
      idx( 0 ),
      input(),
      output(),
      pars( param ),
      process()
{
    KROT_LAUNCH( "Launching Arnirot constructor" );
    msgbox = new QMessageBox( i18n( "Arnirot" ), i18n( "Arnirot calculation is running." ),
			      QMessageBox::Information, QMessageBox::Cancel | QMessageBox::Escape, 0, 0,
			      0, "Arnirot MessageBox", false );
    msgbox->setButtonText( QMessageBox::Cancel, i18n( "Cancel" ) );
    //    connect( msgbox, SIGNAL(),
    //	     this, SLOT( abort() ) );
    // put fully qualified program name into command line
    QString prog( kapp->kde_bindir() );
    prog += "/krot-arnirot";
    // prog = "/bin/cat";
    // prog = "/home/jochen/source/krot/krot/krot/arnirot/krot-arnirot";
    process << prog.data();
    connect( &process, SIGNAL( wroteStdin( KProcess * ) ), 
	     this, SLOT( send( KProcess * ) ) );
    connect( &process, SIGNAL( receivedStdout( KProcess *, char *, int ) ),
	     this, SLOT( readStdout( KProcess *, char *, int ) ) );
    connect( &process, SIGNAL( processExited( KProcess * ) ), 
	     this, SLOT( finished( KProcess * ) ) );
    return;
}



Arnirot::~Arnirot()
{
    KROT_LAUNCH( "Launching Arnirot destructor" );
    process.kill();
    return;
}
    


void Arnirot::send( KProcess * )
{
    KROT_LAUNCH1( "Launching Arnirot::send, idx = %d", idx );
    // generate parameter input and send it arnirot
    // this looks like a pretty poor solution, if you have any ideas, change it or tell me (Jochen)
    char buffer[128];
    switch( idx ) {
    case 0:
	sprintf( buffer, "-1\n" );  // indicate short formats
	input += buffer;
	sprintf( buffer, "%f %d %f %d \n",
		 pars.constant( CalculationParameter::GroundState, CalculationParameter::A ),
		 pars.fit( CalculationParameter::GroundState, CalculationParameter::A ),
		 pars.constant( CalculationParameter::ExcitedState, CalculationParameter::A ),
		 pars.fit( CalculationParameter::ExcitedState, CalculationParameter::A ) );
	input += buffer;
	sprintf( buffer, "%f %d %f %d\n",
		 pars.constant( CalculationParameter::GroundState, CalculationParameter::B ),
		 pars.fit( CalculationParameter::GroundState, CalculationParameter::B ),
		 pars.constant( CalculationParameter::ExcitedState, CalculationParameter::B ),
		 pars.fit( CalculationParameter::ExcitedState, CalculationParameter::B ) );
	input += buffer;
	sprintf( buffer, "%f %d %f %d\n",
		 pars.constant( CalculationParameter::GroundState, CalculationParameter::C ),
		 pars.fit( CalculationParameter::GroundState, CalculationParameter::C ),
		 pars.constant( CalculationParameter::ExcitedState, CalculationParameter::C ),
		 pars.fit( CalculationParameter::ExcitedState, CalculationParameter::C ) );
	input += buffer;
	sprintf( buffer, "%f %d %f %d\n",
		 pars.constant( CalculationParameter::GroundState, CalculationParameter::DX ),
		 pars.fit( CalculationParameter::GroundState, CalculationParameter::DX ),
		 pars.constant( CalculationParameter::ExcitedState, CalculationParameter::DX ),
		 pars.fit( CalculationParameter::ExcitedState, CalculationParameter::DX ) );
	input += buffer;
	sprintf( buffer, "%f %d %f %d\n",
		 pars.constant( CalculationParameter::GroundState, CalculationParameter::DY ),
		 pars.fit( CalculationParameter::GroundState, CalculationParameter::DY ) ? 1 : 0,
		 pars.constant( CalculationParameter::ExcitedState, CalculationParameter::DY ),
		 pars.fit( CalculationParameter::ExcitedState, CalculationParameter::DY ) ? 1 : 0 );
	input += buffer;
	sprintf( buffer, "%f %d %f %d\n",
		 pars.constant( CalculationParameter::GroundState, CalculationParameter::DZ ),
		 pars.fit( CalculationParameter::GroundState, CalculationParameter::DZ ) ? 1 : 0,
		 pars.constant( CalculationParameter::ExcitedState, CalculationParameter::DZ ),
		 pars.fit( CalculationParameter::ExcitedState, CalculationParameter::DZ ) ? 1 : 0 );
	input += buffer;
	sprintf( buffer, "%f %d %f %d\n",
		 pars.constant( CalculationParameter::GroundState, CalculationParameter::DJ ),
		 pars.fit( CalculationParameter::GroundState, CalculationParameter::DJ ) ? 1 : 0,
		 pars.constant( CalculationParameter::ExcitedState, CalculationParameter::DJ ),
		 pars.fit( CalculationParameter::ExcitedState, CalculationParameter::DJ ) ? 1 : 0 );
	input += buffer;
	sprintf( buffer, "%f %d %f %d\n",
		 pars.constant( CalculationParameter::GroundState, CalculationParameter::DJK ),
		 pars.fit( CalculationParameter::GroundState, CalculationParameter::DJK ) ? 1 : 0,
		 pars.constant( CalculationParameter::ExcitedState, CalculationParameter::DJK ),
		 pars.fit( CalculationParameter::ExcitedState, CalculationParameter::DJK ) ? 1 : 0 );
	input += buffer;
	sprintf( buffer, "%f %d %f %d\n",
		 pars.constant( CalculationParameter::GroundState, CalculationParameter::DK ),
		 pars.fit( CalculationParameter::GroundState, CalculationParameter::DK ) ? 1 : 0,
		 pars.constant( CalculationParameter::ExcitedState, CalculationParameter::DK ),
		 pars.fit( CalculationParameter::ExcitedState, CalculationParameter::DK ) ? 1 : 0 );
	input += buffer;
	sprintf( buffer, "%f %d %f %d\n",
		 pars.constant( CalculationParameter::GroundState, CalculationParameter::dJ ),
		 pars.fit( CalculationParameter::GroundState, CalculationParameter::dJ ) ? 1 : 0,
		 pars.constant( CalculationParameter::ExcitedState, CalculationParameter::dJ ),
		 pars.fit( CalculationParameter::ExcitedState, CalculationParameter::dJ ) ? 1 : 0 );
	input += buffer;
	sprintf( buffer, "%f %d %f %d\n",
		 pars.constant( CalculationParameter::GroundState, CalculationParameter::dK ),
		 pars.fit( CalculationParameter::GroundState, CalculationParameter::dK ) ? 1 : 0,
		 pars.constant( CalculationParameter::ExcitedState, CalculationParameter::dK ),
		 pars.fit( CalculationParameter::ExcitedState, CalculationParameter::dK ) ? 1 : 0 );
	input += buffer;
	sprintf( buffer, "%f %d\n",
		 pars.constant( CalculationParameter::Origin ),
		 pars.fit( CalculationParameter::Origin ) );
	input += buffer;
	sprintf( buffer, "%f %f %f\n",
		 pars.constant( CalculationParameter::HybridA ),
		 pars.constant( CalculationParameter::HybridB ),
		 pars.constant( CalculationParameter::HybridC ) );
	input += buffer;
	sprintf( buffer, "%f %f %f\n",
		 pars.constant( CalculationParameter::Temperature ),
		 pars.constant( CalculationParameter::Temperature2 ) == 0.0 ? 1.0 : pars.constant( CalculationParameter::Temperature2 ),
		 pars.constant( CalculationParameter::TemperatureWeight ) );
	input += buffer;
	sprintf( buffer, "%d %d %d %d\n",
		 pars.constant( CalculationParameter::NSSWee ),
		 pars.constant( CalculationParameter::NSSWeo ),
		 pars.constant( CalculationParameter::NSSWoe ),
		 pars.constant( CalculationParameter::NSSWoo ) );
	input += buffer;
	sprintf( buffer, "%d %d\n", 0, 0 ); // axis reorientation
	input += buffer;
	sprintf( buffer, "%f %f\n", 0.0, 0.0 );
	input += buffer;
	sprintf( buffer, "%f %f\n", 0.0, 0.0 );
	input += buffer;
	sprintf( buffer, "%f %f\n", 0.0, 0.0 );
	input += buffer;
	sprintf( buffer, "%d\n", 0 );  // transition moment transformation
	input += buffer;
	sprintf( buffer, "%d %d\n",
		 pars.constant( CalculationParameter::Jmax ),
		 pars.constant( CalculationParameter::DKmax ) );
	input += buffer;
	sprintf( buffer, "0.0001\n" ); // cutoff
	input += buffer;
	sprintf( buffer, "%d\n", pars.fit() );
	input += buffer;
	sprintf( buffer, "%f\n",
		 pars.constant( CalculationParameter::LPU ) );
	input += buffer;
	if( false == process.writeStdin( input.c_str(), input.length() ) ) {
	    msgbox->hide();
	    KMsgBox::message( 0, i18n( "Arnirot error" ),
			      i18n( "Could not send data to arnirot,\nquitting calculation !" ),
			      KMsgBox::EXCLAMATION, i18n( "Close" ) );
	    process.closeStdin();
	    kill();
	}
	KROT_DEBUG_VERBOSE( input.c_str() );
	break;
    case 1:
	// write assignments
    default:
	process.writeStdin( "", 3 );
	process.closeStdin();
	break;
    }
    idx++;
    return;
}



void Arnirot::start()
{
    KROT_LAUNCH( "Launching Arnirot::start" );
    msgbox->show();
    process.start( KProcess::NotifyOnExit, KProcess::All );
    idx = 0;
    send( &process );
    return;
}    



void Arnirot::finished( KProcess *proc )
{
    KROT_LAUNCH( "Launching Arnirot::finished" );
    KROT_DEBUG_VERBOSE( output.c_str() );
    msgbox->hide();
    if( proc->normalExit() ) {
	if( proc->exitStatus() != 0 )
	    KMsgBox::message( 0, i18n( "Arnirot error" ),
			      i18n( "Arnirot has run into an error" ),
			      KMsgBox::EXCLAMATION, i18n( "Close" ) );
	else {
	    // simulation or fit ?
	    istrstream stream( output.c_str() );
	    int fit;
	    Transition trans;
	    vector< Transition > lines;
	    stream >> fit;
	    if( fit != 0 ) {
		// fitting result
		
	    }
	    // lines
	    stream >> trans;
	    while( ! stream.eof() ) {
		lines.push_back( trans );
		stream >> trans;
	    }
	    QString msg;
	    msg.sprintf( "Arnirot calculated %d lines.", lines.size() );
	    emit message( msg.data() );
	    emit done( pars, lines );
	}
    } else {
	KMsgBox::message( 0, i18n( "Arnirot error" ),
			  i18n( "Arnirot has not finished normally" ),
			  KMsgBox::EXCLAMATION, i18n( "Close" ) );
    }
    return;
}



void Arnirot::readStdout( KProcess *, char *buffer, int len )
{
    KROT_LAUNCH( "Launching Arnirot::readStdout" );
    output.append( buffer, len );
    return;
}
