/*
 * Driver for the PCL-812PG dsp card. By dani@enpl.es 
 */
#include <linux/module.h>
#include <linux/kernel.h>

#include <linux/errno.h>
#include <linux/mm.h>
#include <linux/ioport.h>
#include <linux/malloc.h>
#include <linux/interrupt.h>
#include <linux/timer.h>
#include <linux/fcntl.h>
#include <asm/segment.h>
#include <linux/delay.h>
#include <asm/io.h>
#include <asm/dma.h>
#include "pcl.h"


static int pcl_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long argument);
static int pcl_read(struct inode * inode, struct file * file, char * buf, int count);
static int pcl_write(struct inode *, struct file *,const char *, int);
static int pcl_open(struct inode *, struct file *);
void pcl_Irq_Probe_Handler(int irq, void *dev, struct pt_regs *regs);
void pcl_IrqHandler(int irq, void *dev, struct pt_regs *regs);
void pcl_release(struct inode *, struct file *);

static struct file_operations pcl_fops=
{
   NULL,
   pcl_read,
   pcl_write,
   NULL,                   /*  pcl_readaddr            */
   NULL,                   /*  pcl_select              */
   pcl_ioctl,              /*  pcl_ioctl               */
   NULL,                   /*  pcl_mmap                */
   pcl_open,               /*  pcl_open                */
   pcl_release,            /*  pcl_close               */
   NULL,                   /*  pcl_fsync               */
   NULL,                   /*  pcl_fasync              */
   NULL,                   /*  pcl_check_media_change  */
   NULL                    /*  pcl_revalidate          */
};






unsigned int addresses[]={0x200,0x210,0x220,0x230,0x300,0x3f0};
unsigned int base_address=0x200;
unsigned int data;
unsigned int irq=2;
static unsigned int dma_ch;
unsigned short mode;
struct wait_queue *waitq;

unsigned int IRQ,DMA;

static int pcl_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long argument)
{

 unsigned char lower=(unsigned char)argument;
 unsigned char higher=(unsigned char)(argument>>8);
 
 switch(cmd)
 {
 	case PCL_SET_MODE_CONTROL:
 		{
 		 if( ((lower==PCL_SET_MODE_INTERRUPT)&&(!irq))
 		 	||
 		 ((lower==PCL_SET_MODE_DMA)&&(!dma_ch)))
 		  return(-ENODEV);
 		  
 		 mode=lower;
 		 outb(lower,base_address+11);
 		 break;
 		}
	case PCL_SET_TIMER_CONTROL:
		{
		 
		/* Write to timer control register, it's up to the user 
		 * to control the mode, etc..
		 */
		 
		 /* OR 48 -> always user RDWR_MSB_LSB */
		 		 		
		 outb(lower|48,base_address+3);   
		 break;

		}
	case PCL_READ_COUNTER1:
		{
		 /* Before reading counters, latch value */
	         outb(64,base_address+3);      
 		
		 lower=inb(base_address+1);
		 higher=inb(base_address+1);
		 
		 
		 put_fs_byte(lower,(unsigned long *) argument++); /* Copy lsb */
		 put_fs_byte(higher,(unsigned long *) argument);  /* Copy msb */
		 
		 break;
		}
	case PCL_READ_COUNTER2:
		{
		 /* Before reading counters, latch value */
		 outb(128,base_address+3); 
		
		 lower=inb(base_address+2);
		 higher=inb(base_address+2);

		
		 put_fs_byte(lower,(unsigned long *) argument++); /* Copy lsb */
		 put_fs_byte(higher,(unsigned long *) argument);  /* Copy msb */
		 
		 break;
		}
	case PCL_SET_COUNTER2:
		{
		 outb(lower,base_address+2);
		 outb(higher,base_address+2);
		 break;
		}
	case PCL_SET_COUNTER1:
		{
		 outb(lower,base_address+1);
		 outb(higher,base_address+1);
		 break;
		}
	case PCL_SET_GAIN:
		{
		 if(lower<5) outb(lower,base_address+9);
		 else 
		  {
		   printk("PCL:Invalid gain specified.\n");
		   return(-EINVAL);
		  }
		 break;
		} 	
 
 }

 return(0);

}

void pcl_probe_wakeup(unsigned long wait_queue)
{

 wake_up_interruptible(&waitq);

}
static unsigned int probe_dma(unsigned int irqnum)
{

 int channel=3;
 char found=0;
 unsigned char *temp;
 struct timer_list probe_timer;

 request_irq(irqnum,pcl_IrqHandler,0,"pcl-182pg",NULL);
 temp=(unsigned char *)kmalloc(10,3); 
 
 
 while((!found)&&(channel>0))
 {
 
    request_dma(channel,"pcl-812pg");
  
    cli();

    disable_dma(channel);
    clear_dma_ff(channel);
    set_dma_mode(channel,DMA_MODE_READ);
    set_dma_addr(channel,virt_to_phys(temp));
    set_dma_count(channel,10);
    enable_dma(channel);
    
    sti();
    	    
    init_timer(&probe_timer);
    probe_timer.expires=CONVERSION_TIME;
    probe_timer.data=(unsigned long) &waitq;
    probe_timer.function=pcl_probe_wakeup;
     
    
    outb(0,base_address+10);	/* select input chanel */
    outb(2,base_address+11);	        /* Set dma mode */ 
     
    outb(3,base_address+8);		/* clear interrupt enable */

    add_timer(&probe_timer); 
    
    interruptible_sleep_on(&waitq);      
    del_timer(&probe_timer);
     
    if(!get_dma_residue(channel))	/* If dma transfer completed ok */
	found++;   
    else {
          free_dma(channel);
          if(channel==1)	
          	{
          	
          	 kfree(temp);
 		 free_irq(irqnum,NULL);
		 return(0);
          	}
          channel=1;
 	 }
 }


 kfree(temp);
 free_irq(irqnum,NULL);
 return(channel);

}
/* --------------------- PCL_READ_DMA ----------------------*/

static int pcl_read_dma(struct inode *inode, struct file *file, char *buf, int count)
{
 unsigned int copy_size,total_bytes_written=0;
 int minor=MINOR(inode->i_rdev);
 unsigned char *temp,*bak;
/* int tt; */
 
 
 /* Check for valid buf */
 
 if(verify_area(VERIFY_READ,buf,count)==-EFAULT)
   return (-EFAULT);
   
temp=(unsigned char *)kmalloc(DMA_BUFFER,3);
 bak=temp;
 do
 {
  copy_size=(count<=DMA_BUFFER? count: DMA_BUFFER);
  if(copy_size)
   {
 

    		
	/*-------*/
    cli();

    disable_dma(dma_ch);
    clear_dma_ff(dma_ch);
    set_dma_mode(dma_ch,DMA_MODE_READ);
    set_dma_addr(dma_ch,virt_to_phys(temp));
    set_dma_count(dma_ch,copy_size);
    enable_dma(dma_ch);
    
    sti();	    
    outb(minor,base_address+10);	/* select input chanel */
    outb(2,base_address+11);	        /* Set dma mode */ 
     
    outb(3,base_address+8);
     
    interruptible_sleep_on(&waitq);      
     
    memcpy_tofs(buf+total_bytes_written,temp,copy_size);
   
   	/*-------*/
   
    total_bytes_written+=count;
    count-=copy_size;
   }
 
 } while (count>0);

 kfree(temp);
 
   return(total_bytes_written);
}

/* -------------------- */

static int pcl_read_polled(struct inode *inode, struct file *file, char *buf, int count)
{
 int i=0;
 unsigned char higher,lower;
 int minor=MINOR(inode->i_rdev);
 

 /* Check for valid buf */
 if(verify_area(VERIFY_READ,buf,count)==-EFAULT)
   return (-EFAULT);
 if(minor<16)			/* Read analog input */
  {
   outb(minor,base_address+10);  /* Select input chanel */
    while (i<count)
     {

      /* Start an ADC conversion and wait for DataReady */
    

    	outb(5,base_address+12);  /* any value causes trigger */
    	/* wait.. */
    	udelay(CONVERSION_TIME);
        if((higher=(inb(base_address+5)))>=16)
          return(-EINVAL);	/* Nothing readen */
  	
  	lower=inb(base_address+4);
  	
 /*	higher-=8;		 Bypass the DATA_READY */
  	memcpy_tofs(buf++,&lower,1); /* Copy lower byte first  */
  	i++;
  	if(i<count) 
  	 {
  	  memcpy_tofs(buf++,&higher,1);
  	  i++;
  	 }
  	
     }
  }
 else if(minor==18) 		/* Read digital input dio */
  {
   while(i<count)
    {
       lower=inb(base_address+6);	/* read LSB        */
       higher=inb(base_address+7);      /* and add the MSB */

       memcpy_tofs(buf++,&lower,1);
       i++;
       if(i<count)
        {
         memcpy_tofs(buf++,&higher,1);
         i++;
        }
    }
  }
 else return(-ENODEV); 
  
   return(i);
}


static int pcl_write(struct inode * inode, struct file * file,const char * buf, int count)
{
 int i=0;
 unsigned char higher,lower; 
 int minor=MINOR(inode->i_rdev);
 if(verify_area(VERIFY_WRITE,buf,count)==-EFAULT) return (-EFAULT);
 if((minor==16)||(minor==17)) 	/* Write analog output */
 	{
	  outb(minor-16,base_address+10);  /* Select output channel */
	  while (i<count)
	  {
	    
	  lower=get_fs_byte(buf++);
	  i++;
	  if(i<count)  /* Write all 12 bits */
	   {
	    higher=get_fs_byte(buf++);
	    i++;
	   }
	  else    /* Only write 8 bits (8 lower bits) */
	    higher=0;	   
	
	/* Got high 4 bits, and 8 low bits, now write to card */
	
	  outb(lower,base_address+4+(2*(minor-16)));       /* write 8 LSB */
	  outb(higher,base_address+5+(2*(minor-16)));      /* write 8(4) MSB */
	  udelay(30);		/*  Wait 30us for stable ouput */ 
	    
	  } /* end while */
	  
	}
 
 /* else write digital output */
 else
  if(minor==18) 		/* Write to digital output (dio) */
    while(i<count)
     {
     
     
       lower=get_fs_byte(buf++);
	  i++;
	  if(i<count) /* Only write 8 bits (8 lower bits) */
	   {
	    higher=get_fs_byte(buf++);
	    i++;
	   }
	  else
	    higher=0;	    
	  
	/* Got high 4 bits, and 8 low bits, now write to card */	  
	  
	  outb(lower,base_address+13);       /* write 8 LSB */
	  outb(higher,base_address+14);      /* write 8(4) MSB */
	    
	     
     } /* end while */
	  
     
  else 
  	{
  	 return(-ENODEV);  	
 	}
  return(i);

}

void pcl_IrqHandler(int irq, void *dev, struct pt_regs *regs)
{

 wake_up_interruptible(&waitq);

}

/*-------------------------------------------------------*/
/* pcl_read_int:					 */
/* 	We know that minor < 16, we're reading /dev/adc* */
/*-------------------------------------------------------*/

static int pcl_read_int(struct inode *inode, struct file *file, char *buf, int count)
{
 
 int i=0;
 unsigned char higher,lower;
 int minor=MINOR(inode->i_rdev);
 


 /* Check for valid buf */
 if(verify_area(VERIFY_READ,buf,count)==-EFAULT) return (-EFAULT);

 outb(minor,base_address+10);  /* Select input chanel */
    while (i<count)
     {

      /* Start an ADC conversion and wait for DataReady */
	

    	/* Now we go to sleep, the interrupt will wake_up */
        outb(1,base_address+8);
	sti();	
    	interruptible_sleep_on(&waitq);

        if((higher=(inb(base_address+5)))>=16)
          return(-EINVAL);	/* Nothing readen */
  
  	outb(1,base_address+8);
  	lower=inb(base_address+4);

     /*  higher-=8;		 Bypass the DATA_READY */
	
  	memcpy_tofs(buf++,&lower,1); /* Copy lower byte first  */
  	i++;
  	if(i<count) 
  	 {
  	  memcpy_tofs(buf++,&higher,1);
  	  i++;
  	 }
  	
    }
    return(i);
}



/*------------------------------------------------------*/

static int pcl_read(struct inode *inode, struct file *file, char *buf, int count)
{

 int minor=MINOR(inode->i_rdev);

switch(mode)
{
 case PCL_SET_MODE_POLLING:
 		{
 		return(pcl_read_polled(inode,file,buf,count));
 		}
 case PCL_SET_MODE_INTERRUPT:
 		{
 		if(minor<16)
 		   return(pcl_read_int(inode,file,buf,count));
  			else	  /* Read digital input  (no interrupts) */
   			return(pcl_read_polled(inode,file,buf,count));
 		}
 case PCL_SET_MODE_DMA:
 		{
 		
 		if(minor<16)
 		{
 	   	return(pcl_read_dma(inode,file,buf,count));
 	   	}
  			else	  /* Read digital input  (no interrupts) */
   			return(pcl_read_polled(inode,file,buf,count));	
 		
 		}

 }

 return(-ENODEV);	/* Noexistent mode */
}
static int pcl_open(struct inode * i, struct file *f)
{
 return(0);
}

void pcl_release(struct inode * i, struct file *f)
{
 
}
void pcl_Irq_Probe_Handler(int irq, void *dev, struct pt_regs *regs)
{

        data=33;
        wake_up_interruptible(&waitq);
		
}
int init_module(void) 
{
  unsigned short found=0;
  unsigned short offset=0;
  struct timer_list probe_timer;   
  base_address=addresses[offset];
 
 
   while((!found)&&(offset<5))
   {
    if(!check_region(base_address,16))
    {
    	/* Start an ADC conversion and wait for DataReady */
     	outb(0,base_address+10);  /* Select input chanel */
     	outb(0,base_address+9);	  /* Set Gain = 1 */
     	outb(1,base_address+11);  /* Set pooling mode */
     	
    	
    	outb(5,base_address+12);  /* any value causes trigger */
    	/* wait.. */
    	udelay(CONVERSION_TIME); 


        if((inb(base_address+5))<16)
           found++;
        else base_address=addresses[++offset];  /* Keep searching*/
	
    }
    	else base_address=addresses[++offset];  /* Keep searching*/
   }   
   if(!found)
   {
    printk("No PCL-812PG cards found.\n");
    return(-ENODEV);
   }
   
/*  Now check for its irq */

if(IRQ) irq=IRQ;
 else
{		/* Try to search interrupt */

   found=0; irq=2;
   while((!found)&&(irq<8))	/* irq 2,3,4,5,6,7 */
   {
   /* check if current irq is availiable */
   if(request_irq(irq,pcl_Irq_Probe_Handler,0,"pcl-182pg",NULL))
     {
      irq++;
     }
  else
   {
    
    /* Set timers to trigger and interrupt */
    
    outb(0,base_address+10);  /* Select input chanel */
    outb(0,base_address+9);   /* Set Gain = 1 */
    outb(6,base_address+11);  /* Set interrupt mode */



    data=1;


    outb(PCL_SET_COUNTER2|PCL_COUNTER_RW_LSB_MSB|
    	 PCL_COUNTER_RATE_GEN,base_address+3);

    outb(PCL_SET_COUNTER1|PCL_COUNTER_RW_LSB_MSB|
    	 PCL_COUNTER_RATE_GEN,base_address+3);


    outb(40,base_address+2);
    outb(0,base_address+2);
  
    outb(40,base_address+1);
    outb(0,base_address+1);




    init_timer(&probe_timer);
    probe_timer.expires=10*CONVERSION_TIME;
    probe_timer.data=(unsigned long) &waitq;
    probe_timer.function=pcl_probe_wakeup;
     

    outb(1,base_address+8);        
    add_timer(&probe_timer); 
    
    interruptible_sleep_on(&waitq);      
    del_timer(&probe_timer);

    if(data==33)
    	{
    	 found++;
    	 outb(1,base_address+8);
    	 free_irq(irq,NULL);
       
    	}
    	else
    	{
    	 free_irq(irq,NULL);
    	 irq++;		/* check next irq */   
   	}
   
   }  
  
  
  }
}  	/* end of irq search */

  if(irq==8)		/* No irq found */
    {
     irq=0;
     printk("PCL-812PG found at 0x%x, pooling, using major %d\n",base_address,PCL_MAJOR); 
    }
 else
  {
   if(DMA) dma_ch=DMA; 		/* use dma specified by insmod */
      else dma_ch=probe_dma(irq);  /* search dma channel */
      
   if(!dma_ch)
   { 
    printk("PCL-812PG found at 0x%x, IRQ%x, using major %d\n",base_address,irq,PCL_MAJOR);
   }
   else
   {
    printk("PCL-812PG found at 0x%x, IRQ%x, DMA%x using major %d\n",base_address,irq,dma_ch,PCL_MAJOR);
   }
 
  if(register_chrdev(PCL_MAJOR, "pcl-812pg", &pcl_fops))
    {
      printk("unable to get major number %d\n", PCL_MAJOR);
      return -EBUSY;
    }
  }
  request_region(base_address,16,"pcl-812pg");
  if(irq) 
    {
    
     request_irq(irq,pcl_IrqHandler,0,"pcl-812pg",NULL);
    
    }
  if(dma_ch)
   {
   
    request_dma(dma_ch,"pcl-812pg");
   
   }  

   outb(1,base_address+11);	/* set pooling mode for default */
   mode=PCL_SET_MODE_POLLING;	/* default mode */  
   return 0;
}



/*  Global structures for card info.                                         */



void cleanup_module(void)
{
  
   if(unregister_chrdev(PCL_MAJOR, "pcl-812pg")) 
   {
      printk("device unregistering failed\n");
   }
   else 
   {
   
  /* Too fast frequency may drive the card nuts */
  
    outb(40,base_address+2);
    outb(0,base_address+2);
  
    outb(40,base_address+1);
    outb(0,base_address+1);
 
      release_region(base_address,16);
      if(irq) 
      	 free_irq(irq,NULL);
      if(dma_ch)
       free_dma(dma_ch);

      printk("Pcl-812pg driver unloaded\n");
      
   }
}



