#include <math.h>
#include <qdrawutl.h>
#include "qwt_sldbase.h"

#include "qwt_sldbase.moc"

//---------------------------------------------------------
//.H QwtSliderBase|3|27/04/97|Qwt Widget Library|Qwt Programmer's manual
//
//.S NAME
//	QwtSliderBase
//
//.S	SYNOPSIS
//	#include <qwt_sldbase.h>
//.S INHERITED CLASSES
//	QWidget (virtual), \l{qwt_drange.html}{QwtDblRange}
//.S PUBLIC MEMBERS
//.t
//	\R{QwtSliderBase::QwtSliderBase}( QWidget *parent = 0, const char *name = 0 ) --
//							constructor
//	~\R{QwtSliderBase::QwtSliderBase}() -- destructor
//	void \R{QwtSliderBase::stopMoving}() -- stop moving 
//	void \R{QwtSliderBase::setTracking}(bool enable)  --
//		enable or disable tracking
//
//.S SIGNALS
//.t
//  void valueChanged(double) -- 
//	Notifies a change of value. In the default setting 
//	(tracking enabled), this signal will be emitted every 
//	time the value changes ( see setTracking() ). 
//
//  void sliderPressed() --
//	This signal is emitted when the user presses the 
//	moveable part of the slider (start ScrMouse Mode).
//
//  void sliderReleased() --
//	This signal is emitted when the user releases the
//	moveable part of the slider.
//	
//  void sliderMoved(double value) --
//	This signal is emitted when the user moves the
//	slider with the mouse.
//
//.S	PUBLIC SLOTS
//.t
//	void \R{QwtSliderBase::moveTo}(double val) -- move to value
//	void \R{QwtSliderBase::adjustTo}(double val) --
//		move to nearest integer multiple of the step size
//.S PROTECTED MEMBERS
//.t
//	void \R{QwtSliderBase::setUpdateTime}(int t) --
//			Change the update time for automatic scrolling
//	virtual void \R{QwtSliderBase::setMass}(double val) --
//		set the mass for flywheel effect
//	void \R{QwtSliderBase::setPosition}(const QPoint &p) -- adjust the position
//	virtual void \R{QwtSliderBase::valueChange}() --  notify change of value
//	virtual double \R{QwtSliderBase::mass}() const -- return the mass
//
//	virtual double \R{QwtSliderBase::getValue}(const QPoint & p) = 0 --
//		Determine the value corresponding to a point p
//	virtual void \R{QwtSliderBase::getScrollMode}( const QPoint &p, int &scrollMode, int &direction) = 0 --
//			     Determine the scrolling mode corresponding to a point p
//.S	DESCRITPTION
//	QwtSliderBase is a base class which can be used to build
//	slider-like widgets. QwtSliderBase handles the mouse events
//	and updates the slider's value accordingly. Derived classes
//	only have to implement the \R{QwtSliderBase::getValue} and 
//	\R{QwtSliderBase::getScrollMode} members, and should react to a
//	\R{QwtSliderbase::valueChange}, which normally requires repainting. 
//-----------------------------------------------------------


//------------------------------------------------------------
//
//.S	QwtSliderBase::QwtSliderBase
//	Constructor
//
//.s	Parameters
//.p	QWidget *parent, const char *name
//
//.s	Syntax
//.f	 QwtSliderBase::QwtSliderBase(QWidget *parent, const char *name)
//
//------------------------------------------------------------
QwtSliderBase::QwtSliderBase(QWidget *parent, const char *name)
: QWidget(parent,name)
{
	d_tmrID = 0;
	d_updTime = 150;
	d_mass = 0.0;
	d_tracking = TRUE;
	d_mouseOffset = 0.0;
}

//------------------------------------------------------------
//
//.S	QwtSliderBase::~QwtSliderBase
//	Destructor
//
//.s	Syntax
//.f	 QwtSliderBase::~QwtSliderBase()
//
//------------------------------------------------------------
QwtSliderBase::~QwtSliderBase()
{
	if(d_tmrID) killTimer(d_tmrID);
}

//------------------------------------------------------------
//
//.S	QwtSliderBase::stopMoving
//	Stop updating if automatic scrolling is active
//
//.s	Syntax
//.f	void QwtSliderBase::stopMoving()
//
//------------------------------------------------------------
void QwtSliderBase::stopMoving() 
{
	if(d_tmrID) killTimer(d_tmrID);
}


//------------------------------------------------------------
//
//.S	QwtSliderBase::setUpdateTime
//	Specify the update interval for automatic scrolling
//
//.s	Parameters
//.p	int t	-- update interval in milliseconds
//
//.s	Syntax
//.f	void QwtSliderBase::setUpdateTime(int t)
//
//.s	See also
//	\R{QwtSliderBase::getScrollMode}
//
//------------------------------------------------------------
void QwtSliderBase::setUpdateTime(int t) 
{
	if (t < 0) t = -t;
	d_updTime = t;
}


//------------------------------------------------------------
//.-
//.S	QwtSliderBase::mousePressEvent
//	Mouse press event handler
//
//.s	Parameters
//.p	QMouseEvent *e -- Qt Mouse press event
//
//.s	Syntax
//.f	void QwtSliderBase::mousePressEvent(QMouseEvent *e)
//
//------------------------------------------------------------
void QwtSliderBase::mousePressEvent(QMouseEvent *e) 
{

	const QPoint &p = e->pos();

	d_timerTick = 0;

	getScrollMode(p, d_scrollMode, d_direction);
	if (d_tmrID) killTimer(d_tmrID);

	switch(d_scrollMode)
	{
	case ScrTimer:
		d_mouseOffset = 0;
		d_tmrID = startTimer(d_updTime);
		break;
		
	case ScrMouse:
		d_time.start();
		d_speed = 0;
		d_mouseOffset = getValue(p) - value();
		emit sliderPressed();
		break;
		
	default:
		d_mouseOffset = 0;
		d_direction = 0;
		break;
		
	}
	
}


//------------------------------------------------------------
//.-
//.S	QwtSliderBase::buttonRelease
//	Emit a valueChanged() signal if necessary
//
//.s	Syntax
//.f	void QwtSliderBase::buttonReleased()
//
//------------------------------------------------------------
void QwtSliderBase::buttonReleased()
{
	if ((!d_tracking) && (value() != prevValue()))
	   emit valueChanged(value());
}


//------------------------------------------------------------
//
//.S	QwtSliderBase::mouseReleaseEvent
//	Mouse Release Event handler
//
//.s	Parameters
//.p	QMouseEvent *e -- Qt Mouse Event
//
//.s	Syntax
//.f	void QwtSliderBase::mouseReleaseEvent(QMouseEvent *e)
//
//------------------------------------------------------------
void QwtSliderBase::mouseReleaseEvent(QMouseEvent *e)
{
	int ms = 0;
	switch(d_scrollMode) 
	{

	case ScrMouse:

		setPosition(e->pos());
		d_direction = 0;
		d_mouseOffset = 0;
		if (d_mass > 0.0) 
		{
			ms = d_time.elapsed();
			if ((fabs(d_speed) >  0.0) && (ms < 50))
				d_tmrID = startTimer(d_updTime);
		}
		else
		{
			d_scrollMode = ScrNone;
			buttonReleased();
		}
		emit sliderReleased();
		
		break;

	case ScrDirect:
		
		setPosition(e->pos());
		d_direction = 0;
		d_mouseOffset = 0;
		d_scrollMode = ScrNone;
		buttonReleased();
		break;
		
	case ScrTimer:

		killTimer(d_tmrID);
		d_tmrID =0;
		if (!d_timerTick)
		   fitValue(value() + double(d_direction) * step());
		d_timerTick = 0;
		buttonReleased();
		d_scrollMode = ScrNone;
		break;

	default:
		d_scrollMode = ScrNone;
		buttonReleased();
	}
}


//------------------------------------------------------------
//
//.S	QwtSliderBase::setPosition
//	Move the slider to a specified point, adjust the value
//	and emit signals if necessary
//
//.s	Parameters
//.p	const QPoint &p
//
//.s	Syntax
//.f	void QwtSliderBase::setPosition(const QPoint &p)
//
//------------------------------------------------------------
void QwtSliderBase::setPosition(const QPoint &p) 
{
	fitValue(getValue(p) - d_mouseOffset);
}


//------------------------------------------------------------
//
//.S QwtSliderBase::setTracking
//
//	Enables or disables tracking.
//
//.s Parameters
//.p  bool enable -- enable (true) or disable (false) tracking
//
//.s Description
//
//	If tracking is enabled, the slider emits a
//	valueChanged() signal whenever its value
//	changes (the default behaviour). If tracking
//	is disabled, the value changed() signal will only
//	be emitted if
//.i	-- the user releases the mouse
//	button and the value has changed or
//	-- at the end of automatic scrolling. 
//.P
//	Tracking is anabled by default.
//------------------------------------------------------------
void QwtSliderBase::setTracking(bool enable)
{
	d_tracking = enable;
}

//------------------------------------------------------------
//.-
//.S	QwtSliderBase::mouseMoveEvent
//	Mouse Move Event handler
//
//.s	Parameters
//.p	QMouseEvent *e	-- Qt Mouse Move Event
//
//.s	Syntax
//.f	void QwtSliderBase::mouseMoveEvent(QMouseEvent *e)
//
//------------------------------------------------------------
void QwtSliderBase::mouseMoveEvent(QMouseEvent *e)
{
	double ms = 0.0;
	if (d_scrollMode == ScrMouse )
	{
	   	setPosition(e->pos());
	   	if (d_mass > 0.0) 
		{
			ms = double(d_time.elapsed());
			if (ms < 1.0) ms = 1.0;
			d_speed = (exactValue() - exactPrevValue()) / ms;
			d_time.start();
		}
		if (value() != prevValue())
			emit sliderMoved(value());
	}
	
}

//------------------------------------------------------------
//
//.S	QwtSliderBase::timerEvent
//	Timer event handler
//
//.s	Parameters
//.p	QTimerEvent *e	--	Qt timer event
//
//.s	Syntax
//.f	void QwtSliderBase::timerEvent(QTimerEvent *e)
//
//------------------------------------------------------------
void QwtSliderBase::timerEvent(QTimerEvent *e)
{
	double newval;

	d_timerTick = 1;
	switch (d_scrollMode)
	{
	case ScrMouse:
		if (d_mass > 0.0)
		{
			d_speed *= exp( - double(d_updTime) * 0.001 / d_mass );
			newval = exactValue() + d_speed * double(d_updTime);
			fitValue(newval);
			// stop if d_speed < one step per second
			if (fabs(d_speed) < 0.001 * fabs(step()))
			{
				d_speed = 0;
				killTimer(d_tmrID);
				buttonReleased();
			}

		}
		else
			killTimer(d_tmrID);

	break;
			
	case ScrTimer:
		fitValue(value() +  double(d_direction) * step());
		break;
	default:
		killTimer(d_tmrID);
		break;
	}
}


//------------------------------------------------------------
//
//.S	QwtSliderBase::valueChange
//	Notify change of value
//
//.s	Description
//	This function can be reimplemented by derived classes
//	in order to keep track of changes, i.e. repaint the widget.
//	The default implementation emits a valueChanged() signal
//	if tracking is enabled.
//.s Parameters
//	double x  --    new value
//
//------------------------------------------------------------
void QwtSliderBase::valueChange() 
{
	if (d_tracking)
		emit valueChanged(value());	
}

//------------------------------------------------------------
//
//.S QwtSliderBase::setMass
//	Set the slider's mass for flywheel effect.
//
//.s Parameters
//.p	    double val    --      new mass in kg
//
//
//.s Description
//
//     If the slider's mass is greater then 0, it will continue
//     to move after the mouse button has been released. Its speed
//     decreases with time at a rate depending on the slider's mass.
//     A large mass means that it will continue to move for a
//     long time.
//
//     Limits: If the mass is smaller than 1g, it is set to zero.
//     The maximal mass is limited to 100kg.
//
//     Derived widgets may overload this function to make it public.
//
//------------------------------------------------------------
void QwtSliderBase::setMass(double val)
{
	if (val < 0.001)
	   d_mass = 0.0;
	else if (val > 100.0)
	   d_mass = 100.0;
	else
	   d_mass = val;
}


//------------------------------------------------------------
//
//.S	QwtSliderBase::moveTo
//	Move the slider to a specified value
//
//.s	Parameters
//.p	double val	--	new value
//
//.s	Description
//	This function can be used to move the slider to a value
//	which is not an integer multiple of the step size.
//
//.s	See also
//	\R{QwtSliderBase::adjustTo}
//------------------------------------------------------------
void QwtSliderBase::moveTo(double val)
{
	stopMoving();
	setValue(val);
}


//------------------------------------------------------------
//
//.S	QSlider::adjustTo
//	Set the slider's value to the nearest integer multiple
//      of the step size.
//
//.s	See also:
//	\R{QwtSliderBase::moveTo}
//------------------------------------------------------------
void QwtSliderBase::adjustTo(double val)
{
	fitValue(val);
}






//------------------------------------------------------------
//
//.S	QwtSliderBase::getValue
//	Determine the value corresponding to a specified poind
//
//.s	Parameters
//.p	const QPoint &p	-- point 
//
//.s	Description
//	This is an abstract virtual function whis is called when
//	the user presses or releases a mouse button or moves the
//	mouse. It has to be implemented by the derived class.
//
//.s	Syntax
//.f	void QwtSliderBase::getValue(const QPoint &p)
//
//------------------------------------------------------------

//------------------------------------------------------------
//
//.S	QwtSliderBase::getScrollMode
//	Determine what to do when the user presses a mouse button.
//
//.s	Input Parameters
//.p	const QPoint &p	--	point where the mouse was pressed
//
//.s	Output parameters
//	int &scrollMode --	The scrolling mode
//	int &direction  --	direction: 1, 0, or -1.
//
//.s	Description
//	This function is abstract and has to be implemented by derived classes.
//	It is called on a mousePress event. The derived class can determine
//	what should happen next in dependence of the posision where the mouse
//	was pressed by returning scrolling mode and direction. QwtSliderBase
//	knows the following modes:
//.t
//	QwtSliderBase::ScrNone -- Scrolling switched off. Don't change the value.
//	QwtSliderBase::ScrMouse -- Change the value while the user keeps the
//					button pressed and moves the mouse.
//	QwtSliderBase::ScrTimer -- Automatic scrolling. Increment the value
//					in the specified direction as long as
//					the user keeps the button pressed.
//
//.s	Syntax
//.f	void QwtSliderBase::getScrollMode(const QPoint &p, int &scrollMode, int &direction)
//
//------------------------------------------------------------













