#include <qwt_scldraw.h>
#include <qpainter.h>
#include <qfontmet.h>
#include <qrect.h>
#include <qwt_math.h>


int const QwtScaleDraw::minLen = 10;

const double step_eps = 1.0e-6;
const double WorstCase = -8.8888888888888888888888e-88;


//------------------------------------------------------------
//
//.S	QwtScaleDraw::setScale
//	Adjust the range of the scale
//
//.s	Parameters
//.p	double x1	--	value at the left/low endpoint of the scale
//	double x2	--	value at the right/high enpoint of the scale
//	double step	--	step size (default : 0.0)
//	int logscale	--	logarithmic scale (default : 0)
//
//.s	Return Type
//	void
//
//.s	Description
//	If step == 0.0, the step width is calculated automatically
//	dependent on the maximimal number of scale ticks.
//	
//.s	See also:
//	setMaxTicks()
//.s	Syntax
//.f	void QwtScaleDraw::setScale(double x1, double x2, double step, int logscale)
//
//------------------------------------------------------------
void QwtScaleDraw::setScale(double x1, double x2, int maxMajIntv,
			    int maxMinIntv, double step, int logscale)
{
    d_scldiv.rebuild(x1,x2,maxMajIntv, maxMinIntv,logscale,step);
    setDblRange(d_scldiv.lBound(),d_scldiv.hBound(),d_scldiv.logScale());
}


//------------------------------------------------------------
//
//.S	QwtScaleDraw::setScale
//
//
//.s	Parameters
//.p	const QwtAutoScale &as
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	void QwtScaleDraw::setScale(QwtAutoScale &as)
//
//------------------------------------------------------------
void QwtScaleDraw::setScale(const QwtScaleDiv &s)
{
    d_scldiv = s;
    setDblRange(d_scldiv.lBound(),d_scldiv.hBound(),d_scldiv.logScale());
}

//------------------------------------------------------------
//
//.S QwtScaleDraw::QwtScaleDraw
//
// Constructor 
//
//------------------------------------------------------------
QwtScaleDraw::QwtScaleDraw()
{
    d_hpad = 6;
    d_vpad = 3;
    d_majLen = 8;
    d_minLen = 4;
    d_medLen = 6;

    d_minAngle = -135 * 16;
    d_maxAngle = 135 * 16;
    d_fmt = 'g';
    d_prec = 4;
        
    // initialize scale and geometry
    setGeometry(0,0,100,Bottom);
    setScale(0,100,0,0,10);
   d_font=QFont("Helvetica",10);
}



//------------------------------------------------------------
//
//.S QwtScaleDraw::draw
//
// 
//
//.s Parameters
//  
//
//.s Return Type void
//
//.s Return Value
//
//.s Description
//
//------------------------------------------------------------

void QwtScaleDraw::setFont(const QFont &f)
{
   d_font=f;
}

void QwtScaleDraw::draw(QPainter *p) const
{

    double val,hval,majTick;
    
    int i,k,kmax;
   
   p->setFont(d_font);
//   printf("drawing\n");
   
   if(d_scldiv.majCnt()==0 && d_scldiv.minCnt()==0)return;
   
   for (i=0; i< d_scldiv.majCnt(); i++)
    {
       val = d_scldiv.majMark(i);
       drawTick(p, val, d_majLen);
       drawLabel(p, val);
    }

    if (d_scldiv.logScale())
    {
	for (i=0; i< d_scldiv.minCnt(); i++)
	{
	    drawTick(p,d_scldiv.minMark(i),d_minLen);
	}
    }
    else
    {
	k = 0;
	kmax = d_scldiv.majCnt() - 1;
	if (kmax > 0) 
	{
	   majTick = d_scldiv.majMark(0);
	   hval = majTick - 0.5 * d_scldiv.majStep();

	   for (i=0; i< d_scldiv.minCnt(); i++)
	   {
	       val = d_scldiv.minMark(i);
	       if  (val > majTick)
	       {
		   if (k < kmax)
		   {
		       k++;
		       majTick = d_scldiv.majMark(k);
		   }
		   else
		   {
		       majTick += d_scldiv.majMark(kmax) + d_scldiv.majStep();
		   }
		   hval = majTick - 0.5 * d_scldiv.majStep();
		   
	       }
	       if (fabs(val-hval) < step_eps * d_scldiv.majStep())
		  drawTick(p, val, d_medLen);
	       else
		  drawTick(p, val, d_minLen);
	   }
	}
    }

    //
    // draw backbone
    //
    //if (d_baseEnabled)
       drawBackbone(p);

}


//------------------------------------------------------------
//
//.S QwtScaleDraw::drawTick
//
// 
//
//.s Parameters
//  double val, const QPainter *p
//
//.s Return Type void
//
//.s Return Value
//
//.s Description
//
//------------------------------------------------------------
void QwtScaleDraw::drawTick(QPainter *p, double val, int len) const
{
    
  int tval = transform(val);
  float arc;
  int bw2;

  bw2 = p->pen().width() / 2;
    
  switch(d_orient)
    {
    case Right:

	  p->drawLine(d_xorg, tval, d_xorg + len, tval);
	  break;

    case Bottom: 

	  p->drawLine(tval, d_yorg, tval, d_yorg + len);
	  break;

    case Left:

	  p->drawLine(d_xorg, tval, d_xorg - len, tval);
	  break;

    case Round:

	  arc = float(tval) / 16.0 - 90;
	  p->translate(d_xorg, d_yorg);
	  p->rotate(arc);
	  p->drawLine(d_len + bw2,0,d_len + len + bw2,0);
	  p->resetXForm();
	  break;
	
    case Top:
    default:
	
	  p->drawLine(tval, d_yorg, tval, d_yorg - len);
	  break;
	

    }
    
}




//------------------------------------------------------------
//
//.S QwtScaleDraw::drawLabel
//
// 
//
//.s Parameters
//  QPainter *p
//  double val
//
//.s Return Type void
//
//.s Return Value
//
//.s Description
//
//------------------------------------------------------------
void QwtScaleDraw::drawLabel(QPainter *p, double val) const
{

    static QString label;
    static double pi_4 = M_PI * 0.25;
    static double pi_75 = M_PI * 0.75;
    
    double arc;
    int xpos, ypos;
    int tval;

    QFontMetrics fm = p->fontMetrics();
    
    tval = transform(val);

    // correct rounding errors if val = 0
    if ((!d_scldiv.logScale()) && (fabs(val) < fabs(step_eps * d_scldiv.majStep())))
       val = 0.0;
    
    label.setNum(val, d_fmt, d_prec);

    switch(d_orient)
    {
    case Right:
	p->drawText(d_xorg + d_majLen + d_hpad,
		    tval + (fm.ascent()-1) / 2,
		    label);
	break;
    case Left:
       p->drawText(0,tval-(fm.ascent())/2,d_xorg - d_majLen - d_hpad,
		   tval+fm.ascent()/2+fm.descent(),AlignRight|AlignTop,
		   label);
       
       //	p->drawText(d_xorg - d_majLen - d_hpad - fm.width(label),
//		    tval + (fm.ascent() -1) / 2,
//		    label);
	break;
    case Bottom:
	p->drawText(tval - (fm.width(label)-1) / 2, d_yorg + d_majLen + d_vpad + fm.ascent(), label);
	break;
    case Round:
	arc = double(tval) / 16.0 * M_PI / 180.0;

	// Map arc into the interval -pi <= arc <= pi
	if ((arc < -M_PI) || (arc > M_PI))
	   arc -= floor((arc + M_PI) / M_PI * 0.5) * 2.0 * M_PI;
	
	xpos = d_xorg + int(rint(double(d_len + d_majLen + d_vpad) * sin(arc)));
	ypos = d_yorg - int(rint(double(d_len + d_majLen + d_vpad) * cos(arc)));
	
	if (arc < -pi_75)
	{
	    p->drawText(xpos - int(rint(double(fm.width(label))
					* (1.0 + (arc + pi_75) * M_2_PI) )),
			ypos + fm.ascent() - 1,
			label);
	}
	else if (arc < -M_PI_4)
	{
	    p->drawText(xpos - fm.width(label),
			ypos - int(rint(double(fm.ascent() - 1)
					* (arc + M_PI_4) * M_2_PI)),
			label);
	}
	else if (arc < pi_4)
	{
	    p->drawText(xpos + int(rint(double(fm.width(label))
					* ( arc - M_PI_4 ) * M_2_PI )),
			ypos,
			label);
	}
	else if (arc < pi_75)
	{
	    p->drawText(xpos,
			ypos + int(rint(double(fm.ascent() - 1)
					* (arc - M_PI_4) * M_2_PI)),
			label);
	}
	else
	{
	    p->drawText(xpos - int(rint(double(fm.width(label))
					* ( arc - pi_75) * M_2_PI )),
			ypos + fm.ascent() - 1,
			label);
	}
	break;
    case Top:
    default:
	p->drawText(tval - (fm.width(label)-1) / 2, d_yorg - d_majLen - d_vpad, label);
	break;
    }
    


}

//------------------------------------------------------------
//
//.S QwtScaleDraw::drawBackbone
//
// 
//
//.s Parameters
//  QPainter *p
//
//.s Return Type void
//
//.s Return Value
//
//.s Description
//
//------------------------------------------------------------
void QwtScaleDraw::drawBackbone(QPainter *p) const
{
  int bw2;
  int a1, a2;
  bw2 = p->pen().width() / 2;
    
    
  switch(d_orient)
    {
    case Left:
	  p->drawLine(d_xorg - bw2, d_yorg, d_xorg - bw2, d_yorg + d_len - 1);
	  break;
    case Right:
	  p->drawLine(d_xorg + bw2, d_yorg, d_xorg + bw2, d_yorg + d_len - 1);
	  break;
    case Round:

	  a1 = qwtMin(i1(), i2()) - 90 * 16; 
	  a2 = qwtMax(i1(), i2()) - 90 * 16; 
	
	  p->drawArc(d_xorg - d_len - bw2, d_yorg - d_len - bw2,
				 2*(d_len + bw2) + 1,
				 2*(d_len + bw2) + 1,
				 -a2, a2 - a1);	// counterclockwise

	  break;
	
    case Top:
	  p->drawLine(d_xorg, d_yorg - bw2, d_xorg + d_len - 1, d_yorg-bw2);
	  break;
    case Bottom:
	  p->drawLine(d_xorg, d_yorg+bw2, d_xorg + d_len - 1, d_yorg+bw2);
	  break;
    default:
	  p->drawLine(d_xorg, d_yorg, d_xorg + d_len - 1, d_yorg);
	  break;
    }
    
}


//------------------------------------------------------------
//
//.S QwtScaleDraw::setGeometry
//
// 
//
//.s Parameters
//  int xorigin, int yorigin, int length, Orientation o
//
//.s Return Type void
//
//.s Return Value
//
//.s Description
//
//------------------------------------------------------------
void QwtScaleDraw::setGeometry(int xorigin, int yorigin, int length, Orientation o)
{

    d_xorg = xorigin;
    d_yorg = yorigin;

    if (length > minLen)
       d_len = length;
    else
       d_len = minLen;

    d_orient = o;
    
    switch(d_orient)
    {
    case Left:
    case Right:
	setIntRange(d_yorg + d_len - 1, d_yorg);
	break;
    case Round:
	setIntRange(d_minAngle, d_maxAngle);
	break;
    case Top:
    case Bottom:
    default:
	setIntRange(d_xorg, d_xorg + d_len - 1);
	break;
    }
}



//------------------------------------------------------------
//
//.S	maxWidth
//
//
//.s	Parameters
//.p	QPainter *p
//	bool worst
//
//.s	Return Type
//	int
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	int maxWidth(QPainter *p)
//
//------------------------------------------------------------
int QwtScaleDraw::maxWidth(QPainter *p, bool worst) const
{
    int rv = 0;
    int bw = p->pen().width();

    QString s;
    
    
    QFontMetrics fm = p->fontMetrics();

   rv = maxLabelWidth(p,worst);
   switch (d_orient)
    {
    case Left:
    case Right:
	rv += bw + d_hpad + bw + d_majLen;
	break;
    case Round:
	rv += bw + d_vpad + bw + d_majLen;
	break;
    case Top:
    case Bottom:
    default:
	rv += d_len;
    }
    return rv;
    
}

//------------------------------------------------------------
//
//.S	QwtScaleDraw::maxHeight
//
//
//.s	Parameters
//.p	QPainter *p
//
//.s	Return Type
//	int
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	int QwtScaleDraw::maxHeight(QPainter *p)
//
//------------------------------------------------------------
int QwtScaleDraw::maxHeight(QPainter *p) const 
{

    int rv = 0;
    int bw = p->pen().width();
    
    p->save();
    QFontMetrics fm = p->fontMetrics();

    switch (d_orient)
    {
    case Top:
    case Bottom:
    case Round:
	rv = bw + d_vpad + bw + d_majLen + fm.ascent();
	break;
    case Left:
    case Right:
    default:
	rv = d_len + (fm.ascent() + fm.descent() + 1) / 2;
    }
    
    return rv;

}

//------------------------------------------------------------
//
//.S QwtScaleDraw:maxBoundingRect
//
// 
//
//.s Parameters
//  QPainter *p
//
//.s Return Type void
//
//.s Return Value
//
//.s Description
//
//------------------------------------------------------------
QRect QwtScaleDraw::maxBoundingRect(QPainter *p) const
{
    int wl,h,wmax,hmax;
    QRect r;

    QFontMetrics fm = p->fontMetrics();

    wl = maxLabelWidth(p, TRUE);
    h = fm.height();

    switch(d_orient)
    {
    case Left:

	r = QRect( d_xorg - d_hpad - d_majLen - wl,
		  d_yorg - fm.ascent(),
		  d_majLen + d_hpad + wl,
		  d_len + fm.height());
	break;
	
    case Right:

	r = QRect( d_xorg,
		  d_yorg - fm.ascent(),
		  d_majLen + d_hpad + wl,
		  d_len + fm.height());
	break;
	
    case Top:

	r = QRect ( d_xorg - wl / 2,
		   d_yorg - d_majLen - fm.ascent(),
		   d_len + wl,
		   d_majLen + d_vpad + fm.ascent());
	break;
	
    case Bottom:

	r = QRect ( d_xorg - wl / 2,
		   d_yorg,
		   d_len + wl,
		   d_majLen + d_vpad + fm.height());
	break;

    case Round:

	wmax = d_len + d_majLen + d_hpad + wl;
	hmax = d_len + d_majLen + d_hpad + fm.ascent();

	r = QRect ( d_xorg - wmax,
		   d_yorg - hmax,
		   2*wmax,
		   2*hmax);
	break;
    }

    return r;
	
}

//------------------------------------------------------------
//
//.s	QwtScaleDraw::setAngleRange
//	Adjust the angle range for round scales.
//
//.s	Parameters
//.p	double angle1, double angle2 
//			boundaries of the angle interval in degrees.
//
//.s	Return Type
//	void
//
//.s	Description
//	The angle range will be set to [ min(angle1,angle2), max(angle1, angle2) ].
//	The settings have no effect if the scale orientation is not set to
//	QwtScaleDraw::Round. The default setting is [ -135, 135 ].
//	An angle of 0 degrees corresponds to the 12 o'clock position,
//	and positive angles count in a clockwise direction.
//
//.s	Syntax
//.f	void QwtScaleDraw::setAngleRange(double angle1, double angle2)
//
//------------------------------------------------------------
void QwtScaleDraw::setAngleRange(double angle1, double angle2)
{
    int amin, amax;
    
    amin = int(rint(qwtMin(angle1, angle2) * 16.0));
    amax = int(rint(qwtMax(angle1, angle2) * 16.0));

    if (amin == amax)
    {
	amin -= 1;
	amax += 1;
    }

    d_minAngle = amin;
    d_maxAngle = amax;
    setIntRange(d_minAngle, d_maxAngle);

}


//------------------------------------------------------------
//
//.S	QwtScaleDraw::setLabelFormat
//
//
//.s	Parameters
//.p	char f, int prec
//
//.s	Return Type
//		void
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	void QwtScaleDraw::setLabelFormat(char f, int prec)
//
//------------------------------------------------------------
void QwtScaleDraw::setLabelFormat(char f, int prec)
{
    d_fmt = f;
    d_prec = prec;
}

//------------------------------------------------------------
//
//.S	QwtScaleDraw::maxLabelWidth
//
//
//.s	Parameters
//.p	QPainter *p, int worst
//
//.s	Return Type
//		int
//
//.s	Return Value
//
//.s	Description
//
//.s	Syntax
//.f	int QwtScaleDraw::maxLabelWidth(QPainter *p, int worst)
//
//------------------------------------------------------------
int QwtScaleDraw::maxLabelWidth(QPainter *p, int worst) const
{

    int i,rv = 0;
    double val;
    QString s;
    
    
    QFontMetrics fm = p->fontMetrics();

    if (worst) // worst case
    {
	s.setNum(WorstCase, d_fmt, d_prec);
	rv = fm.width(s);
    }
    else // actual width
    {
	for (i=0;i<d_scldiv.majCnt(); i++)
	{
	    val = d_scldiv.majMark(i);
	    // correct rounding errors if val = 0
	    if ((!d_scldiv.logScale()) && (fabs(val) < step_eps * fabs(d_scldiv.majStep())))
	       val = 0.0;
	    s.setNum(val, d_fmt, d_prec);
	    rv = qwtMax(rv,fm.width(s));
	}
    }


    return rv;
    
}













