#include <math.h>
#include <stdlib.h>
#include "qwt_scldiv.h"
#include "qwt_math.h"

//------------------------------------------------------------
//.H QwtScaleDiv|3|27/04/97|Qwt Widget Library|Qwt Programmer's manual
//.S NAME
//	QwtScaleDiv - A class for building  scale divisions
//
//.S SYNOPSIS
//	#include <qwt_scldiv.h>
//
//.S PUBLIC MEMBERS 
//.t
//	\R{QwtScaleDiv::QwtScaleDiv}() -- constructor
//	\R{QwtScaleDiv::~QwtScaleDiv}() -- destructor
//	\R{QwtScaleDiv::QwtScaleDiv}(const QwtScaleDiv &s) -- copy constructor
//	double lbound()	-- lower boundary
//	double hBound()   -- upper boundary
//	int minCnt()	-- number of minor marks
//	int majCnt()	--  number of major marks
//	bool logScale()	-- logarithmic or linear
//	double majMark(int i) -- major mark with index i
//	double minMark(int i) -- minor mark with index i
//	double majStep() -- major step width
//	double \R{QwtScaleDiv::rebuild}(double lBound, double hBound, int maxMaj, int maxMin, bool log, double step = 0.0, bool ascend = true) -- build a new scale division
//	void \R{QwtScaleDiv::reset}() -- reset the scale
//	QwtScaleDiv& \R{QwtScaleDiv::operator=} (const QwtScaleDiv &s) -- assignment operator
//	bool \R{QwtScaleDiv::operator==}(const QwtScaleDiv &s) -- equality
//	bool \R{QwtScaleDiv::operator!=}(const QwtScaleDiv &s) -- inequality
//
//
//.S DESCRIPTION
//	The QwtScaleDiv class is used both as a container for a scale
//	division and as a means for generating such a scale division. It can build
//	linear and logarithmic scale divisions for specified
//	intervals and it uses an adjustable algorithm to
//	generate the major and minor step widths automatically.
//	A scale division has a minimum value, a maximum value,
//	a vector of major marks, and a vector of minor marks.
//	
//.S NOTE
//	QwtScaleDiv uses implicit sharing for the mark vectors.
//
//.S EXAMPLE
//	Build a logarithmic scale division from 0.01 to 1000
//	and print out the major and minor marks. 
//.c
//	#include <qwt_scldiv.h>
//	#include <iostream.h>
//
//	main()
//	{
//	    int i,k;
//	    QwtScaleDiv sd;
//
//	    sd.rebuild(0.01, 100, 10, 10, true, 0.0);
//
//	    k=0;
//	    for (i=0;i<sd.majCnt();i++)
//	    {
//		while(k < sd.minCnt())
//		{
//		    if(sd.minMark(k) < sd.majMark(i))
//		    {
//			cout << " - " << sd.minMark(i) << "\n";
//			k++;
//		    }
//		    else
//		       break;
//		}
//		cout << "-- " << sd.majMark(i) << "\n";
//	    }
//	    while(k < sd.minCnt())
//	    {
//		cout << " - " << sd.minMark(i) << "\n";
//		k++;
//	    }
//	}
//
//------------------------------------------------------------

//------------------------------------------------------
//.C MEMBER FUNCTION DESCRIPTION
//------------------------------------------------------

const double step_eps = 1.0e-3;
const double border_eps = 1.0e-10;

bool qwtLimRange(double &val, double v1, double v2, double eps_rel = 0.0,
		double eps_abs = 0.0)
{
    
    bool rv = true;
    double vmin = qwtMin(v1, v2);
    double vmax = qwtMax(v1, v2);
    double delta_min = qwtMax(fabs(eps_rel * vmin), fabs(eps_abs));
    double delta_max = qwtMax(fabs(eps_rel * vmax), fabs(eps_abs));
    
    if (val < vmin) 
    {
	if (val < vmin - delta_min) rv = false;
	val = vmin;
    }
    else if (val > vmax)
    {
	if (val > vmax + delta_max) rv = false;
	val = vmax;
    }
    return rv;

}



//------------------------------------------------------------
//
//.S	QwtScaleDiv::QwtScaleDiv
//	Construct a QwtScaleDiv instance.
//
//.s	Syntax
//.f	 QwtScaleDiv::QwtScaleDiv()
//
//------------------------------------------------------------
QwtScaleDiv::QwtScaleDiv()
{
    d_lBound = 0.0;
    d_hBound = 0.0;
    d_majStep = 0.0;
    d_log = false;
}



//------------------------------------------------------------
//
//.S	QwtScaleDiv::~QwtScaleDiv
//	Destroy a QwtScaleDiv instance.
//
//.s	Syntax
//.f	 QwtScaleDiv::~QwtScaleDiv()
//
//------------------------------------------------------------
QwtScaleDiv::~QwtScaleDiv()
{
}

//------------------------------------------------------------
//
//.S	QwtScaleDiv::QwtScaleDiv
//      Copy Constructor
//
//.s	Parameters
//.p	const QwtScaleDiv &s
//
//.s	Syntax
//.f	 QwtScaleDiv::QwtScaleDiv(const QwtScaleDiv &s)
//
//------------------------------------------------------------
QwtScaleDiv::QwtScaleDiv(const QwtScaleDiv &s)
{
    copy(s);
}


//------------------------------------------------------------
//
//.S	QwtScaleDiv::operator=
//	Assignment operator
//
//.s	Parameters
//.p	const QwtScaleDiv &s
//
//.s	Return Type
//		QwtScaleDiv &
//
//.s	Return Value
//		
//.s	Description
//
//.s	Syntax
//.f	QwtScaleDiv & QwtScaleDiv::operator=(const QwtScaleDiv &s)
//
//------------------------------------------------------------
QwtScaleDiv& QwtScaleDiv::operator=(const QwtScaleDiv &s)
{
    copy(s);
    return *this;
}

//------------------------------------------------------------
//.-
//.S	QwtScaleDiv::copy
//	Copy member data from another QwtScaleDiv instance. 
//
//.s	Parameters
//.p	const QwtScaleDiv &s
//
//.s	Syntax
//.f	void QwtScaleDiv::copy(const QwtScaleDiv &s)
//
//------------------------------------------------------------
void QwtScaleDiv::copy(const QwtScaleDiv &s)
{
    d_lBound = s.d_lBound;
    d_hBound = s.d_hBound;
    d_log = s.d_log;
    d_majStep = s.d_majStep;
    d_minMarks = s.d_minMarks;
    d_majMarks = s.d_majMarks;
}




//------------------------------------------------------------
//
//.S	QwtScaleDiv::rebuild
//	Build a scale widh major and minor divisions
//
//
//.s Parameters
//.p
//	double x1	--	first boundary value
//	double x2	--	second boundary value
//	int maxMajSteps	--	max. number of major step intervals
//	int maxMinSteps	--	max. number of minor step intervals
//      bool log	--	logarithmic division (true/false)
//	double step 	--	fixed major step width. Defaults to 0.0.
//	bool ascend	--	if true, sort in ascending order from min(x1, x2)
//				to max(x1, x2). If false, sort in the direction
//				from x1 to x2.	Defaults to true.
//
//.s Return Type bool
//
//.s Return Value
//	True if the arrays have been allocated successfully.
//
//.s Description
//	If no fixed step width is specified or if it is set to 0, the
//	major step width will be calculated automatically according to the
//	the value of maxMajSteps. The maxMajSteps parameter has no effect
//	if a fixed step size is specified. The minor step width is always
//	calculated automatically.
//	If the step width is to be calculated automatically, the algorithm
//	tries to find reasonable values fitting into the scheme {1,2,5}*10^n
//	with an integer number n for linear scales.
//	For logarithmic scales, there are three different cases:
//.i
//	-- If the major step width is one decade, the minor marks
//		will fit into one of the schemes {1,2,...9}, {2,4,6,8}, {2,5} or {5},
//		depending on the maxMinSteps parameter.
//	-- If the major step size spans
//		more than one decade, the minor step size will be {1,2,5}*10^n decades
//		with a natural number n.
//	-- If the whole range is less than one decade, a linear scale
//		division will be built
//	
//.s Note
//	For logarithmic scales, the step width is measured in decades.
//
//------------------------------------------------------------
bool QwtScaleDiv::rebuild(double x1, double x2, int maxMajSteps, int maxMinSteps,
			 bool log, double step, double ref)
{

   int i, majSize,rv;
   double lxmax, lxmin, xmax, xmin, xstep;
   bool ascend=bool(x2>=x1);
   
  d_lBound = qwtMin(x1, x2);
  d_hBound = qwtMax(x1, x2);
  d_log = log;

//   printf("scldiv x1:%f,x2:%f,step:%f,ref:%f,ascend:%i\n",x1,x2,step,ref,ascend);
   
  if (d_log)
      rv = buildLogDiv(maxMajSteps,maxMinSteps,step, ref);
  else
      rv = buildLinDiv(maxMajSteps, maxMinSteps, step, ref);

  if ((!ascend) && (x2 < x1))
  {
      d_lBound = x1;
      d_hBound = x2;
      qwtTwistArray(d_majMarks.data(), d_majMarks.size());
      qwtTwistArray(d_minMarks.data(), d_minMarks.size());
  }

  return rv;
  
}



//------------------------------------------------------------
//.-
//.S	QwtScaleDiv::buildLinDiv
//	Build a linear scale division in ascending order
//
//.s	Parameters
//.p	int maxSteps -- max. number of step intervals
//	double step -- fixed step width
//
//.s	Return Type
//		bool
//
//.s	Return Value
//		true if array has been successfully resized
//
//.s	Description
//		If the 'step' parameter is set to 0.0, this function
//		cal[culates the step width automatically according to
//		the value of 'maxSteps'. MaxSteps must be greater than or
//		equal to 2. It will be guessed if an invalid value is specified.
//		The maximum possible number of steps is	limited to 10000.
//		The maxSteps parameter has no effect if a fixed step width is
//		specified.
//
//.s	Note
//	This function uses the data members d_lBound and d_hBound and assumes
//	that d_hBound > d_lBound.
//
//.s	Syntax
//.f	bool QwtScaleDiv::buildLinDiv(int majSteps, int minSteps, double step)
//
//------------------------------------------------------------
bool QwtScaleDiv::buildLinDiv(int maxMajSteps, int maxMinSteps, double step, double ref)
{
    
    int nMaj, nMin, minSize, i0,i,k;
    double val, mval;
    double firstTick, lastTick;
    double minStep;
    QArray<double> buffer;
    bool rv = true;


   
    // parameter range check
    maxMajSteps = qwtMax(1, maxMajSteps);
    maxMinSteps = qwtMax(0, maxMinSteps);
    step = fabs(step);

   //printf("scldiv step: %f, %i\n",step, maxMajSteps);
   
    // detach arrays
    d_majMarks.duplicate(0,0);
    d_minMarks.duplicate(0,0);

    //
    // Set up major divisions
    //
    if (step == 0.0)
       d_majStep = qwtCeil125(fabs(d_hBound - d_lBound) * 0.999999
			      / double(maxMajSteps));
    else
       d_majStep = step;

    firstTick = ref+ceil( (d_lBound-ref - step_eps * d_majStep) / d_majStep) * d_majStep;
    lastTick = ref+floor( (d_hBound-ref + step_eps * d_majStep) / d_majStep) * d_majStep;

    nMaj = qwtMin(10000, int(rint((lastTick - firstTick) / d_majStep)) + 1);

    if ((rv = d_majMarks.resize(nMaj)))
       qwtLinSpace(d_majMarks.data(), d_majMarks.size(), firstTick, lastTick);
    else
       return false;

    //
    // Set up minor divisions
    //
    if (maxMinSteps < 1) // no minor divs
       return true;

    minStep = qwtCeil125( d_majStep  /  double(maxMinSteps) );
    
    nMin = abs(int(rint(d_majStep / minStep))) - 1; // # minor steps per interval

    // Do the minor steps fit into the interval?
    if ( fabs(double(nMin +  1) * minStep - d_majStep) >  step_eps * d_majStep)
    {
	nMin = 1;
	minStep = d_majStep * 0.5;
    }
    
    // Are there minor ticks below the first major tick?
    if (d_majMarks[0] > d_lBound )
       i0 = -1;	
    else
       i0 = 0;

    // resize buffer to the maximum possible number of minor ticks
    rv = buffer.resize(nMin * (nMaj + 1));
       
    // calculate minor ticks
    if (rv)
    {
	minSize = 0;
	for (i = i0; i < (int)d_majMarks.size(); i++)
	{
	    if (i >= 0)
	       val = d_majMarks[i];
	    else
	       val = d_majMarks[0] - d_majStep;

	    for (k=0; k< nMin; k++)
	    {
		mval = (val += minStep);
		if (qwtLimRange(mval, d_lBound, d_hBound, border_eps))
		{
		    buffer[minSize] = mval;
		    minSize++;
		}
	    }
	}
	d_minMarks.duplicate(buffer.data(), minSize);
    }

    return rv;
    
}



//------------------------------------------------------------
//.-
//.S	QwtScaleDiv::buildLogDiv
//	Build a logarithmic scale division
//
//.s	Parameters
//.p	int maxMajSteps, int maxMinSteps, int majStep
//
//.s	Return Type
//		bool
//
//.s	Return Value
//		True if memory has been successfully allocated
//
//.s	Note
//	This function uses the data members d_lBound and d_hBound and assumes
//	that d_hBound > d_lBound.
//
//
//.s	Syntax
//.f	bool QwtScaleDiv::buildLogDiv(int maxMajSteps, int maxMinSteps, int majStep)
//
//------------------------------------------------------------
bool QwtScaleDiv::buildLogDiv(int maxMajSteps, int maxMinSteps, double majStep, double ref)
{

    double firstTick, lastTick, stepWidth;
    double lFirst, lLast;
    double val, sval, minStep, minFactor;
    int nMaj, nMin, minSize, i, k, k0, kstep, kmax, i0;
    int rv = true;
    double width;
    
    QArray<double> buffer;
    

    // Parameter range check
    maxMajSteps = qwtMax(1, abs(maxMajSteps));
    maxMinSteps = qwtMax(0, abs(maxMinSteps));
    majStep = fabs(majStep);

    // boundary check
    qwtLimRange(d_hBound, LOG_MIN, LOG_MAX);
    qwtLimRange(d_lBound, LOG_MIN, LOG_MAX);
    
    // detach arrays
    d_majMarks.duplicate(0,0);
    d_minMarks.duplicate(0,0);

    // scale width in decades
    width = log10(d_hBound) - log10(d_lBound);

    // scale width is less than one decade -> build linear scale
    if (width < 1.0) 
    {
	rv = buildLinDiv(maxMajSteps, maxMinSteps, 0.0, ref);
	// convert step width to decades
	if (d_majStep > 0)
	   d_majStep = log10(d_majStep);

	return rv;
    }

    //
    //  Set up major scale divisions
    //
    if (majStep == 0.0)
       d_majStep = qwtCeil125( width * 0.999999 / double(maxMajSteps));
    else
       d_majStep = majStep;

    // major step must be >= 1 decade
    d_majStep = qwtMax(d_majStep, 1.0);
    

    lFirst = ceil((log10(d_lBound) - step_eps * d_majStep) / d_majStep) * d_majStep;
    lLast = floor((log10(d_hBound) + step_eps * d_majStep) / d_majStep) * d_majStep;

    firstTick = pow(10.0, lFirst);
    lastTick = pow(10.0, lLast);
    
    nMaj = qwtMin(10000, int(rint(fabs(lLast - lFirst) / d_majStep)) + 1);

    if (d_majMarks.resize(nMaj))
       qwtLogSpace(d_majMarks.data(), d_majMarks.size(), firstTick, lastTick);
    else
       return false;


    //
    // Set up minor scale divisions
    //

    if ((d_majMarks.size() < 1) || (maxMinSteps < 1)) return true; // no minor marks

    if (d_majStep < 1.1)			// major step width is one decade
    {
	if (maxMinSteps >= 8)
	{
	    k0 = 2;
	    kmax = 9;
	    kstep = 1;
	    minSize = (d_majMarks.size() + 1) * 8;
	}
	else if (maxMinSteps >= 4)
	{
	    k0 = 2;
	    kmax = 8;
	    kstep = 2;
	    minSize = (d_majMarks.size() + 1) * 4;
	}
	else if (maxMinSteps >= 2)
	{
	    k0 = 2;
	    kmax = 5;
	    kstep = 3;
	    minSize = (d_majMarks.size() + 1) * 2;
	}
	else
	{
	    k0 = 5;
	    kmax = 5;
	    kstep = 1;
	    minSize = (d_majMarks.size() + 1);
	}
	
	// resize buffer to the max. possible number of minor marks
	buffer.resize(minSize);

	// Are there minor ticks below the first major tick?
	if ( d_lBound < firstTick )
	    i0 = -1;
	else
	   i0 = 0;
	
	minSize = 0;
	for (i = i0; i< (int)d_majMarks.size(); i++)
	{
	    if (i >= 0)
	       val = d_majMarks[i];
	    else
	       val = d_majMarks[0] / pow(10.0, d_majStep);
	    
	    for (k=k0; k<= kmax; k+=kstep)
	    {
		sval = val * double(k);
		if (qwtLimRange(sval, d_lBound, d_hBound, border_eps))
		{
		    buffer[minSize] = sval;
		    minSize++;
		}
	    }
	}

	// copy values into the minMarks array
	d_minMarks.duplicate(buffer.data(), minSize);

    }
    else				// major step > one decade
    {
	
	// substep width in decades, at least one decade
	minStep = qwtCeil125( (d_majStep - step_eps * (d_majStep / double(maxMinSteps)))
			 /  double(maxMinSteps) );
	minStep = qwtMax(1.0, minStep);

	// # subticks per interval
	nMin = int(rint(d_majStep / minStep)) - 1;

	// Do the minor steps fit into the interval?
	if ( fabs( double(nMin + 1) * minStep - d_majStep)  >  step_eps * d_majStep)
	    nMin = 0;

	if (nMin < 1) return true;		// no subticks

	// resize buffer to max. possible number of subticks
	buffer.resize((d_majMarks.size() + 1) * nMin );
	
	// substep factor = 10^substeps
	minFactor = qwtMax(pow(10,minStep), 10.0);

	// Are there minor ticks below the first major tick?
	if ( d_lBound < firstTick )
	    i0 = -1;
	else
	   i0 = 0;
	
	minSize = 0;
	for (i = i0; i< (int)d_majMarks.size(); i++)
	{
	    if (i >= 0)
	       val = d_majMarks[i];
	    else
	       val = firstTick / pow(10.0, d_majStep);
	    
	    for (k=0; k< nMin; k++)
	    {
		sval = (val *= minFactor);
		if (qwtLimRange(sval, d_lBound, d_hBound, border_eps))
		{
		    buffer[minSize] = sval;
		    minSize++;
		}
	    }
	}
	d_minMarks.duplicate(buffer.data(), minSize);
    }
    
    return rv;
    
}

//------------------------------------------------------------
//
//.S	QwtScaleDiv::operator==
//	Equality operator
//
//.s	Parameters
//.p	const QwtScaleDiv &s
//
//.s	Return Type
//		int
//
//.s	Return Value
//		true if this instance is equal to s
//.s	Syntax
//.f	int QwtScaleDiv::operator==(const QwtScaleDiv &s)
//
//------------------------------------------------------------
int QwtScaleDiv::operator==(const QwtScaleDiv &s) const
{
    if (d_lBound != s.d_lBound) return 0;
    if (d_hBound != s.d_hBound) return 0;
    if (d_log != s.d_log) return 0;
    if (d_majStep != s.d_majStep) return 0;
    if (d_majMarks != s.d_majMarks) return 0;
    return (d_minMarks == s.d_minMarks);
}

//------------------------------------------------------------
//
//.S	QwtScaleDiv::operator!=
//	Inequality
//
//.s	Parameters
//.p	const QwtScaleDiv &s
//
//.s	Return Type
//		int
//
//.s	Return Value
//		true if this instance is not equal to s
//.s	Syntax
//.f	int QwtScaleDiv::operator!=(const QwtScaleDiv &s)
//
//------------------------------------------------------------
int QwtScaleDiv::operator!=(const QwtScaleDiv &s) const
{
    return (!(*this == s));
}

//------------------------------------------------------------
//
//.S	QwtScaleDiv::reset
//	Detach the shared data and set everything to zero. 
//
//.s	Syntax
//.f	void QwtScaleDiv::reset()
//
//------------------------------------------------------------
void QwtScaleDiv::reset()
{
    // detach arrays
    d_majMarks.duplicate(0,0);
    d_minMarks.duplicate(0,0);

    d_lBound = 0.0;
    d_hBound = 0.0;
    d_majStep = 0.0;
    d_log = false;

}


