/**************************************************************************
 * $Id: GraphPanel.cpp 1.1 Thu, 03 Dec 1998 12:49:42 +0100 samo $
 * $ReleaseVersion: 1.3.1 $
 *
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include "GraphPanel.h"

#include "SamplinInput.h"
#include "qwt_symbol.h"

#include <qbitmap.h>
#include <qdrawutl.h>
#include <qfiledlg.h>
#include <qmessagebox.h>

#include "GraphPanel.moc"

#include <samplin.h>

#define LIST_WIDTH 160
#define HANDLE_WIDTH 16

#define SERIES_BEGIN 5

#define OLD_GRAPH_EXT ".1.SampLin"
#define GRAPH_EXT ".Graph"
#define EXPORT_TXT_EXT ".txt"
#define EXPORT_PIC_EXT ".pic"
#define EXPORT_BMP_EXT ".bmp"


SamplinCurve *clipboard_curve=0;

void PropItem::paint( QPainter *p )
{
   p->drawPixmap( 2, 1, *pm );
   QFontMetrics fm = p->fontMetrics();
   int yPos;                       // vertical text position
   if ( pm->height() < fm.height() )
     yPos = 1+fm.ascent() + fm.leading()/2;
   else
     yPos = 1+pm->height()/2 - fm.height()/2 + fm.ascent();
   p->drawText( pm->width() + 10, yPos, text() );
}

int PropItem::height(const PropBox *lb ) const
{
   return QMAX( pm->height()+2, lb->fontMetrics().lineSpacing() + 1 );
}

int PropItem::width(const PropBox *lb ) const
{
   return pm->width() + lb->fontMetrics().width( text() ) + 6+5+4;
}

uint GraphPropBox::currentKey(void)
{
   uint ret;
   int pos=currentItem();
   if(pos==-1)ret=0;
   else {
      ret=((PropItem*)item(pos))->key();
   }

   return ret;
}

int GraphPropBox::object(int index)
{
   int ret;

   if(index==-1)ret=-1;
   else {
      ret=((PropItem*)item(index))->object();
   }

   return ret;
}

GraphPanel::GraphPanel ( QWidget* parent, const char* name )
:QFrame( parent, name )
{
   spath=NULL;
   grseries=NULL;
   groptions=NULL;
   graxis=NULL;
   
   show_list=FALSE;
   current_object=-1;
   
   setBackgroundColor( QColor("light grey") );   
   setMinimumSize(250,160);
   graph = new SamplinPlot(this,name);
   
//   graph->setPlotColor( QColor("grey") );   
//   graph->setGridPen(QPen(black,1,DotLine));
//   graph->enableGridXMin();

//   graph->setAxisTitle(SamplinPlot::xBottom, "X axis");
//   graph->setAxisTitle(SamplinPlot::yLeft, "Y axis");   
   
   connect(graph, SIGNAL(objClick(const QMouseEvent *,int,uint)),
	   SLOT(graphClick(const QMouseEvent *,int,uint)));
   connect(graph, SIGNAL(objDoubleClick(const QMouseEvent *,int,uint)),
	   SLOT(graphDoubleClick(const QMouseEvent *,int,uint)));   
   
   list = new GraphPropBox(this,"Series");
/*
   QPalette pal;
   QColorGroup ogp;
   pal=list->palette();
   ogp=pal.normal();
   QColorGroup ngp(ogp.foreground(),ogp.background(),ogp.light(),ogp.dark(),ogp.mid(),ogp.text(),QColor("light grey"));
//   pal.setNormal(ngp);
   ogp=pal.disabled();
   QColorGroup dgp(ogp.foreground(),ogp.background(),ogp.light(),ogp.dark(),ogp.mid(),ogp.text(),QColor("light grey"));
//   pal.setDisabled(dgp);
   ogp=pal.active();
   QColorGroup agp(ogp.foreground(),ogp.background(),ogp.light(),ogp.dark(),ogp.mid(),ogp.text(),QColor("light grey"));
//   pal.setActive(agp);   
   list->setPalette( pal );   
 */
   listFill();
   list->hide();
   
   connect( list, SIGNAL(selected(int)), this, SLOT(objectProperties(int)));
   connect(list,SIGNAL(mouseClick(QMouseEvent *, int)),this,SLOT(listPopup(QMouseEvent *, int)));
   connect(list,SIGNAL(highlighted(int)),this,SLOT(listHighlight(int)));

   handlebutton= new QPushButton(this);
   connect( handlebutton, SIGNAL( pressed() ), this, SLOT( handleClick() ));
  
   popexport = new KPopupMenu("Export");
   CHECK_PTR(popexport);
   popexport->insertItem("&Text data",this,SLOT(exportData()));
   popexport->insertItem("&Qt picture",this,SLOT(exportPic()));
   popexport->insertItem("&BMP bitmap",this,SLOT(exportBMP()));

   
   popgraph = new KPopupMenu("Graph");
   CHECK_PTR(popgraph);
   popgraph->insertItem("&Open",this,SLOT(load()));
   popgraph->insertItem("&Save",this,SLOT(save()));
   popgraph->insertItem("Save &as",this,SLOT(saveAs()));
   popgraph->insertItem("&Export",popexport);
   popgraph->insertItem("&Print",this,SLOT(print()));
   popgraph->insertSeparator();
   popgraph->insertItem("Proper&ties",this,SLOT(propGraph()));

   
   popseries = new KPopupMenu("Series");
   CHECK_PTR(popseries);
   popseries->insertItem("&Copy",this,SLOT(editCopy()));
   popseries->insertItem("C&ut",this,SLOT(editCut()));
   popseries->insertItem("&Paste",this,SLOT(editPaste()));
   popseries->insertSeparator();
   popseries->insertItem("Proper&ties",this,SLOT(propSeries()));
   popseries->insertItem("&Rename",this,SLOT(editRename()));
   popseries->insertSeparator();
   popseries->insertItem("&Top",this,SLOT(propTop()));
   popseries->insertItem("&Bottom",this,SLOT(propBottom()));
   
   popxbaxis = new KPopupMenu("Bottom axis");
   CHECK_PTR(popxbaxis);
   popxbaxis->insertItem("Proper&ties",this,SLOT(propXBAxis()));

   popylaxis = new KPopupMenu("Left axis");
   CHECK_PTR(popylaxis);
   popylaxis->insertItem("Proper&ties",this,SLOT(propYLAxis()));
   
   popxtaxis = new KPopupMenu("Top axis");
   CHECK_PTR(popxtaxis);
   popxtaxis->insertItem("Proper&ties",this,SLOT(propXTAxis()));   
   
   popyraxis = new KPopupMenu("Right axis");
   CHECK_PTR(popyraxis);
   popyraxis->insertItem("Proper&ties",this,SLOT(propYRAxis()));      

}


GraphPanel::~GraphPanel()
{
   if(spath!=NULL)delete spath;
}

void GraphPanel::graphClick(const QMouseEvent *m,int object,uint key)
{
   int bt=m->button();
   if(bt==LeftButton && (object==OBJ_PLOT || object==OBJ_SERIES)){
      float x,y;
      x=graph->transform(m->pos().x(),xBottom);
      y=graph->transform(m->pos().y(),yLeft);
      
      emit pointShow(bt,x,y);      
   }
   if(bt==LeftButton || bt==RightButton)
     selectObject(object,key);
   if(bt==RightButton){
//      graph->blinkObject(object,key);
      QPoint pt=graph->mapToGlobal(QPoint(m->pos().x(),m->pos().y()));
      objectPopup(pt,object);
   }
   
}

void GraphPanel::graphDoubleClick(const QMouseEvent *m,int object,uint key)
{
   if(m->button()==LeftButton){
      graph->blinkObject(object,key);
      objectProperties(object);
   }
}

void GraphPanel::propXBAxis(){
   propAxes(xBottom);   
}
void GraphPanel::propYLAxis(){
   propAxes(yLeft);   
}
void GraphPanel::propXTAxis(){
   propAxes(xTop);   
}
void GraphPanel::propYRAxis(){
   propAxes(yRight);   
}

void GraphPanel::propCurrent()
{
   objectProperties(current_object);
}

void GraphPanel::objectProperties(int obj)
{
   switch(obj){
    case -1:
      break;
    case OBJ_GRAPH:
      propGraph();
      break;
    case OBJ_XBOTTOM:
      propAxes(xBottom);
      break;
    case OBJ_YLEFT:
      propAxes(yLeft);
      break;
    case OBJ_XTOP:
      propAxes(xTop);
      break;
    case OBJ_YRIGHT:
      propAxes(yRight);
      break;
    default:
      propSeries();
      break;
   }   
}

void GraphPanel::objectPopup(QPoint &pt,int obj)
{
   SamplinCurve *crv;
   
   switch(obj){
    case OBJ_GRAPH:
    case OBJ_PLOT:
      popgraph->popup(pt);      
      break;
    case OBJ_XBOTTOM:
      popxbaxis->popup(pt);            
      break;
    case OBJ_YLEFT:
      popylaxis->popup(pt);            
      break;
    case OBJ_XTOP:
      popxtaxis->popup(pt);            
      break;	
    case OBJ_YRIGHT:
      popyraxis->popup(pt);            
      break;	
    default:
      if((crv=currentCurve())!=0){
	 popseries->setTitle(QString("Series [")+crv->name()+"]");
	 popseries->popup(pt);            	   
      }
      break;
   }
}

void GraphPanel::listPopup(QMouseEvent *e,int index)
{
   if(e->button()==RightButton){
      QPoint pt=list->mapToGlobal(QPoint(e->x()+8,(int)((index+0.5)*list->itemHeight(index))));
//      debug("index=%i",index);
//      if(index>=5)index=index-5+OBJ_SERIES; // OBJ_PLOT=5
      objectPopup(pt,list->object(index));
   }
}


void GraphPanel::listFill(bool repaint)
{
   int pos;
   KIconLoader iconLoader;
   QString s;
   
   if(list->count()>0 )pos=list->currentItem();
   else pos=0;
   if(pos==-1)pos=0;
   
   list->setAutoUpdate(FALSE);   
   list->clear();
   
   list->insertItem(new PropItem("Graph",iconLoader.loadMiniIcon("samplin_gp-graph.xpm"),0,OBJ_GRAPH));
   list->insertItem(new PropItem("Bottom axis",iconLoader.loadMiniIcon("samplin_gp-xbottom.xpm"),0,OBJ_XBOTTOM));
   list->insertItem(new PropItem("Left axis",iconLoader.loadMiniIcon("samplin_gp-yleft.xpm"),0,OBJ_YLEFT));
   list->insertItem(new PropItem("Top axis",iconLoader.loadMiniIcon("samplin_gp-xtop.xpm"),0,OBJ_XTOP));
   list->insertItem(new PropItem("Right axis",iconLoader.loadMiniIcon("samplin_gp-yright.xpm"),0,OBJ_YRIGHT));

   SamplinCurve *crv = graph->firstCurve();
   while((crv=graph->currentCurve())){

      if(strlen(crv->name())>0)
	s=QString("[")+crv->name()+"]";
      else s="[ ]";

      list->insertItem(new PropItem(s,&(crv->pixmap()),crv->key(),OBJ_SERIES));            
      graph->nextCurve();
   }

//   printf("list count:%i, pos:%i, set:%i\n",list->count(),pos,set);
//   if(pos!=-1 && list->count()>0);//list->setCurrentItem(0);

   if(repaint==TRUE){
      int set=MIN((int)(list->count()-1),pos);
      if(list->count()>0)list->setCurrentItem(set);
      list->setAutoUpdate(TRUE);
      list->repaint();      
   }
}

void GraphPanel::resizeEvent( QResizeEvent *e )
{

   QWidget::resizeEvent(e);
   handlebutton->setGeometry(width()-HANDLE_WIDTH-2,3,HANDLE_WIDTH, height()-6);
   if(show_list)
     list->setGeometry(width()-HANDLE_WIDTH-LIST_WIDTH-2,3,LIST_WIDTH, height()-6);
   graph->setGeometry(3,3,width()-LIST_WIDTH*show_list-HANDLE_WIDTH-12, height()-6);      
}


void GraphPanel::paintEvent(QPaintEvent *e)
{
   static QPainter paint;
   QColorGroup g = QFrame::colorGroup();   
   
   QFrame::paintEvent(e);   
  
   paint.begin(this);
   qDrawShadeRect(&paint,0,0,width()-HANDLE_WIDTH-6-(LIST_WIDTH-1)*show_list,height(),colorGroup(),FALSE,1,1,NULL);
   qDrawShadeRect(&paint,width()-HANDLE_WIDTH-6-(LIST_WIDTH-1)*show_list,0,HANDLE_WIDTH+LIST_WIDTH*show_list+6,height(),colorGroup(),FALSE,1,1,NULL);   
     
   paint.end();
}

void GraphPanel::handleClick()
{
 
   if(show_list ==FALSE )
     {
	show_list = TRUE;
	list->show();
	list->setGeometry(width()-HANDLE_WIDTH-LIST_WIDTH-2,3,LIST_WIDTH, height()-6);
     }
   else {show_list=FALSE; list->hide();}

   graph->setGeometry(3,3,width()-LIST_WIDTH*show_list-HANDLE_WIDTH-12, height()-6);      
//   handlebutton->clearFocus();
   repaint();
}

void GraphPanel::print()
{
   graph->print();
}

void GraphPanel::replot(void)
{
   graph->replot();
   list->repaint(FALSE);
   //   listFill();
}

void GraphPanel::editCut()
{
   int i;
   uint key;
   
   i=list->currentItem();
   if(i>=SERIES_BEGIN && i!=-1){
      list->setAutoUpdate(FALSE);
      key=list->currentKey();      
//      printf("removing item %i key %i\n",i,key);

      if(clipboard_curve!=NULL)delete clipboard_curve;
      clipboard_curve = new SamplinCurve( *(graph->findCurve(key) ) );
      
      graph->removeCurve(key);
      graph->replot();
      listFill();
   }

}

void GraphPanel::editCopy()
{
   int i;
   uint key;
   
   i=list->currentItem();
   if(i>=SERIES_BEGIN && i!=-1){
      key=list->currentKey();
      if(clipboard_curve!=NULL)delete clipboard_curve;
      clipboard_curve = new SamplinCurve( *(graph->findCurve(key) ) );
   }
}

void GraphPanel::editPaste()
{
   SamplinCurve *crv;
   uint key;
   
   if(clipboard_curve!=NULL){
      crv = new SamplinCurve(*clipboard_curve);
      key=graph->insertCurve(crv);
      graph->replot();

      listFill(FALSE);
      list->setCurrentItem(list->count()-1);
      list->setAutoUpdate(TRUE);
      list->repaint();

   }
}

bool GraphPanel::load(const char *fn)
{
   bool ret;

   ret=loadDlg(fn);
   graph->replot();
   listFill();
   return ret;
}

bool GraphPanel::save(const char *fn)
{
    return saveDlg(fn);
}

bool GraphPanel::loadTempl(const char *fn)
{
   bool ret;

   ret=graph->loadTempl(fn);
   graph->replot();
//   listFill();
   return ret;
}

bool GraphPanel::saveTempl(const char *fn)
{
    return graph->saveTempl(fn);
}

bool GraphPanel::export(const char *fn)
{
   QString fname;
   if(fn==0)fname=selectFile("Export graph (*.txt, *.pic, *.bmp)","");
   else fname=fn;
   
   if(fname.isEmpty())
     return 0;

   if(fname.find(EXPORT_PIC_EXT,fname.length()-strlen(EXPORT_PIC_EXT),TRUE)!=-1)
     return graph->exportPic(fname);
   else if(fname.find(EXPORT_BMP_EXT,fname.length()-strlen(EXPORT_BMP_EXT),TRUE)!=-1)   
     return graph->exportBMP(fname);
   else return graph->exportData(fname);
}

bool GraphPanel::exportData(const char *fn)
{
   QString fname;
   
   if(fn==0)fname=selectFile("Export graph data",EXPORT_TXT_EXT);
   else fname=fn;
   
   if(!fname.isEmpty())
     return graph->exportData(fname);
   else return 0;
}

bool GraphPanel::exportPic(const char *fn)
{
   QString fname;
   
   if(fn==0)fname=selectFile("Export graph picture",EXPORT_PIC_EXT);
   else fname=fn;
   
   if(!fname.isEmpty())
     return graph->exportPic(fname);
   else return 0;
}

bool GraphPanel::exportBMP(const char *fn)
{
   QString fname;
   
   if(fn==0)fname=selectFile("Export BMP image",EXPORT_BMP_EXT);
   else fname=fn;
   
   if(!fname.isEmpty())
     return graph->exportBMP(fname);
   else return 0;
}


bool GraphPanel::load()
{
   return load(NULL);
}

bool GraphPanel::saveAs()
{
   return save(NULL);
}
bool GraphPanel::save()
{
   int ret;
   
   if(strlen(graph->fileName())){
      ret=graph->save(NULL);
      if(!ret)
	QMessageBox::warning(this, "Graph window", graph->fileError());     
   }
   else{
      ret=saveDlg(NULL);
   }
   return ret;
}


bool GraphPanel::loadTempl()
{
   return loadTempl(NULL);
}

bool GraphPanel::saveTempl()
{
   return saveTempl(NULL);
}

bool GraphPanel::exportData()
{
   return exportData(NULL);
}

bool GraphPanel::exportPic()
{
   return exportPic(NULL);
}

bool GraphPanel::exportBMP()
{
   return exportBMP(NULL);
}

void GraphPanel::propGraph()
{
   if(groptions==NULL)
     {
	groptions = new GraphOptions(NULL,"Graph properties");
	groptions->setData(graph);
	connect(groptions, SIGNAL(update()), this, SLOT(replot()));
     }
   groptions->show();
   delete groptions;
   groptions=NULL;
//   listFill();   
}

void GraphPanel::propAxes(int axis)
{
   if(graxis==NULL)
     {
	graxis = new Graph_axis(NULL,"Axis properties");
	graxis->setData(graph->axis(axis));
	connect(graxis, SIGNAL(update()), this, SLOT(replot()));
     }
   graxis->show();
   delete graxis;
   graxis=NULL;
}

void GraphPanel::propSeries()
{
   SamplinCurve *crv;

   crv=currentCurve();
   if(grseries==NULL && crv!=NULL)
     {
	grseries = new Graph_series(NULL,"Series options");
	grseries->setData(crv);
	connect(grseries, SIGNAL(update()), this, SLOT(replot()));
	grseries->show();
	delete grseries;
	grseries=NULL;   
//	listFill();
     }
}

void GraphPanel::editRename()
{
   SamplinCurve *crv;
   SamplinInput *inp;
   int i;
   
   i=list->currentItem();

   if(i>=SERIES_BEGIN){
      crv=currentCurve();
      inp = new SamplinInput(this, "input");
      inp->setTitle("Rename series");
      inp->setText(crv->name());
      inp->show();
      if(inp->result()==QDialog::Accepted)
	crv->setName(inp->getText());
      delete inp;
      listFill();
      replot();
   }
}

SamplinCurve *GraphPanel::currentCurve()
{
   SamplinCurve *crv=NULL;
   crv=graph->findCurve(list->currentKey());
   
   return crv;
}

void GraphPanel::propTop()
{
   graph->raiseCurve(list->currentKey());
   graph->replot();
   listFill();
   if(list->currentItem()!=5)list->setCurrentItem(5);
}

void GraphPanel::propBottom()
{
   graph->lowerCurve(list->currentKey());
   graph->replot();
   listFill();
   if(list->currentItem()!=(int)list->count()-1)list->setCurrentItem(list->count()-1);
}

void GraphPanel::listHighlight(int index)
{
   uint key=0;
   
   current_object=list->object(index);
   key=list->currentKey();

   emit objectChange(current_object,key);
}

void GraphPanel::selectObject(int obj, uint key)
{
   int index;
   
   if(obj==OBJ_SERIES){
     if(key!=0)
	index=5+/*obj*/+graph->curveIndex(key);
      else index=OBJ_GRAPH;
   }
   else if(obj==OBJ_PLOT)index=OBJ_GRAPH;
   else index=obj;
   
   if(index>=0 && list->currentItem()!=index)
     list->setCurrentItem(index);
}

bool GraphPanel::loadDlg(const char *n)
{
   bool ret=FALSE;
   QString fn;
   QFileDialog *fdlg;
   QStrList *sl=new QStrList();

   sl->insert(0,"All files (*)");
   sl->insert(0,"Old SampLin formats (*"OLD_GRAPH_EXT")");
   sl->insert(0,"SampLin 1.3 Graph (*"GRAPH_EXT")");

   
   if(n==0){
//      fdlg = new QFileDialog( cnf.root_dir+cnf.data_dir, 0/*"*"+QString(GRAPH_EXT)*/, this, "Open graph",TRUE );
      fdlg = new QFileDialog( this, "Open graph",TRUE );
      fdlg->setCaption("Open graph");
      fdlg->setFilters(*sl);
      fdlg->setDir(cnf.root_dir+cnf.data_dir);

      fdlg->show();
      fn = fdlg->selectedFile();
      if(fdlg->result()!=QDialog::Accepted)fn="";
      delete fdlg;
      //fn = QFileDialog::getOpenFileName( cnf.root_dir+cnf.data_dir, "*"+QString(SAMPLIN_EXT), this, "Open graph",TRUE );
   }
   else
     fn= cnf.root_dir+cnf.data_dir + n;
   
   if ( !fn.isEmpty() ) {
      if(fn.find(GRAPH_EXT,fn.length()-strlen(GRAPH_EXT),TRUE)==-1 &&
	 fn.find(OLD_GRAPH_EXT,fn.length()-strlen(OLD_GRAPH_EXT),TRUE)==-1
	 )fn+=GRAPH_EXT;

	if(isSaved()==FALSE)
	  if( QMessageBox::warning( this, "Graph window",
				     "Session contains unsaved data\n"
				     "Discard old data ?",
				     "&Ok", "&Cancel",
				     0, 1 ) == 1 ) return FALSE;
      
      ret=graph->load(fn);
      if(!ret)
	QMessageBox::warning(this, "Graph window", graph->fileError());
   }

   delete sl;
   return ret;
}

bool GraphPanel::saveDlg(const char *n)
{
   QString fn;
   bool ret=FALSE;
   QFileDialog *fdlg;
   
   if(n==0){
      fdlg = new QFileDialog( cnf.root_dir+cnf.data_dir, "*"+QString(GRAPH_EXT), this, "Save graph",TRUE );
      fdlg ->setCaption("Save graph");
      fdlg ->show();
      fn = fdlg->selectedFile();
      if(fdlg->result()!=QDialog::Accepted)fn="";
      delete fdlg;
//      fn = QFileDialog::getSaveFileName( cnf.root_dir+cnf.data_dir, "*"+QString(SAMPLIN_EXT), this, "Save graph" );
   }
   else
     fn= cnf.root_dir+cnf.data_dir + n;

   if ( !fn.isEmpty() ) {
      if(fn.find(GRAPH_EXT,fn.length()-strlen(GRAPH_EXT),TRUE)==-1)fn+=GRAPH_EXT;

      QFile f(fn);
      if(f.exists()==TRUE)
	switch( QMessageBox::warning( this, "Graph window",
					 "File already exists\n"
					 "Do you want to overwrite old file ?",
					 "&Ok", "&Cancel",
					 0, 1 ) ) {
	 case 0: 
	   ret=graph->save(fn);
	   if(!ret)
	     QMessageBox::warning(this, "Graph window", graph->fileError());
	   break;
	 case 1:
	   break;
	}
      else {
	 ret=graph->save(fn);
	 if(!ret)
	   QMessageBox::warning(this, "Graph window", graph->fileError());
      }
   }
   return ret;
}

QString GraphPanel::selectFile(const char *caption, const char *ext)
{
   QString fn;
   QFileDialog *fdlg;

   fdlg = new QFileDialog( cnf.root_dir+cnf.data_dir, "*"+QString(ext), this, caption,TRUE );
   fdlg ->setCaption(caption);
   fdlg ->show();
   fn = fdlg->selectedFile();
   if(fdlg->result()!=QDialog::Accepted)fn="";
   delete fdlg;

   if ( !fn.isEmpty()) {
      if(!QFile::exists(fn) && fn.find(ext,fn.length()-strlen(ext),TRUE)==-1)fn+=ext;

      if(QFile::exists(fn)==TRUE)
	switch( QMessageBox::warning( this, "Graph window",
					 "File already exists\n"
					 "Do you want to overwrite old file ?",
					 "&Ok", "&Cancel",
					 0, 1 ) ) {
	 case 0: 
	   break;
	 case 1:
	   fn="";
	   break;
	}
   }
   return fn;    
}

