/**************************************************************************
 * $Id: rserial_server.c 1.1 Thu, 03 Dec 1998 12:49:42 +0100 samo $
 * $ReleaseVersion: 1.3.1 $
 *
 * rdevice RPC server
 * 
 * This file is part of SampLin data acquisition software
 * Copyright (C) 1997,98 Samuel Kvasnica
 *
 * SampLin is free software; you can redistribute it and/or modify it
 * under the terms of the version 2 of GNU General Public License as
 * published by the Free Software Foundation.
 *
 * SampLin is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 *
 * You should have received a copy of the GNU General Public License
 * (see the file LICENSE) along with SampLin package; if not, write to the
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 **************************************************************************/

#include "rserial.h"

#include <rpc/pmap_clnt.h>
#include <stdio.h>   /* Standard input/output definitions */
#include <string.h>  /* String function definitions */
#include <unistd.h>  /* UNIX standard function definitions */
#include <fcntl.h>   /* File control definitions */
#include <errno.h>   /* Error number definitions */
#include <termios.h> /* POSIX terminal control definitions */
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/time.h>
#include <netdb.h>
#include <limits.h>

//#define RETRY_READ 30000
#define CONF_FILE "/etc/rdevice.conf"
#define STATUS_FILE "/tmp/rdevice.status"

int debug_flag;

typedef struct {
   int uid;
   int gid;
   char host[16];
   int fd;
   char *file;
   int flags;
} fileitem;

fileitem *openfiles[OPEN_MAX];

void writeitems(void);
void createitem(char *host, int uid, int gid, int fd, int flags, char *file);
void deleteitem(char *host, int uid, int gid, int fd);
int  finditem(char *host, int uid, int gid, int fd);
int  findpitem(char *host, int uid, int gid, int flags, char *file);

void rserialprog_1(struct svc_req *, register SVCXPRT *);


int rdeviceAuth(char *dev, char *host)
{
   char line[1024];
   int ret=0;
   FILE *fp;
   char *ptr,*newptr;
   struct hostent *he;
   char clnt_host[32], found_host[32];
   
   if(debug_flag)printf("Testing permissions for host %s, device %s\n",host, dev);
   
   fp=fopen(CONF_FILE,"rt");
   if(fp==NULL){
      fprintf(stderr,"<Error> Unable to open permission file %s\n",CONF_FILE);
      return(0);
   }

   he=gethostbyname(host);
   if(h_errno!=0){
      fclose(fp);
      return(0);
   }
   
   memcpy(clnt_host,he->h_addr_list[0],he->h_length);

   while(!feof(fp)){
      fgets(line,1024,fp);
      ptr=strchr(line,'\n');
      if(ptr!=NULL)*ptr=0;
      if((ptr=strchr(line,':'))!=NULL){
	 *ptr=0;++ptr;
	 if(strncmp(line,dev,strlen(dev)))continue;

	 do{
	    newptr=strchr(ptr,',');
	    if(newptr!=NULL){*newptr=0;++newptr;}

//	    printf("Got host name: %s\n",ptr);
	    he=gethostbyname(ptr);
	    ptr=newptr;
	    if(h_errno!=0)continue;	 

	    memcpy(found_host,he->h_addr_list[0],he->h_length);
	    if(memcmp(found_host,clnt_host,he->h_length))continue;
	    else {
	       fclose(fp);
	       if(debug_flag)printf("Permissions Ok\n");
	       return(1);	 
	    }
	    
	 }while(ptr!=NULL);
 
      }
   }
   
   fclose(fp);
   return(ret);
}



int rserial_perm(struct svc_req *rqstp, SVCXPRT *transp, serial_request *req)
{
   struct authunix_parms *unix_cred=0;
   
   if(rqstp->rq_proc == RSERIAL_GETHANDLE){
      switch(rqstp->rq_cred.oa_flavor){
       case AUTH_UNIX:
	 unix_cred = (struct authunix_parms *)rqstp->rq_clntcred;
	 if(debug_flag)printf("Remote request from %s (uid=%d, gid=%d)\n",
		 unix_cred->aup_machname,
		 unix_cred->aup_uid,
		 unix_cred->aup_gid
		 );
	 if(!rdeviceAuth(req->device,unix_cred->aup_machname)){
	    fprintf(stderr, "<Error> Host %s is not permitted to request device %s\n",unix_cred->aup_machname, req->device);
	    svcerr_weakauth(transp);
	    return(0);
	 }
	 break;
       default:
	 fprintf(stderr,"<Error> No authentication packets sent, request from %s rejected\n", unix_cred->aup_machname);
	 svcerr_weakauth(transp);
	 return(0);
	 break;
      }
   }
   return(1);
}


void * 
rserial_null_1_svc(void *argp, struct svc_req *rqstp)
{

   static char* result;
                         
   return((void*) &result);
}

int * 
rserial_gethandle_1_svc(serial_request *argp, struct svc_req *rqstp)
{
   static int fd, ret; 
   int item;
   char *dev_path;
   struct authunix_parms *unix_cred;
   
   dev_path=argp->device;
   fd=argp->handle;

   unix_cred = (struct authunix_parms *)rqstp->rq_clntcred;

   
   switch(argp->request){
      
    case REQ_OPEN:
      if(debug_flag)printf("Opening device %s for host %s, flags %i\n",dev_path, argp->client, argp->arg);

      item=findpitem(unix_cred->aup_machname, unix_cred->aup_uid,
		     unix_cred->aup_gid, argp->arg, dev_path);
      if(item==-1)
	fd = open(dev_path, argp->arg );
      else fd=openfiles[item]->fd;
      
      if (fd == -1)
	{                                              /* Could not open the port */
	   fprintf(stderr, "<Error> Unable to open device %s - %s\n",dev_path,
		   strerror(errno));
	}
      else{
	 if(debug_flag)printf("Got handle %i for device %s\n",fd,dev_path);
	 
	 if(item==-1)createitem(unix_cred->aup_machname, unix_cred->aup_uid,
				unix_cred->aup_gid, fd, argp->arg, dev_path);
      }
      
      
      return(&fd);
      break;
      

    case REQ_CLOSE:
      if(debug_flag)printf("Closing device handle %i\n",fd);
      
      ret=close(fd);
      
      if(debug_flag)printf("Device %i closed, result=%i\n",fd, ret);

      if(!ret)deleteitem(unix_cred->aup_machname,unix_cred->aup_uid,
		 unix_cred->aup_gid,fd);
      
      return(&ret);
    
      break;

    default:
      if(debug_flag)printf("<Error> Invalid request.\n");
      return(0);
      break;
   }
}

serial_request * 
rserial_dorequest_1_svc(serial_request *argp, struct svc_req *rqstp)
{
   static serial_request  result;
   char chout, *ptr;
   int i,ret,req_len,ii, RETRY_READ, item;
   static char *buffer=NULL;
   int retry_count;
//   struct termios options;
   int maxfd;
   fd_set readfs;
   struct timeval timeout;
   struct authunix_parms *unix_cred;   
   
   RETRY_READ=argp->arg;
   req_len=argp->count;
   memset(&result,0, sizeof(serial_request));
   result.request=argp->request;
   result.buffer.buffer_val=NULL;
   result.buffer.buffer_len=0;
   result.arg=0;
   result.ret=0;
   result.count=0;
   result.client="";
   result.device="";
     
   unix_cred = (struct authunix_parms *)rqstp->rq_clntcred;
   item=finditem(unix_cred->aup_machname, unix_cred->aup_uid,
		  unix_cred->aup_gid, argp->handle);
   if(item==-1){
      result.ret=-1;
      return(&result);
   }
   
   switch(argp->request){
    case REQ_WRITE:
      if(debug_flag){
	 printf("Handle %i: request WRITE %i bytes: ",argp->handle, argp->buffer.buffer_len);
	 for(i=0;i<req_len;++i)printf("%i,",argp->buffer.buffer_val[i]);
	 printf("\n");
      }
      write(argp->handle,argp->buffer.buffer_val,argp->buffer.buffer_len);
      break;

    case REQ_READ_STR:
      if(buffer)free(buffer);
      buffer=calloc(1024,sizeof(char));
      memset(buffer,0,1024);
      
      if(debug_flag)printf("Handle %i: request READ STRING %i bytes\n", argp->handle, req_len);

      i=0;
      
      if(RETRY_READ==0){
	 ret=read(argp->handle, buffer, req_len);
	 if(ret!=-1)i=ret;

	 *(buffer+i)=0;
	 ptr=strchr(buffer,'\n');
	 if(ptr!=NULL){
	    *ptr=0;
	    i=(int)(ptr-buffer);
	 }
      }
      else{

	 do{
	    retry_count=0;
	    do{
	       ret=read(argp->handle, &chout, sizeof(chout));
	       ++retry_count;
	    }while(ret==-1 && errno==11 && retry_count<RETRY_READ);
	    if(debug_flag)
	      if(retry_count==RETRY_READ)
		printf("<Error> MAX RETRY REACHED, offset %i\n",i);	 
	    
	    if(ret!=-1 && chout!='\n'){
	       *(buffer+i)=chout;
	       ++i;
	    }
	 }while(ret!=0 && ret!=-1 && chout!='\n' && i<req_len);
      }

      *(buffer+i)=0;
      result.buffer.buffer_val=buffer;
      result.buffer.buffer_len=i+1;
      result.count=i;
      if(debug_flag)printf("Handle %i: READ STRING done, %i bytes read: %s\n", argp->handle, i, buffer);
      break;
    case REQ_READ_BIN:
      if(buffer)free(buffer);
      buffer=calloc(req_len+4,sizeof(char));
      memset(buffer,0,req_len+4);
      
      if(debug_flag)printf("Handle %i: request READ BINARY %i bytes\n",argp->handle, req_len);

      i=0;

      if(RETRY_READ==0){
	 ret=read(argp->handle, buffer, req_len);
	 if(ret!=-1)i=ret;
      }
      else{
      
	 do{
	    retry_count=0;
	    do{
	       ret=read(argp->handle, &chout, sizeof(chout));
	       ++retry_count;
	    }while(ret==-1 && errno==11 && retry_count<RETRY_READ);
	    if(debug_flag)
	      if(retry_count==RETRY_READ)
		printf("<Error> MAX RETRY REACHED, offset %i\n",i);
	    
	    if(ret!=-1){
	       *(buffer+i)=chout;
	       ++i;
	    }
	 }while(ret!=0 && ret!=-1 && i<req_len);
      }

      if(debug_flag){
	 printf("Handle %i: READ BINARY done, %i bytes read:\n",argp->handle,i);
	 if(ret<=0)printf("Result=%i, %s\n",ret,strerror(errno));
	 for(ii=0;ii<i;++ii)
	   printf("%i,",*(buffer+ii));	          	
	 printf("\n");
//	 if(ret<=0)printf("Result=%i, %s\n",ret,strerror(errno));
      }

      result.buffer.buffer_val=buffer;
      result.buffer.buffer_len=i;
      result.count=i;

      break;      
    case REQ_TEST:
//      if(debug_flag)printf("Handle %i: request TEST DATA READY\n", argp->handle);
      maxfd=argp->handle+1;
      FD_SET(argp->handle,&readfs);
      timeout.tv_usec=0;
      timeout.tv_sec=0;
      ret=select(maxfd, &readfs, NULL,NULL,&timeout);
      result.ret=ret;
//      if(debug_flag)printf("Handle %i: TEST DATA READY done, result=%i\n", argp->handle,ret);
      break;

    case REQ_SETFL:

      ret=fcntl(argp->handle, F_SETFL, argp->arg);
      if(debug_flag)printf("Handle %i: request SET FLAGS: %i\n",argp->handle, argp->arg);      
      result.ret=ret;
      break;
      
    case REQ_TCSET:
      ret=tcsetattr(argp->handle, argp->arg,(struct termios *) argp->buffer.buffer_val);
      if(debug_flag)printf("Handle %i: request TCSET\n",argp->handle);      
      result.ret=ret;
      break;      
      
    case REQ_IOCTL:
      ret=ioctl(argp->handle, argp->arg,*(long *) argp->buffer.buffer_val);
      if(debug_flag)printf("Handle %i: request IOCTL, service %i\n",argp->handle, argp->arg);      
      result.ret=ret;
      break;      
      
    default:
      if(debug_flag)printf("<Error> Invalid request.\n");
      break;
   
   }
   
   return(&result);
}

int
main(int argc, char **argv)
{
   int i;
   register SVCXPRT *transp;
   
   for(i=0;i<OPEN_MAX;++i)
     openfiles[i]=NULL;
   writeitems();
   
   if(argc==2 && !strcmp(argv[1],"--debug")){
      debug_flag=TRUE;
      printf("Starting in debug mode\n");
   }
   else debug_flag=FALSE;
   
	(void) pmap_unset(RSERIALPROG, RSERIALVERS);

	transp = svcudp_create(RPC_ANYSOCK);
	if (transp == NULL) {
		fprintf(stderr, "cannot create udp service.");
		exit(1);
	}
	if (!svc_register(transp, RSERIALPROG, RSERIALVERS, rserialprog_1, IPPROTO_UDP)) {
		fprintf(stderr, "unable to register (RSERIALPROG, RSERIALVERS, udp).");
		exit(1);
	}

	transp = svctcp_create(RPC_ANYSOCK, 0, 0);
	if (transp == NULL) {
		fprintf(stderr, "cannot create tcp service.");
		exit(1);
	}
	if (!svc_register(transp, RSERIALPROG, RSERIALVERS, rserialprog_1, IPPROTO_TCP)) {
		fprintf(stderr, "unable to register (RSERIALPROG, RSERIALVERS, tcp).");
		exit(1);
	}

	svc_run();
	fprintf(stderr, "svc_run returned");
	exit(1);
}

void createitem(char *host, int uid, int gid, int fd, int flags, char *file)
{
   fileitem *item;
   struct hostent *he;
   int i;
   
   i=0;
   while( (i<OPEN_MAX) && (openfiles[i]!=NULL) )++i;
   if(i!=OPEN_MAX){
      item = malloc(sizeof(fileitem));
      item->uid=uid;
      item->gid=gid;
      item->fd=fd;
      item->flags=flags;
      he=gethostbyname(host);
      memcpy(item->host,he->h_addr_list[0],he->h_length);
      item->file=strdup(file);
      openfiles[i]=item;
      writeitems();
   }
   else printf("<Error> Max open files reached.\n");

}

void deleteitem(char *host, int uid, int gid, int fd)
{
   int index;
   fileitem *ptr;
   
   index=finditem(host,uid,gid,fd);
   if(index!=-1){
      ptr=openfiles[index];
      free(ptr->file);
      free(ptr);
      openfiles[index]=NULL;
      writeitems();  
   }
}

int finditem(char *host, int uid, int gid, int fd)
{
   int i;
   struct hostent *he;   
   
   he=gethostbyname(host);
     
   i=0;
   while(i<OPEN_MAX){   
      if(openfiles[i]!=NULL)
	if( (uid==openfiles[i]->uid) && (gid==openfiles[i]->gid) &&
	   (fd==openfiles[i]->fd) && 
	   !memcmp(openfiles[i]->host,he->h_addr_list[0],4))
	  return(i);
      
      ++i;
   }   

   return(-1);
}

int findpitem(char *host, int uid, int gid, int flags, char *file)
{
   int i;
   struct hostent *he;   
   
   he=gethostbyname(host);
     
   i=0;
   while(i<OPEN_MAX){   
      if(openfiles[i]!=NULL)
	if( (uid==openfiles[i]->uid) && (gid==openfiles[i]->gid) &&
	   (strlen(file)==strlen(openfiles[i]->file)) && 
	   !strcmp(file, openfiles[i]->file) &&
	   (flags==openfiles[i]->flags) &&
	   !memcmp(openfiles[i]->host,he->h_addr_list[0],4))
	  return(i);
      
      ++i;
   }   

   return(-1);
}

void writeitems(void)
{
   int i;
   FILE *fp;
   char hst[32];
   unsigned char *ptr;
   
   fp=fopen(STATUS_FILE,"wt");
   if(fp!=NULL){
      fprintf(fp,"%-17s %-6s %-6s %-5s %-6s %s\n","Host","uid","gid","fd","flags","Device");
      for(i=0;i<OPEN_MAX;++i)
	 if(openfiles[i]!=NULL){
	    ptr=openfiles[i]->host;
	    sprintf(hst,"%d.%d.%d.%d",ptr[0],ptr[1],ptr[2],ptr[3]);
	    fprintf(fp,"%-17s %-6i %-6i %-5i %-6i %s\n",hst,openfiles[i]->uid,
		    openfiles[i]->gid,openfiles[i]->fd,openfiles[i]->flags,
		    openfiles[i]->file);
	 }
      fclose(fp);
   }
   
}
