#include "kpuissance4.h"
#include <kapp.h>
#include <kstdaccel.h>
#include <kiconloader.h>
#include <kfiledialog.h>
#include <iostream.h>
#include "resource.h"
#include "options.h"

KPuissance4::KPuissance4(QWidget *parent, const char *name)
	: KTMainWindow(name)
{
	readConfig(kapp->getConfig());
	initMenuBar();
	initToolBar();
	initStatusBar();
	initBoard();
	resize(Board->width(),Board->height());
}

KPuissance4::~KPuissance4()
{
}

void KPuissance4::readConfig(KConfig *conf)
{
	conf->setGroup("Configuration");
	BoardOptions.NumCols = conf->readNumEntry("NumCols",7);
	BoardOptions.NumRows = conf->readNumEntry("NumRows",6);
	BoardOptions.NumPuis = conf->readNumEntry("NumPuis",4);
	BoardOptions.Player1 = conf->readNumEntry("Player1",Human);
	BoardOptions.Player2 = conf->readNumEntry("Player2",Computer);
	BoardOptions.Level = conf->readNumEntry("Level",Medium);
	BoardOptions.UseBackgroundPixmap = conf->readBoolEntry("UseBackgroundPixmap",false);
	QString		defaultPixmap = (KApplication::kde_wallpaperdir() + "/circuit.jpg");
	BoardOptions.BackgroundPixmap = conf->readEntry("BackgroundPixmap",defaultPixmap.data());
	BoardOptions.BackgroundColor = conf->readColorEntry("BackgroundColor",&(colorGroup().base()));
}

void KPuissance4::saveConfig(KConfig *conf)
{
	conf->setGroup("Configuration");
	conf->writeEntry("NumCols",BoardOptions.NumCols);
	conf->writeEntry("NumRows",BoardOptions.NumRows);
	conf->writeEntry("NumPuis",BoardOptions.NumPuis);
	conf->writeEntry("Player1",BoardOptions.Player1);
	conf->writeEntry("Player2",BoardOptions.Player2);
	conf->writeEntry("Level",BoardOptions.Level);
	conf->writeEntry("UseBackgroundPixmap",BoardOptions.UseBackgroundPixmap);
	conf->writeEntry("BackgroundPixmap",BoardOptions.BackgroundPixmap);
	conf->writeEntry("BackgroundColor",BoardOptions.BackgroundColor);
}

void KPuissance4::initMenuBar()
{
	KStdAccel	keys;

	QPopupMenu	*filePop = new QPopupMenu;
	filePop->insertItem(i18n("New game"),this,SLOT(newGame()),Key_N+CTRL,FILE_NEW_GAME_ID);
	filePop->insertItem(i18n("Open game"),this,SLOT(openGame()),keys.open(),FILE_OPEN_GAME_ID);
	filePop->insertItem(i18n("Save game"),this,SLOT(saveGame()),keys.save(),FILE_SAVE_GAME_ID);
	filePop->insertSeparator();
	filePop->insertItem(i18n("Quit"),this,SLOT(quitGame()),keys.quit(),FILE_EXIT_ID);
	menuBar()->insertItem(i18n("&File"),filePop);

	QPopupMenu	*optPop = new QPopupMenu;
	optPop->insertItem(i18n("Configure"),this,SLOT(configure()),Key_T+CTRL,CONFIGURE_ID);
	menuBar()->insertItem(i18n("&Options"),optPop);

	QPopupMenu	*helpPop = kapp->getHelpMenu(true,"KPuissance4 v0.1\n\n"
							  "(C) 1999\n\n"
							  "Authors :\n"
							  "  AI  : Jean Bort (jbort@ufr.lirmm.fr)\n"
							  "  GUI : Michael Goffioul (goffioul@emic.ucl.ac.be)\n\n"
							  "Maintainer :\n"
							  "  Michael Goffioul");
	menuBar()->insertItem(i18n("&Help"),helpPop);
}

void KPuissance4::initToolBar()
{
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("fileopen.xpm"),FILE_OPEN_GAME_ID,SIGNAL(clicked()),this,SLOT(openGame()),true,i18n("Open game"));
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("filefloppy.xpm"),FILE_SAVE_GAME_ID,SIGNAL(clicked()),this,SLOT(saveGame()),true,i18n("Save game"));
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("reload.xpm"),FILE_NEW_GAME_ID,SIGNAL(clicked()),this,SLOT(newGame()),true,i18n("New game"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("undo.xpm"),ACTION_UNDO_ID,SIGNAL(clicked()),this,SLOT(undo()),true,i18n("Undo"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("idea.xpm"),ACTION_HINT_ID,SIGNAL(clicked()),this,SLOT(hint()),true,i18n("Hint"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("exit.xpm"),FILE_EXIT_ID,SIGNAL(clicked()),this,SLOT(quitGame()),true,i18n("Quit"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(kapp->getIconLoader()->loadIcon("help.xpm"),HELP_ID,SIGNAL(clicked()),this,SLOT(showHelp()),true,i18n("Help"));
}

void KPuissance4::initStatusBar()
{
	statusBar()->insertItem("",STATUSMSG_ID);
}

void KPuissance4::newGame()
{
	Board->restart();
}

void KPuissance4::saveGame()
{
	QString		filename = KFileDialog::getSaveFileName(0,"*.kp4",this);
	if (!filename.isNull()) Board->save(filename.data());
}

void KPuissance4::openGame()
{
	QString		filename = KFileDialog::getOpenFileName(0,"*.kp4",this);
	if (!filename.isNull()) Board->load(filename.data());
}

void KPuissance4::configure()
{
	OptionsDialog	*dlg = new OptionsDialog(&BoardOptions,this);
	if (dlg->exec()) {
		dlg->readOptions(&BoardOptions);
		delete Board;
		initBoard();
		resizeEvent(0);
	}
	delete dlg;
}

void KPuissance4::undo()
{
	Board->undo();
}

void KPuissance4::hint()
{
	Board->hint();
}

void KPuissance4::quitGame()
{ close();}

void KPuissance4::showHelp()
{ kapp->invokeHTMLHelp("kpuissance4/index.html","");}

bool KPuissance4::queryClose()
{
	saveConfig(kapp->getConfig());
	return true;
}

void KPuissance4::slotMessage(const char *text)
{
	statusBar()->changeItem(text,STATUSMSG_ID);
	QApplication::flushX();
}

void KPuissance4::initBoard()
{
	Board = new BoardWidget(&BoardOptions,this);
	connect(Board,SIGNAL(message(const char*)),SLOT(slotMessage(const char*)));
	setView(Board);
	slotMessage(i18n("Choose \"New game\" to start."));
	if (isVisible()) Board->show();
}