#include <qrect.h>

#include "board.h"
#include "bitmaps.h"

Board::Board(int size) : QArray<int> (size)
{
    sz = size;				// set size of board
    init(Intro);			// initialize varibales
}

void Board::init(Image image)
{
    prisonEntry = OUT;
    prisonExit = OUT;
    fruitHome = OUT;
    fruitPosition = OUT;
    pacmanHome = OUT;
    pacmanPosition = OUT;
    for (int m = 0; m < 8; m++) {
	monsterHome[m] = OUT;
	monsterPosition[m] = OUT;
    }
    fill(0);
    numPoints = 0;
    numSwitches = 0;
    numMonsters = 0;
    switch (image) {
	case Intro : 
// setup(demo_bits);
		     break;
	case Demo  : setup(demo_bits);
		     break;
	case Level : setup(demo_bits);
		     break;
    }
}

void Board::setup(const uchar *buf)
{
    for ( int index = 0; buf[index] != 0 && index < BoardWidth*BoardHeight; index++ ) {
	switch (buf[index]) {
	    case '*' : set(index, brick); break;
	    case '+' : set(index, out); break;
	    case '#' : set(index, prison); break;
	    case '-' : set(index, gate); break;
	    case '.' : set(index, Point); break;
	    case 'o' : set(index, Switch); break;
	    case 'I' : set(index, prisonentry); break;
	    case 'O' : set(index, prisonexit); break;
	    case 'F' : set(index, fruithome); break;
      	    case 'P' : set(index, pacmanhome);  break;
	    default  : if (buf[index] >= '0' && buf[index] <= '7') {
		           set(index, monsterhome, buf[index]-(uchar)'0');
		       }
	}
    }
}

bool Board::inBounds(int pos)
{
  return ( pos < 0 || pos > sz-1 ? FALSE : TRUE);
}

void Board::set(int pos, Square sq, int m)
{
    if (inBounds(pos))
    	switch (sq) {
	    case out         : at(pos) = OUT; break;
    	    case Point       : at(pos) |= pointBit; numPoints++; break;
    	    case Switch      : at(pos) |= switchBit; numSwitches++; break;
    	    case fruit	     : at(pos) |= fruitBit; fruitPosition = pos; break;
    	    case pacman	     : at(pos) |= pacmanBit; pacmanPosition = pos; break;
	    case monster     : at(pos) |= (monsterBit << m); 
	   	               monsterPosition[m] = pos; break;
            case prisonentry : prisonEntry = pos; at(pos) = empty; break;
            case prisonexit  : prisonExit = pos; at(pos) = empty; break;
            case fruithome   : fruitHome = pos; at(pos) = empty; break;
            case pacmanhome  : pacmanHome = pos; at(pos) = empty; break;
            case monsterhome : monsterHome[m] = pos; at(pos) = empty; 
                               if (m == 0 && prisonExit == OUT)
			           prisonExit = pos;
			       if (m == 1 && prisonEntry == OUT)
			           prisonEntry = pos;
			       numMonsters++;
		               break;
	    default          : at(pos) = sq;
        }
}

void Board::reset(int pos, Square sq, int m)
{
    if (inBounds(pos))
    	switch (sq) {
            case out     : at(pos) = empty; break;
    	    case Point   : at(pos) = at(pos) ^ pointBit; numPoints--; break;
    	    case Switch  : at(pos) = at(pos) ^ switchBit; numSwitches--; break;
    	    case fruit   : at(pos) = at(pos) ^ fruitBit; fruitPosition = OUT; break;
    	    case pacman  : at(pos) = at(pos) ^ pacmanBit; pacmanPosition = OUT; break;
	    case monster : at(pos) = at(pos) ^ (monsterBit << m); 
		           monsterPosition[m] = OUT; break;
            default      : at(pos) = at(pos) & varBits;
	}
}

int Board::position(Square sq, int m)
{
    switch(sq) {
	case prisonentry  : return prisonEntry;
	case prisonexit	  : return prisonExit;
	case fruit	  : return fruitPosition;
	case fruithome	  : return fruitHome;
	case pacman	  : return pacmanPosition;
	case pacmanhome	  : return pacmanHome;
	case monster	  : return monsterPosition[m];
	case monsterhome  : return monsterHome[m];
	default		  : return OUT;
    }
}

bool Board::isOut(int pos)
{
    if (inBounds(pos))
	return (at(pos) == OUT ? TRUE : FALSE);
    return TRUE;
}

bool Board::isEmpty(int pos)
{
    if (inBounds(pos))
	return ((at(pos) & fixBits) == empty ? TRUE : FALSE);
    return TRUE;
}

bool Board::isBrick(int pos)
{
    if (inBounds(pos))
	return ((at(pos) & fixBits) == brick ? TRUE : FALSE);
    return FALSE;
}

bool Board::isPrison(int pos)
{
    if (inBounds(pos))
	return ((at(pos) & fixBits) == prison ? TRUE : FALSE);
    return FALSE;
}

bool Board::isGate(int pos)
{
    if (inBounds(pos))
    	return ((at(pos) & fixBits) == gate ? TRUE : FALSE);
    return FALSE;
}

bool Board::isPoint(int pos)
{
    if (inBounds(pos) && at(pos) != OUT)
     	return ((at(pos) & pointBit) != 0 ? TRUE : FALSE);
    return FALSE;
}

bool Board::isSwitch(int pos)
{
    if (inBounds(pos) && at(pos) != OUT)
    	return ((at(pos) & switchBit) != 0 ? TRUE : FALSE);
    return FALSE;
}

bool Board::isFruit(int pos)
{
    if (inBounds(pos) && at(pos) != OUT)
    	return ((at(pos) & fruitBit) != 0 ? TRUE : FALSE);
    return FALSE;
}

bool Board::isPacman(int pos)
{
    if (inBounds(pos) && at(pos) != OUT)
    	return ((at(pos) & pacmanBit) != 0 ? TRUE : FALSE);
    return FALSE;
}

bool Board::isMonster(int pos)
{
    if (inBounds(pos) && at(pos) != OUT)
    	return ((at(pos) & monsterBits) != 0 ? TRUE : FALSE);
    return FALSE;
}

bool Board::isWay(int pos, int dir, Square sq) {
    int p1 = move(pos, dir, 2);
    if (p1 == OUT)
	return (sq == out ? TRUE : FALSE);
    int p2, p3;
    if (dir == N || dir == S) {
	p2 = move(p1, E);
	p3 = move(p1, W);
    } else {
	p2 = move(p1, N);
	p3 = move(p1, S);
    }
    switch (sq) {
    	case out    : return isOut(p1) | isOut(p2) | isOut(p3); 
    	case empty  : return isEmpty(p1) & isEmpty(p2) & isEmpty(p3);
    	case brick  : return isBrick(p1) | isBrick(p2) | isBrick(p3);
    	case prison : return isPrison(p1) | isPrison(p2) | isPrison(p3);
    	case gate   : return isGate(p1) & isGate(p2) & isGate(p3);
	default     : return FALSE;
    }
}

int Board::move(int pos, int dir, int steps)
{
    if (steps < 0) {			// move backwards
	dir = turn(dir);		// turn around and do your steps
	steps *= -1;
    }

    while (steps-- != 0) {		// until all steps are gone
	switch (dir) {
	    case NW: pos = pos >= BoardWidth && x(pos) > 0 ? (pos-BoardWidth)-1 : sz-1;
		     break;
	    case N:  pos = pos >= BoardWidth ? pos-BoardWidth : (sz-BoardWidth)+x(pos);
		     break;
	    case NE: pos = pos >= BoardWidth && x(pos) < BoardWidth-1 ?
                        (pos-BoardWidth)+1 : sz-BoardWidth;
                     break;
	    case W:  pos = x(pos) > 0 ? pos-1 : pos+(BoardWidth-1);
		     break;
	    case E:  pos = x(pos) < BoardWidth-1 ? pos+1 : pos-(BoardWidth-1);
		     break;
	    case SW: pos = pos < sz-BoardWidth && x(pos) > 0 ? (pos+BoardWidth)-1 : BoardWidth-1;
	 	     break;
	    case S:  pos = pos < sz-BoardWidth ? pos+BoardWidth : x(pos);
		     break;
	    case SE: pos = pos < sz-BoardWidth && x(pos) < BoardWidth-1 ? (pos+BoardWidth)+1 : 0;
		     break;
	}
    }
    return pos;				// here we are
}

int Board::closeup(int pos, int dir, int target)
{
    if (dir == N || dir == S) {
	if (x(target) < x(pos))
	    return W;
        if (x(target) > x(pos))
            return E;
    } else {
	if (y(target) < y(pos))
	    return N;
	if (y(target) > y(pos))
	    return S;
    }
    return dir;
}

int Board::x(int pos)
{
    return pos % BoardWidth;
}

int Board::y(int pos)
{
    return pos/BoardWidth;
}

int Board::turn(int dir)
{
    switch (dir) {
    	case N  : return S;
    	case NE : return SW;
    	case E  : return W;
    	case SE : return NW;
    	case S  : return N;
    	case SW : return NE;
    	case W  : return E;
    	case NW : return SE;
    	default : return dir;
    }
}

int Board::points()
{
    return numPoints;
}

int Board::switches()
{
    return numSwitches;
}

int Board::monsters()
{
    return numMonsters;
}

