/* -*- c++ -*-
 *
 * donkeysocket.cpp
 *
 * Copyright (C) 2003 Petter Stokke <ummo@hellokitty.com>
 * Copyright (C) 2009 Gioacchino Mazzurco <gmazzurco89@gmail.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <kdebug.h>
#include <stdlib.h>

#include "donkeysocket.h"
#include "donkeysocket.moc"

DonkeySocket::DonkeySocket( QObject *parent, const char *name )
    : Q3Socket( parent, name)
{
    fifo.setAutoDelete(true);
    mlHost = QString::null;
    mlPort = 4001;
    connect(this, SIGNAL(readyRead()), SLOT(readMessage()));
}

DonkeySocket::~DonkeySocket()
{
}

void DonkeySocket::connectDonkey()
{
    rstate = 0; ct = 0; sz = 0;
    kDebug() << "Connecting to " << mlHost << ":" << mlPort;
    connectToHost(mlHost, mlPort);
}

void DonkeySocket::connectDonkey(const QString& host, quint16 port)
{
    mlHost = host;
    mlPort = port;
    connectDonkey();
}

int DonkeySocket::sendMessage(const DonkeyMessage& msg)
{
    unsigned char op[4];
    int sz = msg.size() + 2;
    op[0] = (unsigned char)(sz & 0xff);
    op[1] = (unsigned char)((sz >> 8) & 0xff);
    op[2] = (unsigned char)((sz >> 16) & 0xff);
    op[3] = (unsigned char)((sz >> 24) & 0xff);
    Q_LONG r = write((const char *)op, 4);
    if (r != 4) {
        kDebug()<<"DonkeySocket::sendMessage (r!=4) opcode="<<msg.opcode()<<"r="<<r;
        return 0;
    }
    op[0] = (unsigned char)(msg.opcode() & 0xff);
    op[1] = (unsigned char)((msg.opcode() >> 8) & 0xff);
    r = write((const char *)op, 2);
    if (r != 2) {
        kDebug()<<"DonkeySocket::sendMessage (r!=2) opcode="<<msg.opcode()<<"r="<<r;
        return 0;
    }
    r = write((const char*)msg.data(), msg.size());
    int result = r == (Q_LONG)msg.size();
    //kDebug()<<"DonkeySocket::sendMessage opcode="<<msg.opcode()<<"r="<<r<<"msg.size="<<msg.size()<<"result="<<result;
    return result;
}

void DonkeySocket::readMessage()
{
    int done = 1;
    while (bytesAvailable() > 0 && done) {
        done = 0;
        if (rstate == 0 && (int)bytesAvailable() >= 4) {
            //const int bytesavailable = (int)bytesAvailable();
            ct = 0;
            rstate = 1;

            char c;
            bool rd = getChar(&c);
            Q_ASSERT( rd );
            sz = (unsigned char) c;
            rd = getChar(&c);
            Q_ASSERT( rd );
            sz |= (unsigned char)c << 8;
            rd = getChar(&c);
            Q_ASSERT( rd );
            sz |= (unsigned char)c << 16;
            rd = getChar(&c);
            Q_ASSERT( rd );
            sz |= (unsigned char)c << 24;

            sz -= 2;
            //kDebug()<<"DonkeySocket::readMessage 1===> bytesAvailable="<<bytesavailable<<"sz="<<sz;
            Q_ASSERT(sz >= -1);
            Q_ASSERT(sz >= 0);
            done = 1;
        }
        if (rstate == 1 && (int)bytesAvailable() >= 2) {
            //const int bytesavailable = (int)bytesAvailable();
            char c;
            bool rd = getChar(&c);
            Q_ASSERT( rd );
            int opcode = (unsigned char)c;
            rd = getChar(&c);
            Q_ASSERT( rd );
            opcode |= (unsigned char)c << 8;
            //kDebug()<<"DonkeySocket::readMessage 2===> opcode="<<opcode<<"sz="<<sz<<"bytesavailable="<<bytesavailable<<"size="<<size();
            cur = new DonkeyMessage(opcode, sz);
            rstate = 2;
            done = 1;
        }
        if (rstate == 2) {
            const int bytesavailable = (int)bytesAvailable();
            Q_ASSERT( bytesavailable >= 0 );
            int a = ( bytesavailable > (sz-ct) ) ? sz-ct : bytesavailable;
            char* data = (char*)malloc(bytesavailable);
            if (!data) exit(1);

            //kDebug()<<"DonkeySocket::readMessage 3-0 ===> opcode="<<cur->opcode()<<"rstate="<<rstate<<"ct="<<ct<<"sz="<<sz<<"position="<<cur->position()<<"bytesavailable-before="<<bytesavailable<<"bytesavailable-now="<<bytesAvailable();
            int r = (int)read(data, a);
            if (r < 0) {
                rstate = 0;
                free(data);
                delete cur;
                kWarning() << "Read error at DonkeySocket::readMessage!"
                           << "opcode=" << cur->opcode() << "bytesavailable=" << bytesavailable << "position=" << cur->position()
                           << "a=" << a << "ct=" << ct << "sz=" << sz;
                return;
            }

            cur->feedBuffer(data, r);
            free(data);
            ct += r;
            if (ct >= sz) {
                cur->resetPosition();
                fifo.enqueue(cur);
                rstate = sz = ct = 0;
                //kDebug()<<"DonkeySocket::readMessage 3-1 ===> opcode="<<cur->opcode()<<"rstate="<<rstate<<"ct="<<ct<<"sz="<<sz<<"position="<<cur->position()<<"bytesavailable-before="<<bytesavailable<<"bytesavailable-now="<<bytesAvailable();
                cur = NULL;
            }
            else {
                //kDebug()<<"DonkeySocket::readMessage 3-2===> opcode="<<cur->opcode()<<"rstate="<<"ct="<<ct<<"sz="<<sz<<"position="<<cur->position()<<"bytesavailable-before="<<bytesavailable<<"bytesavailable-now="<<bytesAvailable();
            }

            done = 1;
        }
    }

    if (!fifo.isEmpty())
        emit readyMessage();
}

uint DonkeySocket::queueSize()
{
    return fifo.count();
}

DonkeyMessage* DonkeySocket::popMessage()
{
    return fifo.dequeue();
}
